//-----------------------------------------------------------------------------
// SokoWindow-windows.m
//
//	Microsoft Windows-specific window-related functionality.
//
// Copyright (c), 1998,2001, Eric Sunshine <sunshine@sunshineco.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoWindow-windows.m,v 1.1 2001/12/23 16:05:23 sunshine Exp $
// $Log: SokoWindow-windows.m,v $
// Revision 1.1  2001/12/23 16:05:23  sunshine
// v15
// -*- Added SokoWindow and SokoPanel, which are subclasses of NSWindow and
//     NSPanel, respectively.  All windows and panels throughout the
//     application are now instances of these classes.  This allows bugs in
//     the AppKit window and panel classes to be overcome more easily.  For
//     instance, the Windows port works around several Windows-specific
//     AppKit bugs by overriding certain methods in these classes.
//
// -*- On Windows, now ensures that the puzzle view does not get occluded by
//     the menu if the user changes display preferences via the Control
//     Panel.  For instance, the user might assign a larger font to the menu,
//     or the menu might wrap, thus occluding the puzzle view.  This is
//     accomplished by catching the WM_SETTINGCHANGED message from Windows
//     and adjusting the window frame as needed.
//
// -*- Worked around an AppKit bug on Windows which appeared in this release
//     when small versions of the icons were incorporated into the ICO files.
//     The AppKit failed to use these small icons in the window title bars.
//     Instead, it used a scrunched version of the large icon, which was
//     quite ugly.
//
// -*- Worked around an AppKit bug on Windows in which the Preferences panel
//     would suddenly appear on the "Window" menu when -setDocumentEdited:
//     was invoked even though panels are not supposed to appear on this
//     menu.  This problem was further compounded by the fact that the menu
//     item would remain dangling on the menu even after the Preferences
//     panel was closed.
//
// -*- Worked around an AppKit bug on Windows 9x which caused the window size
//     of SokoBoard windows to become completely corrupt when miniaturized.
//     This bug only seems to affect non-resizeable windows (such as
//     SokoBoard) and only occurs on Windows 9x.  (Windows NT and 2000 are
//     not afflicted.)
//
//-----------------------------------------------------------------------------
#import "SokoWindow.h"
#import "SokoAssert.h"
#import <AppKit/NSApplication.h>
#import <Foundation/NSNotification.h>
#import <wtypes.h>
#import <winuser.h>

//-----------------------------------------------------------------------------
// Custom Window Procedure
//	On Windows, although changes to system colors results in a notification
//	being sent so that the application can redraw itself or otherwise
//	refresh its display, changes to window metrics do not generate a
//	notification nor does the AppKit properly resize windows to maintain
//	the content view size.  Consequently, the onus of maintaining the
//	proper content-view size is placed upon the application programmer.
//	This custom window procedure intercepts the WM_SETTINGCHANGE message
//	when window metrics change and sends a notification message.  Modules
//	wishing to manually maintain their content size (by resizing the window
//	frame) should listen for this notification.
//-----------------------------------------------------------------------------
static FARPROC APPKIT_WINDOW_PROC = 0;

LRESULT CALLBACK SokoWindowProc( HWND h, UINT m, WPARAM w, LPARAM l )
    {
    LRESULT rc;
    SOKO_ASSERT( APPKIT_WINDOW_PROC != 0 );
    rc = CallWindowProc( APPKIT_WINDOW_PROC, h, m, w, l );
    if (m == WM_SETTINGCHANGE &&
	(w == SPI_SETNONCLIENTMETRICS || w == SPI_SETBORDER))
	{
	NSWindow* window = [NSApp windowWithWindowHandle:h];
	if (window != 0)
	    [[NSNotificationCenter defaultCenter] postNotificationName:
		SokoWindowMetricsChangedNotification object:window];
	}
    return rc;
    }

static void install_window_proc( HWND h )
    {
    APPKIT_WINDOW_PROC =
	(FARPROC)SetWindowLong( h, GWL_WNDPROC, (LONG)SokoWindowProc );
    }


//-----------------------------------------------------------------------------
// finalize_icon
//	On Microsoft Windows, the AppKit only sets the window's "big" icon even
//	though the corresponding ICO file might also contain a "small" icon.
//	Consequently, the small icon is ignored and the large icon gets scaled
//	down, which typically looks ugly.  To work around this problem, we must
//	also explicitly set the small icon.  Unfortunately, this is only
//	possible if NSWindow object already has an associated Microsoft window
//	handle.  The window handle is created immediately for non-deferred
//	windows, but only when the window is first ordered on-screen for
//	deferred windows.  Therefore, calling code must ensure that this method
//	is called only after the window handle has been created.
//-----------------------------------------------------------------------------
static void finalize_icon( HWND handle )
    {
    LRESULT icon = SendMessage( handle, WM_GETICON, ICON_BIG, 0 );
    if (icon != 0)
	SendMessage( handle, WM_SETICON, ICON_SMALL, (LPARAM)icon );
    }


//=============================================================================
// SokoWindow
//=============================================================================
@implementation SokoWindow (Microsoft)

//-----------------------------------------------------------------------------
// finalize
//	The NSWindow class suffers from some shortcomings on Microsoft Windows.
//	The -finalize method attempts to address these shortcomings, but it can
//	only do so once a valid window handle exists.  The window handle is
//	created immediately for non-deferred windows, but only when the window
//	is first ordered on-screen for deferred windows.  Therefore, calling
//	code must ensure that this method is called only after the window
//	handle has been created.
//-----------------------------------------------------------------------------
- (void)finalize
    {
    if (info == 0)
	{
	HWND h = (HWND)[self windowHandle];
	SOKO_ASSERT( h != 0 );
	finalize_icon(h);
	install_window_proc(h);
	info = (void*)~0;
	}
    }


//-----------------------------------------------------------------------------
// initWithContentRect:styleMask:backing:defer:
//	There is a bug in the AppKit on Windows 9x where the window size of
//	non-resizeable windows becomes corrupted when miniaturized.  Rather
//	than deminiaturizing to normal size, the window deminiaturizes only to
//	the height of its title bar, and to a width just large enough to
//	contain the title bar buttons and a small section of the title.
//	Sizeable windows do not suffer from this problem.  Therefore, to work
//	around this problem, on Windows 9x, SokoWindow automatically overrides
//	the non-resizeable flag for such windows and makes them sizeable.  To
//	compensate for the fact that windows which should not have been
//	sizeable are now sizeable, the window's owner must explicitly prevent
//	the window from resizing, either by setting its minimum and maximum
//	sizes or by overriding -windowWillResize:toSize:.  This problem does
//	not affect Windows NT/2000 or Mach.
//-----------------------------------------------------------------------------
- (id)initWithContentRect:(NSRect)r
    styleMask:(unsigned int)style
    backing:(NSBackingStoreType)backing
    defer:(BOOL)defer
    {
    if ((style & NSMiniaturizableWindowMask) != 0 &&
	(style & NSResizableWindowMask) == 0)
	{
	BOOL buggy = YES; // Assume worst, if GetVersionEx() fails.
	OSVERSIONINFO v;
	v.dwOSVersionInfoSize = sizeof(v);
	if (GetVersionEx(&v) != 0)
	    buggy = (v.dwPlatformId == VER_PLATFORM_WIN32_WINDOWS); // Win9x?
	if (buggy)
	    style |= NSResizableWindowMask;
	}
    return [super initWithContentRect:r styleMask:style backing:backing
	defer:defer];
    }

@end


//=============================================================================
// SokoPanel
//=============================================================================
@implementation SokoPanel (Microsoft)

//-----------------------------------------------------------------------------
// Document Edited Support
//	On Windows, -setDocumentEdited: is broken.  Even if a window/panel has
//	been excluded from the "Window" menu, as soon as -setDocumentEdited: is
//	invoked, the window suddenly appears in the menu.  Furthermore, closing
//	the window fails to remove it from the menu (even though closing a
//	window would normally remove it from the "Window" menu).  To work
//	around this problem, we simulate the functionality and ignore the
//	AppKit's own implementation.
// *TITLE*
//	-[NSWindow setTitle:] will add the "*" to the window's title for us
//	when -isDocumentEdited returns YES, however in order to force it to do
//	so, we must make it think that the title changed.
//-----------------------------------------------------------------------------
- (BOOL)isDocumentEdited { return (BOOL)(int)info; }

- (void)setDocumentEdited:(BOOL)flag
    {
    BOOL edited = (info != 0);
    if (flag != edited)
	{
	NSString* s = [self title];
	info = (void*)(int)flag;
	[self setTitle:[s stringByAppendingString:@" "]]; // *TITLE*
	[self setTitle:s];
	}
    }

@end
