//-----------------------------------------------------------------------------
// SokoURL.m
//
//	Subclass of NSTextField which displays and dispatches a URL.
//
// Copyright (c), 2001, Eric Sunshine <sunshine@sunshineco.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoURL.m,v 1.1 2001/12/24 20:16:43 sunshine Exp $
// $Log: SokoURL.m,v $
// Revision 1.1  2001/12/24 20:16:43  sunshine
// v15
// -*- Added the SokoURL class for use by the Info panel.  This class
//     displays fly-over, clickable URLs.  On MacOS/X, it knows how to
//     dispatch any URL understood by -[NSWorkspace openURL:].  On Windows,
//     it knows how to dispatch any URL understood by ShellExecute().  On
//     MacOS/X Server 1.0 (Rhapsody), OpenStep/Mach, and NextStep, it knows
//     how to dispatch mailto: URLs directly; all others are handed off to
//     OmniWeb's "Open URL" service.
//
// -*- Email addresses (and names) on Info panel are now active links which
//     launch the mail program.  The email addresses appears underlined as
//     the mouse flies over it.
//
// -*- Added a SokoSave URL to the Info panel.  The URL is an active link
//     which launches the web browser when clicked.  The URL appears
//     underlined as the mouse flies over it.
//
//-----------------------------------------------------------------------------
#import "SokoURL.h"
#import <AppKit/NSAttributedString.h>
#import <AppKit/NSCell.h>
#import <AppKit/NSColor.h>
#import <AppKit/NSCursor.h>
#import <AppKit/NSEvent.h>
#import <AppKit/NSImage.h>
#import <AppKit/NSParagraphStyle.h>
#import <AppKit/NSText.h>
#import <AppKit/NSWindow.h>
#import <Foundation/NSDictionary.h>
#import <Foundation/NSString.h>
#import <Foundation/NSValue.h>
#import <math.h>

#if defined(SOKO_URL_DISPATCHER)
   BOOL soko_dispatch_url( NSString* url );
#else
#  define soko_dispatch_url(U) (NO)
#endif

//=============================================================================
// SokoURL
//=============================================================================
@implementation SokoURL

- (BOOL)acceptsFirstMouse:(NSEvent*)e { return YES; }
- (NSString*)url { return url; }

//-----------------------------------------------------------------------------
// handCursor
//-----------------------------------------------------------------------------
+ (NSCursor*)handCursor
    {
    static NSCursor* cursor = 0;
    if (cursor == 0)
	{
	NSImage* image = [NSImage imageNamed:@"SokoHand"];
	if (image == 0)
	    cursor = [[NSCursor arrowCursor] retain];
	else
	    {
	    cursor = [[NSCursor alloc]
		initWithImage:image hotSpot:NSMakePoint(6,0)];
	    [cursor setOnMouseEntered:YES];
	    }
	}
    return cursor;
    }


//-----------------------------------------------------------------------------
// setColor:underline:
//-----------------------------------------------------------------------------
- (void)setColor:(NSColor*)c underline:(BOOL)u
    {
    NSAttributedString* a;
    NSMutableDictionary* d = [NSMutableDictionary dictionary];
    NSMutableParagraphStyle* p =
	[[NSParagraphStyle defaultParagraphStyle] mutableCopy];
    [p setAlignment:[self alignment]];
    [d setObject:p forKey:NSParagraphStyleAttributeName];
    [d setObject:c forKey:NSForegroundColorAttributeName];
    [d setObject:[self font] forKey:NSFontAttributeName];
    if (u)
	[d setObject:[NSNumber numberWithInt:NSSingleUnderlineStyle]
	    forKey:NSUnderlineStyleAttributeName];
    a = [[NSAttributedString alloc]
	initWithString:[self stringValue] attributes:d];
    [super setAttributedStringValue:a];
    [a release];
    [p release];
    }


//-----------------------------------------------------------------------------
// adjustAttributes
//-----------------------------------------------------------------------------
- (void)adjustAttributes
    {
    NSColor* c;
    if (!urlValid)
	c = [NSColor controlTextColor];
    else if (!mouseInside)
	c = [NSColor purpleColor];
    else if (trackingMouse) // && mouseInside
	c = [NSColor redColor];
    else // (mouseInside && !trackingMouse)
	c = [NSColor blueColor];
    [self setColor:c underline:(urlValid && mouseInside)];
    }


//-----------------------------------------------------------------------------
// setStringValue:
//-----------------------------------------------------------------------------
- (void)setStringValue:(NSString*)s
    {
    [super setStringValue:s];
    [self adjustAttributes];
    }


//-----------------------------------------------------------------------------
// validateUrl:
//-----------------------------------------------------------------------------
- (BOOL)validateUrl:(NSString*)u
    {
#if defined(SOKO_URL_DISPATCHER)
    return ![url isEqualToString:@""];
#else
    return NO;
#endif
    }


//-----------------------------------------------------------------------------
// setUrl:
//-----------------------------------------------------------------------------
- (void)setUrl:(NSString*)u
    {
    if (u != url)
	{
	[url release];
	url = [u copy];
	urlValid = [self validateUrl:url];
	if (!urlValid)
	    {
	    mouseInside = trackingMouse = NO;
	    if (trackTag != 0)
		[self removeTrackingRect:trackTag];
	    }
	[[self window] invalidateCursorRectsForView:self];
	[self adjustAttributes];
	}
    }


//-----------------------------------------------------------------------------
// setStringValue:url:
//-----------------------------------------------------------------------------
- (void)setStringValue:(NSString*)s url:(NSString*)u
    {
    [self setStringValue:s];
    [self setUrl:u];
    }


//-----------------------------------------------------------------------------
// dispatchUrl
//-----------------------------------------------------------------------------
- (BOOL)dispatchUrl
    {
    return (urlValid && soko_dispatch_url( url ));
    }


//-----------------------------------------------------------------------------
// textRect
//-----------------------------------------------------------------------------
- (NSRect)textRect
    {
    NSCell* c = [self cell];
    NSRect b = [c drawingRectForBounds:[self bounds]];
    NSSize s = [c cellSizeForBounds:b];
    if (s.width < b.size.width)
	{
	NSTextAlignment a = [self alignment];
	switch (a)
	    {
	    case NSLeftTextAlignment:
		b.size = s;
		break;
	    case NSRightTextAlignment:
		b.origin.x += b.size.width - s.width;
		b.size = s;
		break;
	    case NSCenterTextAlignment:
		b.origin.x += floor((b.size.width - s.width) / 2.0);
		b.size = s;
		break;
	    case NSJustifiedTextAlignment:
	    case NSNaturalTextAlignment:
		break;
	    }
	}
    return b;
    }


//-----------------------------------------------------------------------------
// visibleTextRect
//-----------------------------------------------------------------------------
- (NSRect)visibleTextRect
    {
    return NSIntersectionRect( [self textRect], [self visibleRect] );
    }


//-----------------------------------------------------------------------------
// resetCursorRects
//-----------------------------------------------------------------------------
- (void)resetCursorRects
    {
    if (urlValid)
	{
	NSRect r = [self visibleTextRect];
	[self addCursorRect:r cursor:[[self class] handCursor]];
	}
    }


//-----------------------------------------------------------------------------
// mouseEntered:
//-----------------------------------------------------------------------------
- (void)mouseEntered:(NSEvent*)e
    {
    if (urlValid)
	{
	mouseInside = YES;
	[self adjustAttributes];
	}
    }


//-----------------------------------------------------------------------------
// mouseExited:
//-----------------------------------------------------------------------------
- (void)mouseExited:(NSEvent*)e
    {
    if (urlValid)
	{
	mouseInside = NO;
	[self adjustAttributes];
	}
    }


//-----------------------------------------------------------------------------
// mouseDown:
//	Track the mouse.  If button is released while the mouse is within our
//	URL area, then dispatch the URL.
// *INSIDE*
//	It is possible for -mouseDown: to be invoked when the tracking
//	rectangle is not established.  This can happen, for instance, if the
//	window is not the key window, and the window's delegate has invoked
//	-stopTracking.  In this case, since -mouseEnter: will not have been
//	received, the `mouseInside' flag will be NO even though the mouse may
//	really be "inside".  Therefore, we must manually check the mouse's
//	position in order to ensure that `mouseInside' is correctly set.
// *DISCARD*
//	Upon mouse-up, discard other pending events which piled up while
//	tracking the mouse.  This prevents mouse-entered and mouse-exited
//	events to this URL field and others on the window from being processed
//	unnecessarily.  (They don't need to be processed because we were
//	manually tracking the mouse.)
//-----------------------------------------------------------------------------
- (void)mouseDown:(NSEvent*)e
    {
    if (urlValid)
	{
	NSRect r = [self visibleTextRect];
	NSWindow* w = [self window];
	mouseInside = [self mouse:				// *INSIDE*
	    [self convertPoint:[e locationInWindow] fromView:0] inRect:r];
	trackingMouse = YES;
	[self adjustAttributes];
	while (trackingMouse)
	    {
	    NSEvent* e = [w nextEventMatchingMask:
		NSLeftMouseUpMask | NSLeftMouseDraggedMask];
	    if ([e type] == NSLeftMouseUp)
		{
		trackingMouse = NO;
		[w discardEventsMatchingMask:			// *DISCARD*
		    (NSMouseEnteredMask | NSMouseExitedMask) beforeEvent:e];
		}
	    else
		{
		BOOL inside = [self mouse:[self convertPoint:
		    [e locationInWindow] fromView:0] inRect:r];
		if (inside != mouseInside)
		    {
		    mouseInside = inside;
		    [self adjustAttributes];
		    }
		}
	    }
	if (mouseInside)
	    [self dispatchUrl];
	[self adjustAttributes];
	}
    }


//-----------------------------------------------------------------------------
// startTracking
//-----------------------------------------------------------------------------
- (void)startTracking
    {
    if (trackTag == 0 && urlValid)
	{
	NSRect r = [self visibleTextRect];
	trackTag = [self addTrackingRect:r owner:self userData:0
	    assumeInside:NO];
	}
    }


//-----------------------------------------------------------------------------
// stopTracking
//-----------------------------------------------------------------------------
- (void)stopTracking
    {
    if (trackTag != 0)
	{
	[self removeTrackingRect:trackTag];
	trackTag = 0;
	mouseInside = NO;
	trackingMouse = NO;
	[self adjustAttributes];
	}
    }


//-----------------------------------------------------------------------------
// initWithFrame:
//-----------------------------------------------------------------------------
- (id)initWithFrame:(NSRect)r
    {
    [super initWithFrame:r];
    [self setDrawsBackground:NO];
    [self setBezeled:NO];
    [self setBordered:NO];
    [self setEditable:NO];
    [self setSelectable:NO];
    url = [@"" retain];
    urlValid = NO;
    trackTag = 0;
    mouseInside = NO;
    trackingMouse = NO;
    return self;
    }


//-----------------------------------------------------------------------------
// dealloc
//-----------------------------------------------------------------------------
- (void)dealloc
    {
    [self stopTracking];
    [url release];
    [super dealloc];
    }

@end
