//-----------------------------------------------------------------------------
// SokoURL-openstep.m
//
//	URL dispatcher for OpenStep/Mach.
//
// Copyright (c), 2001, Eric Sunshine <sunshine@sunshineco.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoURL-openstep.m,v 1.1 2001/12/23 15:30:06 sunshine Exp $
// $Log: SokoURL-openstep.m,v $
// Revision 1.1  2001/12/23 15:30:06  sunshine
// v15
// -*- Added the SokoURL class for use by the Info panel.  This class
//     displays fly-over, clickable URLs.  On MacOS/X, it knows how to
//     dispatch any URL understood by -[NSWorkspace openURL:].  On Windows,
//     it knows how to dispatch any URL understood by ShellExecute().  On
//     MacOS/X Server 1.0 (Rhapsody), OpenStep/Mach, and NextStep, it knows
//     how to dispatch mailto: URLs directly; all others are handed off to
//     OmniWeb's "Open URL" service.
//
// -*- Email addresses (and names) on Info panel are now active links which
//     launch the mail program.  The email addresses appears underlined as
//     the mouse flies over it.
//
// -*- Added a SokoSave URL to the Info panel.  The URL is an active link
//     which launches the web browser when clicked.  The URL appears
//     underlined as the mouse flies over it.
//
//-----------------------------------------------------------------------------
#import <AppKit/NSApplication.h>
#import <AppKit/NSPasteboard.h>
#import <AppKit/NSPanel.h>
#import <AppKit/NSWorkspace.h>
#import <AppKit/obsoleteListener.h>
#import <AppKit/obsoleteSpeaker.h>
#import <Foundation/NSUserDefaults.h>
#import <mach/mach_init.h>
#import <mach/mach_interface.h>
#import "SokoBuild.h"
#import "SokoRelease.h"

#define SOKO_URL_SERVICE @"OmniWeb/Open URL"

//=============================================================================
// Dispatcher for mailto: URLs.
//	Subclass of old NextStep Speaker class.  This works for OpenStep/Mach
//	because Mail.app is still a NextStep application under OpenStep/Mach
//	(that is, it is still based upon the old NextStep API rather than the
//	newer OpenStep API).
//=============================================================================
@interface SokoMailSpeaker : Speaker
    {
    port_t port;
    }
- (id)init;
- (BOOL)sendMailTo:(NSString*)to subject:(NSString*)subject;
@end

@implementation SokoMailSpeaker

//-----------------------------------------------------------------------------
// getMailerPort
//	Get the Mach port advertised by the GUI mailer program.  If no program
//	is advertising that port, try launching the mailer program specified
//	by the user default "Mailer".  If that fails, try launching Mail.app.
//-----------------------------------------------------------------------------
+ (port_t)getMailerPort
    {
    NSString* host = @"localhost";
    port_t p = NXPortNameLookup( @"MailSendDemo", host );
    if (p == PORT_NULL)
	{ // Mailer not running, try "Mailer" from defaults, then "Mail.app".
	BOOL launched = NO;
	NSWorkspace* w = [NSWorkspace sharedWorkspace];
	NSString* mailer =
	    [[NSUserDefaults standardUserDefaults] stringForKey:@"Mailer"];
	if (mailer != 0 && ![mailer isEqualToString:@""])
	    launched = [w launchApplication:mailer];
	if (!launched)
	    launched = [w launchApplication:@"Mail"];
	if (launched)
	    p = NXPortFromName( @"MailSendDemo", host );
	}
    return p;
    }


//-----------------------------------------------------------------------------
// init
//-----------------------------------------------------------------------------
- (id)init
    {
    [super init];
    port = [[self class] getMailerPort];
    if (port != PORT_NULL)
	[self setSendPort:port];
    return self;
    }


//-----------------------------------------------------------------------------
// dealloc
//-----------------------------------------------------------------------------
- (void)dealloc
    {
    if (port != PORT_NULL)
	port_deallocate( task_self(), port );
    [super dealloc];
    }


//-----------------------------------------------------------------------------
// sendMailTo:subject:
//	Open a compose window in the GUI mailer program and fill in the To:
//	and Subject: fields.
//-----------------------------------------------------------------------------
- (BOOL)sendMailTo:(NSString*)to subject:(NSString*)subject
    {
    int composer;
    return (port != PORT_NULL &&
	[self selectorRPC:@"openSend:" paramTypes:(char*)"I", &composer]==0 &&
	[self selectorRPC:@"setTo:inWindow:" paramTypes:(char*)"ci",
	    [to lossyCString], composer]==0 &&
	[self selectorRPC:@"setSubject:inWindow:" paramTypes:(char*)"ci",
	    [subject lossyCString], composer]==0);
    }

@end


//=============================================================================
// soko_dispatch_url
//	Figure out the correct mechanism for dispatching a particular URL
//	and invoke that mechanism.
//
// *SERVICE*
//	There is an apparent bug with the implementaiton of NSPerformService()
//	on OpenStep where it corrupts the incoming pasteboard.  The symptom is
//	that, after the first invocation of NSPerformService(), all subsequent
//	invocations of -declareTypes:owner: & -setString:forType: cause an
//	exception to be thrown.  This problem only occurs if NSPerformService()
//	is invoked.  If it is not invoked, then the pasteboard behaves normally
//	and can be re-used any number of times, as documented, by sending
//	-declareTypes:owner: & setString:forType:.  To work around this
//	problem, it is necessary to create a brand new pasteboard for each
//	invocation of NSPerformService(), so as to avoid re-using the corrupted
//	pasteboard.  Note, also, that the program to which the service request
//	is sent may not be able to handle the request immediately, so we can
//	not destroy the used-pasteboard right away, since it must still exist
//	at the time when the receiving application finally gets around to
//	handling the service request.  Therefore, we only destroy the
//	pasteboard the next time we are asked to dispatch a URL.
//=============================================================================
BOOL soko_dispatch_url( NSString* url )
    {
    BOOL ok = NO;
    if ([[url lowercaseString] hasPrefix:@"mailto:"])
	{
	NSString* to = [url substringFromIndex:[@"mailto:" length]];
	NSString* subject = [NSString stringWithFormat:
	    @"SokoSave feedback (release %s; build %s; OpenStep/Mach)",
	    SOKO_RELEASE, SOKO_BUILD];
	SokoMailSpeaker* speaker = [[SokoMailSpeaker alloc] init];
	ok = [speaker sendMailTo:to subject:subject];
	[speaker release];
	}
    else
	{
	NS_DURING
	    static NSPasteboard* pb = 0;	// *SERVICE*
	    if (pb != 0)
		{
		[pb releaseGlobally];
		[pb release];
		}
	    pb = [[NSPasteboard pasteboardWithUniqueName] retain];
	    [pb declareTypes:[NSArray arrayWithObject:NSStringPboardType]
		owner:0];
	    [pb setString:url forType:NSStringPboardType];
	    ok = NSPerformService( SOKO_URL_SERVICE, pb );
	NS_HANDLER
	    NSRunCriticalAlertPanel( @"SokoSave",
		@"Communication error while dispatching URL: %@", 0,0,0, url );
	NS_ENDHANDLER
	}
    return ok;
    }
