//-----------------------------------------------------------------------------
// SokoScores.m
//
//	"Top-Scores" object for SokoSave.
//
// Copyright (c), 1997,2001,2002, Eric Sunshine <sunshine@sunshineco.com>
// Copyright (c), 1997, Paul McCarthy <zarnuk@high-speed-software.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoScores.m,v 1.6 2002/01/29 20:33:27 sunshine Exp $
// $Log: SokoScores.m,v $
// Revision 1.6  2002/01/29 20:33:27  sunshine
// v17
// -*- SokoPuzzle now calculates two new scores in addition to "moves" and
//     "pushes".  The "runs" score is the number of straight lines in which
//     crates have been pushed.  Looked at another way, it is the number of
//     turns crates have made while being pushed.  The "focus" score is the
//     number of times the player's focus has changed from one crate to
//     another.
//
// -*- Added "runs" and "focus" scores columns to scores panel and to "New
//     Score" panel.
//
// Revision 1.5  2001/12/23 17:45:49  sunshine
// v15
// -*- Extracted core game logic out of GUI code and generalized it so that
//     the same core code can be used by any platform.  Logic from
//     SokoBoard.m now resides in SokoPuzzle.c, etc.
//
// -*- Rewrote the OpenStep port so that it utilizes the new "sokocore" game
//     library rather than implementing that logic directly.
//
// -*- Split new-score entry panel and functionality out of SokoScores and
//     into its own module, SokoNewScore.
//
// -*- For safety, SCORES is now written out immediately after a new score is
//     recorded, rather than only at program termination time.
//
// -*- Now remembers username from New Score panel as default for that panel
//     as a convenience to users of Windows 9x who don't actually login with
//     a username.
//
// -*- For safety, user defaults (including score panel column order and
//     widths) are now committed immediately, rather than at termination
//     time.
//
// -*- Replaced the terminology "maze" with "puzzle" throughout the project,
//     including source code, documentation, and all user-visible UI
//     elements.  The only remaining place where "maze" is still used is in
//     the file extension ".sokomaze".  I haven't decided what, if anything,
//     to do about that, yet.
//
// -*- Renamed "window" instance variable to "panel" for all classes and nibs
//     in which a panel is actually used (everything but SokoBoard, in fact).
//
// -*- Added SokoAlerter which provides a convenience function used by the
//     new SokoPuzzleDelegate and SokoScoreDelegate implementations to send
//     alert messages to the user from those core modules.
//
// -*- Augmented all input/output logic so that it now deals gracefully with
//     line terminators from all common platforms; Unix (LF), Macintosh (CR),
//     and Windows/DOS (CRLF).
//
// -*- Added SokoSetting implementation which provides a platform-independent
//     API for accessing user settings and well-known paths, such as
//     $(SokoSave) and $(SokoUser).
//
// -*- Added SokoWindow and SokoPanel, which are subclasses of NSWindow and
//     NSPanel, respectively.  All windows and panels throughout the
//     application are now instances of these classes.  This allows bugs in
//     the AppKit window and panel classes to be overcome more easily.  For
//     instance, the Windows port works around several Windows-specific
//     AppKit bugs by overriding certain methods in these classes.
//-----------------------------------------------------------------------------
#import "SokoPool.h"
#import "SokoScores.h"
#import "SokoAlerter.h"
#import "SokoFile.h"
#import "SokoSetting.h"
#import "SokoWindow.h"
#import <AppKit/NSApplication.h>
#import <AppKit/NSCell.h>
#import <AppKit/NSImage.h>
#import <AppKit/NSNibLoading.h>
#import <AppKit/NSTableColumn.h>
#import <AppKit/NSTableView.h>
#import <AppKit/NSTextField.h>
#import <Foundation/NSArray.h>
#import <Foundation/NSDateFormatter.h>
#import <Foundation/NSDictionary.h>
#import <Foundation/NSEnumerator.h>
#import <Foundation/NSString.h>
#import <Foundation/NSUserDefaults.h>

#define PUZZLE_KEY @"puzzle"
#define MOVE_KEY   @"moves"
#define PUSH_KEY   @"pushes"
#define RUN_KEY    @"runs"
#define FOCUS_KEY  @"focus"
#define NAME_KEY   @"name"
#define DATE_KEY   @"date"
#define NOTE_KEY   @"notes"

static struct _SokoScoreDelegate SCORE_DELEGATE;

@interface NSTableView(QVController)
- (BOOL)acceptsFirstMouse:(NSEvent*)p;
@end
@implementation NSTableView(QVController)
- (BOOL)acceptsFirstMouse:(NSEvent*)p { return YES; }
@end

static id get_def( NSString* name )
    { return [[NSUserDefaults standardUserDefaults] objectForKey:name]; }

static void set_def( NSString* name, id value )
    { [[NSUserDefaults standardUserDefaults] setObject:value forKey:name]; }

static void score_alert( SokoScore score, SokoScoreDelegate delegate,
    SokoAlert severity, char const* title, char const* msg )
    { SokoSendAlert( severity, title, msg ); }


//=============================================================================
// IMPLEMENTATION
//=============================================================================
@implementation SokoScores

//-----------------------------------------------------------------------------
// sortScores
//-----------------------------------------------------------------------------
- (void)sortScores
    {
    int i, n = [tableView numberOfColumns];
    NSArray* columns = [tableView tableColumns];
    SokoScoreSort* criteria = (SokoScoreSort*)malloc(n * sizeof(criteria[0]));
    for (i = 0; i < n; i++)
	{
	id ident = [[columns objectAtIndex:i] identifier];
	if ([ident isEqual:PUZZLE_KEY])
	    criteria[i] = SOKO_SCORE_PUZZLE;
	else if ([ident isEqual:MOVE_KEY])
	    criteria[i] = SOKO_SCORE_MOVES;
	else if ([ident isEqual:PUSH_KEY])
	    criteria[i] = SOKO_SCORE_PUSHES;
	else if ([ident isEqual:RUN_KEY])
	    criteria[i] = SOKO_SCORE_RUNS;
	else if ([ident isEqual:FOCUS_KEY])
	    criteria[i] = SOKO_SCORE_FOCUS;
	else if ([ident isEqual:NAME_KEY])
	    criteria[i] = SOKO_SCORE_PLAYER;
	else if ([ident isEqual:DATE_KEY])
	    criteria[i] = SOKO_SCORE_DATE;
	else if ([ident isEqual:NOTE_KEY])
	    criteria[i] = SOKO_SCORE_NOTES;
	}
    SokoScore_sort( scores, criteria, n );
    free( criteria );
    }


//-----------------------------------------------------------------------------
// setColumnOrder:
//-----------------------------------------------------------------------------
- (void)setColumnOrder:(NSArray*)order
    {
    int i, lim = [order count];
    if (lim == [tableView numberOfColumns])
	for (i = 0; i < lim; i++)
	    [tableView moveColumn:[tableView columnWithIdentifier:
		[order objectAtIndex:i]] toColumn:i];
    }


//-----------------------------------------------------------------------------
// setColumnSizes:
//-----------------------------------------------------------------------------
- (void)setColumnSizes:(NSDictionary*)sizes
    {
    if ([sizes count] == [tableView numberOfColumns])
	{
	NSEnumerator* e = [[tableView tableColumns] objectEnumerator];
	id col;
	while ((col = [e nextObject]) != 0)
	    {
	    id size = [sizes objectForKey:[col identifier]];
	    if (size != 0)
		[col setWidth:[size floatValue]];
	    }
	}
    }


//-----------------------------------------------------------------------------
// columnOrder
//-----------------------------------------------------------------------------
- (NSArray*)columnOrder
    {
    NSMutableArray* columns = [NSMutableArray array];
    NSEnumerator* enumerator = [[tableView tableColumns] objectEnumerator];
    id col;
    while ((col = [enumerator nextObject]) != 0)
	[columns addObject:[col identifier]];
    return columns;
    }


//-----------------------------------------------------------------------------
// columnSizes
//-----------------------------------------------------------------------------
- (NSDictionary*)columnSizes
    {
    NSMutableDictionary* sizes = [NSMutableDictionary dictionary];
    NSEnumerator* enumerator = [[tableView tableColumns] objectEnumerator];
    id col;
    while ((col = [enumerator nextObject]) != 0)
	[sizes setObject:[[NSNumber numberWithFloat:[col width]] description]
		forKey:[col identifier]];
    return sizes;
    }


//-----------------------------------------------------------------------------
// NSTableView delegate & data source methods
//-----------------------------------------------------------------------------
- (void)tableViewColumnDidResize:(NSNotification*)notification
    {
    set_def( @"ScoreColSizes", [self columnSizes] );
    }

- (void)tableViewColumnDidMove:(NSNotification*)notification
    {
    set_def( @"ScoreColOrder", [self columnOrder] );
    [self sortScores];
    [tableView reloadData];
    }

- (int)numberOfRowsInTableView:(NSTableView*)aTableView
    {
    return SokoScore_score_count( scores );
    }

- (id)objectWithInt:(int)n
    {
    if (n >= 0)
	return [NSNumber numberWithInt:n];
    return @"";
    }

- (id)tableView:(NSTableView*)sender
    objectValueForTableColumn:(NSTableColumn*)column row:(int)row
    {
    SokoHighScore const* s = SokoScore_get_score( scores, row );
    NSString* key = [column identifier];
    if ([key isEqualToString:PUZZLE_KEY])
	return [NSString stringWithCString:s->puzzle];
    else if ([key isEqualToString:MOVE_KEY])
	return [self objectWithInt:s->moves];
    else if ([key isEqualToString:PUSH_KEY])
	return [self objectWithInt:s->pushes];
    else if ([key isEqualToString:RUN_KEY])
	return [self objectWithInt:s->runs];
    else if ([key isEqualToString:FOCUS_KEY])
	return [self objectWithInt:s->focus];
    else if ([key isEqualToString:NAME_KEY])
	return [NSString stringWithCString:s->player];
    else if ([key isEqualToString:DATE_KEY])
	return [NSDate dateWithTimeIntervalSince1970:s->date];
    else if ([key isEqualToString:NOTE_KEY])
	return [NSString stringWithCString:s->notes];
    else
	return 0;
    }


//-----------------------------------------------------------------------------
// init
//	Note: `scores' object must be initialized before nib is loaded since
//	at nib load time, NSTableView calls -numberOfRowsInTableView.
//-----------------------------------------------------------------------------
- (id)init
    {
    id def;
    [super init];

    SCORE_DELEGATE.alert_callback = score_alert;
    SCORE_DELEGATE.info = 0;
    scores = SokoScore_new( &SCORE_DELEGATE, 0 );

    [NSBundle loadNibNamed:@"SokoScores" owner:self];
    [panel setMiniwindowImage:[NSImage imageNamed:@"SokoSave"]];
    [panel setFrameAutosaveName:@"SokoScores"];
    [[[tableView tableColumnWithIdentifier:DATE_KEY] dataCell]
	setFormatter:[[[NSDateFormatter allocWithZone:[tableView zone]]
	initWithDateFormat:get_def(NSShortTimeDateFormatString)
	allowNaturalLanguage:NO] autorelease]];

    if ((def = get_def( @"ScoreColSizes" )) != 0)
	[self setColumnSizes:def];
    if ((def = get_def( @"ScoreColOrder" )) != 0)
	[self setColumnOrder:def];

    [self sortScores];
    [tableView reloadData];
    return self;
    }


//-----------------------------------------------------------------------------
// dealloc
//-----------------------------------------------------------------------------
- (void)dealloc
    {
    [panel setDelegate:0];
    [panel close];
    [panel release];
    SokoScore_destroy( scores );
    [super dealloc];
    }


//-----------------------------------------------------------------------------
// activate
//-----------------------------------------------------------------------------
- (void)activate
    {
    [panel makeKeyAndOrderFront:self];
    }


//-----------------------------------------------------------------------------
// performPrint:
//-----------------------------------------------------------------------------
- (void)performPrint:(id)sender
    {
    [tableView print:self];
    }


//-----------------------------------------------------------------------------
// solved:moves:pushes:runs:focus:name:notes:
//-----------------------------------------------------------------------------
- (void)solved:(SokoPuzzle)puzzle
    moves:(int)moves
    pushes:(int)pushes
    runs:(int)runs
    focus:(int)focus
    name:(NSString*)name
    notes:(NSString*)notes
    {
    int const row = SokoScore_add( scores, puzzle, moves, pushes, runs, focus,
	[name lossyCString], [notes lossyCString] );
    [tableView noteNumberOfRowsChanged];
    [tableView selectRow:row byExtendingSelection:NO];
    [tableView scrollRowToVisible:row];
    [tableView reloadData];
    }


//-----------------------------------------------------------------------------
// globalInstance
//-----------------------------------------------------------------------------
+ (SokoScores*)globalInstance
    {
    static SokoScores* instance = 0;
    if (instance == 0)
	instance = [[self alloc] init];
    return instance;
    }


//-----------------------------------------------------------------------------
// launch
//-----------------------------------------------------------------------------
+ (void)launch
    {
    [[self globalInstance] activate];
    }


//-----------------------------------------------------------------------------
// solved:moves:pushes:runs:focus:name:notes:
//-----------------------------------------------------------------------------
+ (void)solved:(SokoPuzzle)puzzle
    moves:(int)moves
    pushes:(int)pushes
    runs:(int)runs
    focus:(int)focus
    name:(NSString*)name
    notes:(NSString*)notes
    {
    SokoScores* p = [self globalInstance];
    [p solved:puzzle
	moves:moves pushes:pushes runs:runs focus:focus name:name notes:notes];
    [p activate];
    }

@end
