//-----------------------------------------------------------------------------
// SokoPref.m
//
//	User preferences module for SokoSave.
//
// Copyright (c), 1997,2001, Eric Sunshine <sunshine@sunshineco.com>
// Copyright (c), 1997, Paul McCarthy <zarnuk@high-speed-software.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoPref.m,v 1.4 2001/12/23 18:02:06 sunshine Exp $
// $Log: SokoPref.m,v $
// Revision 1.4  2001/12/23 18:02:06  sunshine
// v15
// -*- Extracted core game logic out of GUI code and generalized it so that
//     the same core code can be used by any platform.  Logic from
//     SokoBoard.m now resides in SokoPuzzle.c, etc.
//
// -*- Rewrote the OpenStep port so that it utilizes the new "sokocore" game
//     library rather than implementing that logic directly.
//
// -*- Replaced the terminology "maze" with "puzzle" throughout the project,
//     including source code, documentation, and all user-visible UI
//     elements.  The only remaining place where "maze" is still used is in
//     the file extension ".sokomaze".  I haven't decided what, if anything,
//     to do about that, yet.
//
// -*- Added the new pseudo-variable $(SokoUser) which points at the user's
//     "personal" file space.  This is where user-specific SokoSave files are
//     stored by default.  This variable complements the existing $(SokoSave)
//     variable.
//
// -*- Renamed soko_collapse() to soko_collapse_path().  Renamed
//     soko_expand() to soko_expand_path().
//
// -*- The path setting functions in SokoFile now invoke soko_collapse_path()
//     on all set operations.  This ensures that all paths displayed on the
//     preferences panel are properly collapsed, and removes the onus of this
//     task from the UI code.  Previously, soko_collapse_path() was only
//     called by the UI code for the puzzle path.
//
// -*- Added soko_normalize_path() and soko_denormalize_path() to SokoUtil.
//     All path manipulation functions now utilize these functions in order
//     to ensure correctness of path parsing and composition on all
//     platforms.
//
// -*- Added soko_set_puzzle_directory(), soko_set_save_directory(), and
//     soko_set_score_file() to SokoFile in order to complement the existing
//     "get" functions and to centralize control over these settings.
//
// -*- Added soko_get_default_puzzle_directory(),
//     get_default_save_directory(), and soko_get_default_score_file() to
//     SokoFile.  These functions return values appropriate for the
//     corresponding fields on the Preferences panel of each port when the
//     "Defaults" button is pressed.
//
// -*- Added SokoSetting implementation which provides a platform-independent
//     API for accessing user settings and well-known paths, such as
//     $(SokoSave) and $(SokoUser).
//
// -*- For safety, user defaults (including score panel column order and
//     widths) are now committed immediately, rather than at termination
//     time.
//
// -*- No longer assumes that default location for user-files is
//     $(HOME)/Library, in all cases.  Now uses
//     NSSearchPathForDirectoriesInDomains() to resolve $(SokoUser) if
//     available.  It is available on MacOS/X (Cocoa) and MacOS/X Server 1.0
//     (Rhapsody).  On MacOS/X, "Application Support/SokoSave" is appended to
//     the result of NSSearchPathForDirectoriesInDomains().  In other cases,
//     only "SokoSave" is appended.  For Windows, $(SokoUser) expands to the
//     "personal" folder (typically "My Documents"), if it exists.  If not,
//     then it expands to the first of the following which exist: $(HOME),
//     $(TEMP), $(TMP), root of drive containing "windows" directory, or root
//     of current drive.
//
// -*- Added SokoWindow and SokoPanel, which are subclasses of NSWindow and
//     NSPanel, respectively.  All windows and panels throughout the
//     application are now instances of these classes.  This allows bugs in
//     the AppKit window and panel classes to be overcome more easily.  For
//     instance, the Windows port works around several Windows-specific
//     AppKit bugs by overriding certain methods in these classes.
//
// -*- Renamed "window" instance variable to "panel" for all classes and nibs
//     in which a panel is actually used (everything but SokoBoard, in fact).
//
// -*- Worked around an AppKit bug on Windows in which the Preferences panel
//     would suddenly appear on the "Window" menu when -setDocumentEdited:
//     was invoked even though panels are not supposed to appear on this
//     menu.  This problem was further compounded by the fact that the menu
//     item would remain dangling on the menu even after the Preferences
//     panel was closed.
//
// Revision 1.3  2001/08/19 12:51:40  sunshine
// v11.1
// -*- Converted from Objective-C++ to pure Objective-C.  Changed file
//     extensions from .M to .m, and .cc to .c.  This change makes it easier
//     to support both GNUstep and MacOS/X, neither of which feature an
//     Objective-C++ compiler (though MacOS/X will reportedly support
//     Objective-C++ at some point in the future).
//
// -*- Renamed the following functions in order to avoid potential symbolic
//     collisions in the target environment:
//
//     collapse --> soko_collapse
//     directoryPart --> soko_directory_part
//     expand --> soko_expand
//     filenamePart --> soko_filename_part
//     getFactoryScoresFile --> soko_get_factory_scores_file
//     getMazeDirectory --> soko_get_maze_directory
//     getMazeExtension --> soko_get_maze_extension
//     getSaveDirectory --> soko_get_save_directory
//     getSaveExtension --> soko_get_save_extension
//     getScoresFile --> soko_get_scores_file
//     mazeNameForLevel --> soko_maze_name_for_level
//     mkdirs --> soko_mkdirs
//     runLengthDecodeChar --> soko_run_length_decode_char
//     runLengthDecodeString --> soko_run_length_decode_string
//     runLengthEncodeChar --> soko_run_length_encode_char
//     runLengthEncodeString --> soko_run_length_encode_string
//     saveFileNameForMazeName --> soko_save_filename_for_maze_name
//-----------------------------------------------------------------------------
#import	"SokoPool.h"
#import "SokoPref.h"
#import "SokoFile.h"
#import "SokoUtil.h"
#import "SokoWindow.h"
#import <AppKit/NSButton.h>
#import <AppKit/NSNibLoading.h>
#import <AppKit/NSTextField.h>
#import <Foundation/NSAutoreleasePool.h>
#import <Foundation/NSBundle.h>
#import <Foundation/NSPathUtilities.h>
#import <Foundation/NSString.h>
#import <Foundation/NSUserDefaults.h>
char const* soko_application_data_path( SokoPool );

static SokoPref* COMMON_INSTANCE = 0;
static struct _SokoSettingDelegate SETTING_DELEGATE;

#define S2NS(S) [NSString stringWithCString:S]
#define NS2S(S) [S cString]

#define STD_DEF [NSUserDefaults standardUserDefaults]
#define DEL_FUNC_0(RET,NAME) \
    static RET NAME( SokoSetting x1, SokoSettingDelegate x2 )
#define DEL_FUNC_1(RET,NAME,T1,A1) \
    static RET NAME( SokoSetting x1, SokoSettingDelegate x2, T1 A1 )
#define DEL_FUNC_2(RET,NAME,T1,A1,T2,A2) \
    static RET NAME( SokoSetting x1, SokoSettingDelegate x2, T1 A1, T2 A2 )

DEL_FUNC_1(soko_bool, setting_exists, char const*,name)
    {
    NSAutoreleasePool* pool = [[NSAutoreleasePool alloc] init];
    BOOL exists = ([STD_DEF objectForKey:S2NS(name)] != 0);
    [pool release];
    return exists;
    }

DEL_FUNC_2(char const*, get_setting, char const*,name, SokoPool,pool)
    {
    NSAutoreleasePool* nspool = [[NSAutoreleasePool alloc] init];
    char const* s = 0;
    NSString* value = [STD_DEF stringForKey:S2NS(name)];
    if (value != 0)
	s = SokoPool_store( pool, [value cString] );
    [nspool release];
    return s;
    }

DEL_FUNC_2(void, set_setting, char const*,name, char const*,value)
    {
    NSAutoreleasePool* pool = [[NSAutoreleasePool alloc] init];
    [STD_DEF setObject:S2NS(value) forKey:S2NS(name)];
    [pool release];
    }

DEL_FUNC_1(void, remove_setting, char const*,name)
    {
    NSAutoreleasePool* pool = [[NSAutoreleasePool alloc] init];
    NSString* key = S2NS(name);
    NSUserDefaults* defs = STD_DEF;
    if ([defs objectForKey:key] != 0)
	[defs removeObjectForKey:key];
    [pool release];
    }

DEL_FUNC_1(char const*, factory_path, SokoPool,pool)
    {
    static char const* path = 0;
    if (path == 0)
	{
	NSAutoreleasePool* nspool = [[NSAutoreleasePool alloc] init];
	path = soko_strdup(
	    [[[NSBundle mainBundle] resourcePath] fileSystemRepresentation] );
	[nspool release];
	}
    return path;
    }

DEL_FUNC_1(char const*, user_path, SokoPool,pool)
    {
    static char* path = 0;
    if (path == 0)
	{
	NSAutoreleasePool* nspool = [[NSAutoreleasePool alloc] init];
	path = soko_strdup( soko_add_path_component(
	    pool, soko_application_data_path(pool), "SokoSave") );
	[nspool release];
	}
    return path;
    }

#undef DEL_FUNC_2
#undef DEL_FUNC_1
#undef DEL_FUNC_0
#undef STD_DEF

//=============================================================================
// SokoPref Implementation
//=============================================================================
@implementation SokoPref

//-----------------------------------------------------------------------------
// loadDefaults
//-----------------------------------------------------------------------------
- (void)loadDefaults
    {
    SokoPool pool = SokoPool_new(0);
    [saveDirField setStringValue:S2NS(soko_get_default_save_directory(pool))];
    [autoSaveSwitch setState:1];
    [scoresFileField setStringValue:S2NS(soko_get_default_score_file(pool))];
    [puzzleDirField setStringValue:
	S2NS(soko_get_default_puzzle_directory(pool))];
    [levelField setIntValue:1];
    SokoPool_destroy( pool );
    }


//-----------------------------------------------------------------------------
// loadSettings
//	*NOTE* Ensure that editing is not in progress, since (under some AppKit
//	versions) setting a field while it is being edited is a no-op.
//-----------------------------------------------------------------------------
- (void)loadSettings
    {
    SokoPool pool = SokoPool_new(0);
    [panel makeFirstResponder:panel];	// *NOTE*
    [saveDirField setStringValue:S2NS(soko_get_save_directory( pool ))];
    [autoSaveSwitch setState:
	SokoSetting_get_bool( setting, SOKO_SETTING_AUTO_SAVE, soko_true )];
    [scoresFileField setStringValue:S2NS(soko_get_score_file( pool ))];
    [puzzleDirField setStringValue:S2NS(soko_get_puzzle_directory( pool ))];
    [levelField setIntValue:SokoSetting_get_int(setting,SOKO_SETTING_LEVEL,1)];
    [panel setDocumentEdited:NO];
    SokoPool_destroy( pool );
    }


//-----------------------------------------------------------------------------
// saveSettings
//	*POST-SAVE* Load settings in case some settings were rejected by "save"
//	logic.  Loading settings also clears "dirty" flag as a side-effect.
//-----------------------------------------------------------------------------
- (void)saveSettings
    {
    SokoPool pool = SokoPool_new(0);
    int level;
    NSString* s;
    s = [saveDirField stringValue];
    if (![s isEqualToString:@""])
	soko_set_save_directory( pool, NS2S(s) );
    SokoSetting_set_bool(
	setting, SOKO_SETTING_AUTO_SAVE, [autoSaveSwitch state] );
    s = [scoresFileField stringValue];
    if (![s isEqualToString:@""])
	soko_set_score_file( pool, NS2S(s) );
    s = [puzzleDirField stringValue];
    if (![s isEqualToString:@""])
	soko_set_puzzle_directory( pool, NS2S(s) );
    level = [levelField intValue];
    if (level >= 1)
	SokoSetting_set_int( setting, SOKO_SETTING_LEVEL, level );
    [[NSUserDefaults standardUserDefaults] synchronize];
    [self loadSettings];	// *POST-SAVE*
    SokoPool_destroy( pool );
    }


//-----------------------------------------------------------------------------
// madeDirty:
//-----------------------------------------------------------------------------
- (void)madeDirty:(id)sender
    {
    [panel setDocumentEdited:YES];
    }


//-----------------------------------------------------------------------------
// doSet:
//-----------------------------------------------------------------------------
- (void)doSet:(id)sender
    {
    [panel close];
    [self saveSettings];
    }


//-----------------------------------------------------------------------------
// doCancel:
//-----------------------------------------------------------------------------
- (void)doCancel:(id)sender
    {
    [panel close];
    [self loadSettings];
    }


//-----------------------------------------------------------------------------
// doDefaults:
//-----------------------------------------------------------------------------
- (void)doDefaults:(id)sender
    {
    [self loadDefaults];
    [panel setDocumentEdited:YES];
    }


//-----------------------------------------------------------------------------
// controlTextDidChange:
//-----------------------------------------------------------------------------
- (void)controlTextDidChange:(NSNotification*)notification
    {
    [self madeDirty:self];
    }


//-----------------------------------------------------------------------------
// windowShouldClose:
//-----------------------------------------------------------------------------
- (BOOL)windowShouldClose:(id)sender
    {
    BOOL ok = YES;
    if ([panel isDocumentEdited])
	{
	int const rc = NSRunAlertPanel( @"Preferences",
	    @"Save your changes to the preferences?",
	    @"Save", @"Don't Save", @"Cancel" );
	if (rc == NSAlertDefaultReturn)
	    [self saveSettings];
	else if (rc == NSAlertAlternateReturn)
	    [self loadSettings];
	else // (rc == NSAlertOtherReturn)
	    ok = NO;
	}
    return ok;
    }


//-----------------------------------------------------------------------------
// setNextLevel:
//-----------------------------------------------------------------------------
- (void)setNextLevel:(int)level
    {
    [levelField setIntValue:level];
    }


//-----------------------------------------------------------------------------
// activate
//-----------------------------------------------------------------------------
- (void)activate
    {
    [panel makeKeyAndOrderFront:self];
    }


//-----------------------------------------------------------------------------
// init
//-----------------------------------------------------------------------------
- (id)init
    {
    [super init];
    [NSBundle loadNibNamed:@"SokoPref" owner:self];
    [panel setFrameAutosaveName:@"SokoPref"];
    setting = SokoSetting_new(0);
    [self loadSettings];
    return self;
    }


//-----------------------------------------------------------------------------
// dealloc
//-----------------------------------------------------------------------------
- (void)dealloc
    {
    [panel setDelegate:0];
    [panel close];
    [panel release];
    SokoSetting_destroy( setting );
    [super dealloc];
    }


//-----------------------------------------------------------------------------
// startup
//-----------------------------------------------------------------------------
+ (void)startup
    {
    SETTING_DELEGATE.exists_callback = setting_exists;
    SETTING_DELEGATE.get_callback = get_setting;
    SETTING_DELEGATE.set_callback = set_setting;
    SETTING_DELEGATE.remove_callback = remove_setting;
    SETTING_DELEGATE.factory_path_callback = factory_path;
    SETTING_DELEGATE.user_path_callback = user_path;
    SETTING_DELEGATE.info = 0;
    SokoSetting_set_delegate( &SETTING_DELEGATE );
    }


//-----------------------------------------------------------------------------
// shutdown
//-----------------------------------------------------------------------------
+ (void)shutdown
    {
    if (COMMON_INSTANCE != 0)
	{
	[COMMON_INSTANCE release];
	COMMON_INSTANCE = 0;
	}
    SokoSetting_set_delegate(0);
    [[NSUserDefaults standardUserDefaults] synchronize];
    }


//-----------------------------------------------------------------------------
// getNextLevel
//-----------------------------------------------------------------------------
+ (int)getNextLevel
    {
    SokoSetting s = SokoSetting_new(0);
    int level = SokoSetting_get_int( s, SOKO_SETTING_LEVEL, 1 );
    SokoSetting_destroy(s);
    return level;
    }


//-----------------------------------------------------------------------------
// setNextLevel:
//	*NOTE* If the preference panel has been instantiated, then also set the
//	value of the "next level" field.  This works correctly whether or not
//	the panel is currently visible.  Even if it is visible, just setting
//	the value of the field is sufficient; there is no need to set the
//	"dirty" flag (since the new setting is already saved).
//-----------------------------------------------------------------------------
+ (void)setNextLevel:(int)n
    {
    if (n >= 0)
	{
	SokoSetting s = SokoSetting_new(0);
	SokoSetting_set_int( s, SOKO_SETTING_LEVEL, n );
	if (COMMON_INSTANCE != 0)
	    [COMMON_INSTANCE setNextLevel:n];
	SokoSetting_destroy(s);
	}
    }


//-----------------------------------------------------------------------------
// launch
//-----------------------------------------------------------------------------
+ (void)launch
    {
    if (COMMON_INSTANCE == 0)
	COMMON_INSTANCE = [[self alloc] init];
    [COMMON_INSTANCE activate];
    }

@end

#undef S2NS
#undef NS2S
