//-----------------------------------------------------------------------------
// SokoPref-windows.m
//
//	Preference related functionality for Microsoft Windows.
//
// Copyright (c), 2001, Eric Sunshine <sunshine@sunshineco.com>
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoPref-windows.c,v 1.1 2001/12/23 18:02:36 sunshine Exp $
// $Log: SokoPref-windows.c,v $
// Revision 1.1  2001/12/23 18:02:36  sunshine
// v15
// -*- Extracted core game logic out of GUI code and generalized it so that
//     the same core code can be used by any platform.  Logic from
//     SokoBoard.m now resides in SokoPuzzle.c, etc.
//
// -*- Rewrote the OpenStep port so that it utilizes the new "sokocore" game
//     library rather than implementing that logic directly.
//
// -*- Replaced the terminology "maze" with "puzzle" throughout the project,
//     including source code, documentation, and all user-visible UI
//     elements.  The only remaining place where "maze" is still used is in
//     the file extension ".sokomaze".  I haven't decided what, if anything,
//     to do about that, yet.
//
// -*- Added the new pseudo-variable $(SokoUser) which points at the user's
//     "personal" file space.  This is where user-specific SokoSave files are
//     stored by default.  This variable complements the existing $(SokoSave)
//     variable.
//
// -*- Renamed soko_collapse() to soko_collapse_path().  Renamed
//     soko_expand() to soko_expand_path().
//
// -*- The path setting functions in SokoFile now invoke soko_collapse_path()
//     on all set operations.  This ensures that all paths displayed on the
//     preferences panel are properly collapsed, and removes the onus of this
//     task from the UI code.  Previously, soko_collapse_path() was only
//     called by the UI code for the puzzle path.
//
// -*- Added soko_normalize_path() and soko_denormalize_path() to SokoUtil.
//     All path manipulation functions now utilize these functions in order
//     to ensure correctness of path parsing and composition on all
//     platforms.
//
// -*- Added soko_set_puzzle_directory(), soko_set_save_directory(), and
//     soko_set_score_file() to SokoFile in order to complement the existing
//     "get" functions and to centralize control over these settings.
//
// -*- Added soko_get_default_puzzle_directory(),
//     get_default_save_directory(), and soko_get_default_score_file() to
//     SokoFile.  These functions return values appropriate for the
//     corresponding fields on the Preferences panel of each port when the
//     "Defaults" button is pressed.
//
// -*- Added SokoSetting implementation which provides a platform-independent
//     API for accessing user settings and well-known paths, such as
//     $(SokoSave) and $(SokoUser).
//
// -*- For safety, user defaults (including score panel column order and
//     widths) are now committed immediately, rather than at termination
//     time.
//
// -*- No longer assumes that default location for user-files is
//     $(HOME)/Library, in all cases.  Now uses
//     NSSearchPathForDirectoriesInDomains() to resolve $(SokoUser) if
//     available.  It is available on MacOS/X (Cocoa) and MacOS/X Server 1.0
//     (Rhapsody).  On MacOS/X, "Application Support/SokoSave" is appended to
//     the result of NSSearchPathForDirectoriesInDomains().  In other cases,
//     only "SokoSave" is appended.  For Windows, $(SokoUser) expands to the
//     "personal" folder (typically "My Documents"), if it exists.  If not,
//     then it expands to the first of the following which exist: $(HOME),
//     $(TEMP), $(TMP), root of drive containing "windows" directory, or root
//     of current drive.
//
// -*- Added SokoWindow and SokoPanel, which are subclasses of NSWindow and
//     NSPanel, respectively.  All windows and panels throughout the
//     application are now instances of these classes.  This allows bugs in
//     the AppKit window and panel classes to be overcome more easily.  For
//     instance, the Windows port works around several Windows-specific
//     AppKit bugs by overriding certain methods in these classes.
//
// -*- Renamed "window" instance variable to "panel" for all classes and nibs
//     in which a panel is actually used (everything but SokoBoard, in fact).
//
// -*- Worked around an AppKit bug on Windows in which the Preferences panel
//     would suddenly appear on the "Window" menu when -setDocumentEdited:
//     was invoked even though panels are not supposed to appear on this
//     menu.  This problem was further compounded by the fact that the menu
//     item would remain dangling on the menu even after the Preferences
//     panel was closed.
//
//-----------------------------------------------------------------------------
#include "SokoPool.h"
struct IShellView; // Pacify compiler when including shlobj.h.
#include <ctype.h>
#include <shlobj.h>
#include <stdio.h>
#include <stdlib.h>

//-----------------------------------------------------------------------------
// soko_application_data_path
//	Return the base path into which application-specific files should be
//	stored for the current user.  The returned path does not include the
//	"SokoSave" suffix.  On Windows, tries the following locations in order;
//	returning the first which is valid:
//	    - The "personal" folder (i.e. "My Documents")
//	    - $(HOME)
//	    - $(TEMP)
//	    - $(TMP)
//	    - Root of drive where the "Windows" directory resides
//	    - Root directory of current drive
//-----------------------------------------------------------------------------
char const* soko_application_data_path( SokoPool pool )
    {
    char const* path = 0;
    LPITEMIDLIST pidl;
    LPMALLOC shell_malloc;
    char dir[ FILENAME_MAX ];
    if (SUCCEEDED(SHGetMalloc( &shell_malloc )))
	{
	if (SUCCEEDED(SHGetSpecialFolderLocation( 0, CSIDL_PERSONAL, &pidl )))
	    {
	    if (SHGetPathFromIDList(pidl, dir) && strlen(dir) != 0)
		path = SokoPool_store( pool, dir );
	    shell_malloc->lpVtbl->Free( shell_malloc, pidl );
	    }
	shell_malloc->lpVtbl->Release( shell_malloc );
	}
    if (path == 0)
	{
	char const* s = getenv( "HOME" );
	if (s == 0) s = getenv( "TEMP" );
	if (s == 0) s = getenv( "TMP"  );
	if (s != 0)
	    path = s;
	else if (GetWindowsDirectory( dir, sizeof(dir) ) != 0)
	    {
	    int const n = strlen( dir );
	    if (n >= 2 && isalpha(dir[0]) && dir[1] == ':') // Extract "C:".
		path = SokoPool_store_data_and_terminate( pool, dir, 2 );
	    else if (n >= 3 && dir[0]=='\\' && dir[1]=='\\' && dir[2]!='\\')
		{
		char const* p = dir + 2; // Extract "\\host\share".
		while (*p != '\\' && *p != '\0') // Scan "host".
		    p++;
		if (*p != '\0')
		    {
		    p++;
		    while (*p != '\\' && *p != '\0') // Scan "share".
			p++;
		    path = SokoPool_store_data_and_terminate(pool, dir, p-dir);
		    }
		}
	    }
	}
    return (path != 0 ? path : "/");
    }
