//-----------------------------------------------------------------------------
// SokoMatrixTriangle.m
//
//	A hexagonal-celled matrix for SokoSave.
//
// Copyright (c), 2002, Eric Sunshine <sunshine@sunshineco.com>
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoMatrixTriangle.m,v 1.1 2002/02/19 09:29:10 sunshine Exp $
// $Log: SokoMatrixTriangle.m,v $
// Revision 1.1  2002/02/19 09:29:10  sunshine
// v18
// -*- Added support for new triangular-style Trioban puzzles.
//
// -*- Consolidated all of the geometry-related utility code into the new
//     SokoGeometry facility.  This mechanism provides hit-testing and
//     coverage-testing facilities for square-, hexagonal-, and
//     triangular-tiled grids.  All high-level GUI code now utilizes these
//     facilities rather than relying upon their own copy/paste
//     implementations.
//
//-----------------------------------------------------------------------------
#import "SokoPool.h"
#import "SokoMatrixTriangle.h"
#import "SokoGeometry.h"
#import <AppKit/NSImage.h>
#import <Foundation/NSString.h>

#define CELL_WIDTH      (24)
#define CELL_HEIGHT     (22)
#define CELL_HALF_WIDTH (CELL_WIDTH / 2)

static NSImage* CELL_IMAGES[ SOKO_CELL_MAX * 2 ];
static NSString* const CELL_NAMES[ SOKO_CELL_MAX * 2 ] =
    {
    @"SokoEmptyTriS",		  // SOKO_CELL_EMPTY
    @"SokoEmptyTriN",		  // SOKO_CELL_EMPTY
    @"SokoEmptySafeTriS",	  // SOKO_CELL_EMPTY_SAFE
    @"SokoEmptySafeTriN",	  // SOKO_CELL_EMPTY_SAFE
    @"SokoWallTriS",		  // SOKO_CELL_WALL
    @"SokoWallTriN",		  // SOKO_CELL_WALL
    @"SokoPlayerTriS",		  // SOKO_CELL_PLAYER
    @"SokoPlayerTriN",		  // SOKO_CELL_PLAYER
    @"SokoPlayerSafeTriS",	  // SOKO_CELL_PLAYER_SAFE
    @"SokoPlayerSafeTriN",	  // SOKO_CELL_PLAYER_SAFE
    @"SokoCrateTriS",		  // SOKO_CELL_CRATE
    @"SokoCrateTriN",		  // SOKO_CELL_CRATE
    @"SokoCrateSafeTriS",	  // SOKO_CELL_CRATE_SAFE
    @"SokoCrateSafeTriN",	  // SOKO_CELL_CRATE_SAFE
    @"SokoCrateSelectedTriS",	  // SOKO_CELL_CRATE_SELECTED
    @"SokoCrateSelectedTriN",	  // SOKO_CELL_CRATE_SELECTED
    @"SokoCrateSafeSelectedTriS", // SOKO_CELL_CRATE_SELECTED_SAFE
    @"SokoCrateSafeSelectedTriN", // SOKO_CELL_CRATE_SELECTED_SAFE
    0,				  // SOKO_CELL_NULL
    0                             // SOKO_CELL_NULL
    };

@implementation SokoMatrixTriangle

//-----------------------------------------------------------------------------
// loadImages
//-----------------------------------------------------------------------------
+ (void)loadImages
    {
    static BOOL loaded = NO;
    if (!loaded)
	{
	int i;
	NSBundle* bundle = [NSBundle bundleForClass:self];
	for (i = 0; i < SOKO_CELL_MAX * 2; i++)
	    {
	    NSImage* image = 0;
	    NSString* name = CELL_NAMES[i];
	    NSString* path = [bundle pathForImageResource:name];
	    if (path != 0)
		image = [[NSImage alloc] initWithContentsOfFile:path];
	    CELL_IMAGES[i] = image;
	    }
	loaded = YES;
	}
    }


//-----------------------------------------------------------------------------
// imageForCellType:withRow:column:
//-----------------------------------------------------------------------------
+ (NSImage*)imageForCellType:(SokoCellType)type withRow:(int)r column:(int)c;
    {
    int const north_south = (r & 1) ^ (c & 1);
    [self loadImages];
    return CELL_IMAGES[ type * 2 + north_south ];
    }


//-----------------------------------------------------------------------------
// initWithRows:columns:
//-----------------------------------------------------------------------------
- (id)initWithRows:(int)r columns:(int)c
    {
    [super initWithRows:r columns:c];
    [self setFrameSize:NSMakeSize((c + 1) * CELL_HALF_WIDTH, r * CELL_HEIGHT)];
    return self;
    }


//-----------------------------------------------------------------------------
// drawRect:
//-----------------------------------------------------------------------------
- (void)drawRect:(NSRect)d
    {
    int r, c, rs, cs, nr, nc, rlim, clim;
    NSRect const b = [self bounds];
    soko_coverage_triangle( (int)NSMinX(d), (int)NSMinY(d), (int)NSWidth(d),
	(int)NSHeight(d), rows, cols, CELL_WIDTH, CELL_HEIGHT, CELL_HALF_WIDTH,
	NSMinX(b), NSMinY(b), &rs, &cs, &nr, &nc );
    for (r = rs, rlim = rs + nr; r < rlim; r++)
	for (c = cs, clim = cs + nc; c < clim; c++)
	    [self drawCellAtRow:r column:c];
    }


//-----------------------------------------------------------------------------
// drawCellAtRow:column:
//-----------------------------------------------------------------------------
- (void)drawCellAtRow:(int)r column:(int)c
    {
    NSImage* i = [self imageAtRow:r column:c];
    if (i != 0)
	{
	NSPoint p;
	p.x = c * CELL_HALF_WIDTH;
	p.y = r * CELL_HEIGHT + ([self isFlipped] ? CELL_HEIGHT : 0);
	[i compositeToPoint:p operation:NSCompositeSourceOver];
	}
    }


//-----------------------------------------------------------------------------
// getRow:column:forPoint:
//-----------------------------------------------------------------------------
- (BOOL)getRow:(int*)r column:(int*)c forPoint:(NSPoint)p
    {
    NSRect const b = [self bounds];
    return soko_hit_test_triangle( (int)p.x, (int)p.y, rows, cols,
	CELL_WIDTH, CELL_HEIGHT, CELL_HALF_WIDTH, NSMinX(b), NSMinY(b), r, c );
    }

@end
