//-----------------------------------------------------------------------------
// SokoMatrixHexagon.m
//
//	A hexagonal-celled matrix for SokoSave.
//
// Copyright (c), 2002, Eric Sunshine <sunshine@sunshineco.com>
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoMatrixHexagon.m,v 1.2 2002/02/19 09:28:53 sunshine Exp $
// $Log: SokoMatrixHexagon.m,v $
// Revision 1.2  2002/02/19 09:28:53  sunshine
// v18
// -*- Consolidated all of the geometry-related utility code into the new
//     SokoGeometry facility.  This mechanism provides hit-testing and
//     coverage-testing facilities for square-, hexagonal-, and
//     triangular-tiled grids.  All high-level GUI code now utilizes these
//     facilities rather than relying upon their own copy/paste
//     implementations.
//
// Revision 1.1  2002/01/29 20:27:29  sunshine
// v17
// -*- Added support for the new hexagonal-style puzzles.
//
// -*- Rewrote SokoMatrix.  It is now a generic base class for rendering
//     grids.  No longer based upon NSMatrix (which was only capable of
//     supporting rectangular cells).  Custom rendering is now done, rather
//     than relying upon NSMatrix.  Consequently, animation speed has
//     increased by a factor of five or more.
//
// -*- Added SokoMatrixSquare, a subclass of SokoMatrix, which knows how to
//     draw square-tiled puzzles, and perform appropriate hit-testing.
//
// -*- Added SokoMatrixHexagon, a subclass of SokoMatrix, which knows how to
//     draw hexagon-tiled puzzles, and perform appropriate hit-testing.
//-----------------------------------------------------------------------------
#import "SokoPool.h"
#import "SokoMatrixHexagon.h"
#import "SokoGeometry.h"
#import <AppKit/NSImage.h>
#import <Foundation/NSString.h>

#define CELL_WIDTH   (18)
#define CELL_HEIGHT  (22)
#define CELL_X_INSET (9)
#define CELL_Y_INSET (6)

static NSImage* CELL_IMAGES[ SOKO_CELL_MAX ];
static NSString* const CELL_NAMES[ SOKO_CELL_MAX ] =
    {
    @"SokoEmptyHex",		// SOKO_CELL_EMPTY
    @"SokoEmptySafeHex",	// SOKO_CELL_EMPTY_SAFE
    @"SokoWallHex",		// SOKO_CELL_WALL
    @"SokoPlayerHex",		// SOKO_CELL_PLAYER
    @"SokoPlayerSafeHex",	// SOKO_CELL_PLAYER_SAFE
    @"SokoCrateHex",		// SOKO_CELL_CRATE
    @"SokoCrateSafeHex",	// SOKO_CELL_CRATE_SAFE
    @"SokoCrateSelectedHex",	// SOKO_CELL_CRATE_SELECTED
    @"SokoCrateSafeSelectedHex",// SOKO_CELL_CRATE_SELECTED_SAFE
    0				// SOKO_CELL_NULL
    };

@implementation SokoMatrixHexagon

//-----------------------------------------------------------------------------
// loadImages
//-----------------------------------------------------------------------------
+ (void)loadImages
    {
    static BOOL loaded = NO;
    if (!loaded)
	{
	int i;
	NSBundle* bundle = [NSBundle bundleForClass:self];
	for (i = 0; i < SOKO_CELL_MAX; i++)
	    {
	    NSImage* image = 0;
	    NSString* name = CELL_NAMES[i];
	    NSString* path = [bundle pathForImageResource:name];
	    if (path != 0)
		image = [[NSImage alloc] initWithContentsOfFile:path];
	    CELL_IMAGES[i] = image;
	    }
	loaded = YES;
	}
    }


//-----------------------------------------------------------------------------
// imageForCellType:withRow:column:
//-----------------------------------------------------------------------------
+ (NSImage*)imageForCellType:(SokoCellType)type withRow:(int)r column:(int)c
    {
    [self loadImages];
    return CELL_IMAGES[ type ];
    }


//-----------------------------------------------------------------------------
// initWithRows:columns:
//-----------------------------------------------------------------------------
- (id)initWithRows:(int)r columns:(int)c
    {
    [super initWithRows:r columns:c];
    [self setFrameSize:NSMakeSize(
	(c * CELL_WIDTH) + (r <= 1 ? 0 : CELL_X_INSET),
	 r * (CELL_HEIGHT - CELL_Y_INSET) + CELL_Y_INSET)];
    return self;
    }


//-----------------------------------------------------------------------------
// drawRect:
//-----------------------------------------------------------------------------
- (void)drawRect:(NSRect)d
    {
    int r, c, rs, cs, nr, nc, rlim, clim;
    NSRect const b = [self bounds];
    soko_coverage_hexagon( (int)NSMinX(d), (int)NSMinY(d), (int)NSWidth(d),
	(int)NSHeight(d), rows, cols, CELL_WIDTH, CELL_HEIGHT, CELL_X_INSET,
	CELL_Y_INSET, NSMinX(b), NSMinY(b), &rs, &cs, &nr, &nc );
    for (r = rs, rlim = rs + nr; r < rlim; r++)
	for (c = cs, clim = cs + nc; c < clim; c++)
	    [self drawCellAtRow:r column:c];
    }


//-----------------------------------------------------------------------------
// drawCellAtRow:column:
//-----------------------------------------------------------------------------
- (void)drawCellAtRow:(int)r column:(int)c
    {
    NSImage* i = [self imageAtRow:r column:c];
    if (i != 0)
	{
	NSPoint p;
	p.x = c * CELL_WIDTH + ((r & 1) == 0 ? 0 : CELL_X_INSET);
	p.y = r * (CELL_HEIGHT - CELL_Y_INSET) +
	    ([self isFlipped] ? CELL_HEIGHT : 0);
	[i compositeToPoint:p operation:NSCompositeSourceOver];
	}
    }


//-----------------------------------------------------------------------------
// getRow:column:forPoint:
//-----------------------------------------------------------------------------
- (BOOL)getRow:(int*)r column:(int*)c forPoint:(NSPoint)p
    {
    NSRect const b = [self bounds];
    return soko_hit_test_hexagon( (int)p.x, (int)p.y, rows, cols, CELL_WIDTH,
	CELL_HEIGHT, CELL_X_INSET, CELL_Y_INSET, NSMinX(b), NSMinY(b), r, c );
    }

@end
