//-----------------------------------------------------------------------------
// SokoMatrix.m
//
//	A custom Matrix for SokoSave.
//
// Copyright (c), 1997,2001,2002, Eric Sunshine <sunshine@sunshineco.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoMatrix.m,v 1.6 2002/02/19 09:28:47 sunshine Exp $
// $Log: SokoMatrix.m,v $
// Revision 1.6  2002/02/19 09:28:47  sunshine
// v18
// -*- Consolidated all of the geometry-related utility code into the new
//     SokoGeometry facility.  This mechanism provides hit-testing and
//     coverage-testing facilities for square-, hexagonal-, and
//     triangular-tiled grids.  All high-level GUI code now utilizes these
//     facilities rather than relying upon their own copy/paste
//     implementations.
//
// Revision 1.5  2002/01/29 20:27:39  sunshine
// v17
// -*- Added support for the new hexagonal-style puzzles.
//
// -*- Rewrote SokoMatrix.  It is now a generic base class for rendering
//     grids.  No longer based upon NSMatrix (which was only capable of
//     supporting rectangular cells).  Custom rendering is now done, rather
//     than relying upon NSMatrix.  Consequently, animation speed has
//     increased by a factor of five or more.
//
// -*- Added SokoMatrixSquare, a subclass of SokoMatrix, which knows how to
//     draw square-tiled puzzles, and perform appropriate hit-testing.
//
// -*- Added SokoMatrixHexagon, a subclass of SokoMatrix, which knows how to
//     draw hexagon-tiled puzzles, and perform appropriate hit-testing.
//
// Revision 1.4  2001/12/23 16:08:10  sunshine
// v15
// -*- Extracted core game logic out of GUI code and generalized it so that
//     the same core code can be used by any platform.  Logic from
//     SokoBoard.m now resides in SokoPuzzle.c, etc.
//
// -*- Rewrote the OpenStep port so that it utilizes the new "sokocore" game
//     library rather than implementing that logic directly.
//-----------------------------------------------------------------------------
#import "SokoPool.h"
#import "SokoMatrix.h"
#import "SokoAssert.h"
#import <AppKit/NSImage.h>
#import <AppKit/NSWindow.h>
#import <Foundation/NSException.h>
#import <Foundation/NSString.h>
#import <stdlib.h>

@implementation SokoMatrix

//-----------------------------------------------------------------------------
// initWithRows:columns:
//-----------------------------------------------------------------------------
- (id)initWithRows:(int)r columns:(int)c
    {
    int const sz = r * c * sizeof(matrix[0]);
    [super initWithFrame:NSZeroRect];
    rows = r;
    cols = c;
    matrix = (NSImage**)malloc(sz);
    memset( matrix, 0, sz );
    return self;
    }


//-----------------------------------------------------------------------------
// dealloc
//-----------------------------------------------------------------------------
- (void)dealloc
    {
    int i;
    for (i = rows * cols - 1; i >= 0; i--)
	if (matrix[i] != 0)
	    [matrix[i] release];
    free( matrix );
    [super dealloc];
    }


//-----------------------------------------------------------------------------
// setCellType:atRow:column:
//-----------------------------------------------------------------------------
- (void)setCellType:(SokoCellType)type atRow:(int)r column:(int)c
    {
    int n;
    SOKO_ASSERT( r >= 0 && r < rows );
    SOKO_ASSERT( c >= 0 && c < cols );
    n = r * cols + c;
    [matrix[n] release];
    matrix[n] = [[[self class] imageForCellType:type withRow:r column:c]
	retain];
    if (![self canDraw])
	[self setNeedsDisplay:YES];
    else
	{
	[self lockFocus];
	[self drawCellAtRow:r column:c];
	[self unlockFocus];
	[[self window] flushWindow];
	}
    }


//-----------------------------------------------------------------------------
// imageAtRow:column:
//-----------------------------------------------------------------------------
- (NSImage*)imageAtRow:(int)r column:(int)c
    {
    SOKO_ASSERT( r >= 0 && r < rows );
    SOKO_ASSERT( c >= 0 && c < cols );
    return *(matrix + r * cols + c);
    }


//-----------------------------------------------------------------------------
// Stubs
//-----------------------------------------------------------------------------
- (BOOL)acceptsFirstResponder         { return YES; }
- (BOOL)acceptsFirstMouse:(NSEvent*)e { return YES; }
- (BOOL)isFlipped                     { return YES; }

- (void)setDelegate:(id)d { delegate = d; }
- (id)delegate { return delegate; }

+ (void)subclassErorr:(SEL)cmd
    {
    [NSException raise:NSInvalidArgumentException
	format:@"*** Subclass responsibility: %s", NSStringFromSelector(cmd)];
    }
+ (NSImage*)imageForCellType:(SokoCellType)type withRow:(int)r column:(int)c
    { [self subclassErorr:_cmd]; return 0; }
- (void)drawCellAtRow:(int)r column:(int)c
    { [[self class] subclassErorr:_cmd]; }
- (BOOL)getRow:(int*)r column:(int*)c forPoint:(NSPoint)p
    { [[self class] subclassErorr:_cmd]; return NO; }

#define FWD(M) - (void)M:(NSEvent*)e { [delegate M:e]; }

FWD(mouseDown)
FWD(mouseUp)
FWD(mouseDragged)
FWD(rightMouseDown)
FWD(rightMouseUp)
FWD(rightMouseDragged)
FWD(keyDown)
FWD(keyUp)

#undef FWD

@end
