//-----------------------------------------------------------------------------
// SokoBoard.m
//
//	Implementation of a Sokoban puzzle board for SokoSave.
//
// Copyright (c), 1997,2001,2002, Eric Sunshine <sunshine@sunshineco.com>
// Copyright (c), 1997, Paul McCarthy <zarnuk@high-speed-software.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoBoard.m,v 1.9 2002/02/19 09:27:21 sunshine Exp $
// $Log: SokoBoard.m,v $
// Revision 1.9  2002/02/19 09:27:21  sunshine
// v18
// -*- Added support for new triangular-style Trioban puzzles.
//
// -*- Consolidated the user-interface movement constraints into SokoPuzzle
//     rather than having the same code repeated in each port.  This code
//     decides whether or not the direction movement buttons in the
//     user-interface should be enabled depending upon cell type,
//     orientation, etc.  Added the SokoPuzzle methods can_move_constrained()
//     and can_move_diagonal_constrained() to complement the existing
//     can_move() and can_move_diagonal().
//
// Revision 1.8  2002/01/29 20:35:37  sunshine
// v17
// -*- Added support for the new hexagonal-style puzzles.
//
// -*- No longer fills areas outside the puzzle with wall segments.  Doing so
//     causes some "artistic" effect to be lost from some puzzles.  Added
//     SokoCellType SOKO_CELL_NULL to support of this.
//
// -*- Added SokoPuzzleStyle enumeration to SokoPuzzle.  Styles are
//     SOKO_STYLE_SQUARE and SOKO_STYLE_HEXAGON.
//
// -*- Now recognize these additional file extensions: .xsb, .hsb, .sokohex.
//
// -*- Removed soko_get_puzzle_extension() and soko_get_save_extension() from
//     SokoFile.
//
// -*- Relocated functionality of soko_save_filename_for_puzzle(),
//     soko_puzzle_name_for_level(), and soko_level_for_puzzle_name() from
//     SokoFile to SokoPuzzle, where they are class methods of SokoPuzzle.
//
// -*- Added get_puzzle_extensions(), get_puzzle_extensions_count(), and
//     get_save_extension() as class methods to SokoPuzzle.  There are now
//     multiple recognized puzzle extensions, rather than just the one
//     (.sokomaze).
//
// -*- Added file_is_puzzle(), file_is_save_game(), and get_file_style()
//     class methods to SokoPuzzle.
//
// -*- Extended keyboard movement support.  Numeric keypad can now be used,
//     as well as standard arrow keys.  Added key equivalents for hexagonal
//     puzzles.
//
// -*- SokoPuzzle now calculates two new scores in addition to "moves" and
//     "pushes".  The "runs" score is the number of straight lines in which
//     crates have been pushed.  Looked at another way, it is the number of
//     turns crates have made while being pushed.  The "focus" score is the
//     number of times the player's focus has changed from one crate to
//     another.
//
// -*- Added "runs" and "focus" scores controls to SokoBoard.
//
// -*- Rewrote SokoMatrix.  It is now a generic base class for rendering
//     grids.  No longer based upon NSMatrix (which was only capable of
//     supporting rectangular cells).  Custom rendering is now done, rather
//     than relying upon NSMatrix.  Consequently, animation speed has
//     increased by a factor of five or more.
//
// -*- Added SokoMatrixSquare, a subclass of SokoMatrix, which knows how to
//     draw square-tiled puzzles, and perform appropriate hit-testing.
//
// -*- Added SokoMatrixHexagon, a subclass of SokoMatrix, which knows how to
//     draw hexagon-tiled puzzles, and perform appropriate hit-testing.
//-----------------------------------------------------------------------------
#import "SokoPool.h"
#import "SokoBoard.h"
#import "SokoAlerter.h"
#import "SokoApp.h"
#import "SokoAssert.h"
#import "SokoFile.h"
#import "SokoMatrixHexagon.h"
#import "SokoMatrixSquare.h"
#import "SokoMatrixTriangle.h"
#import "SokoNewScore.h"
#import "SokoPref.h"
#import "SokoSetting.h"
#import "SokoUtil.h"
#import "SokoWindow.h"
#import <AppKit/NSApplication.h>
#import <AppKit/NSBox.h>
#import <AppKit/NSButton.h>
#import <AppKit/NSButtonCell.h>
#import <AppKit/NSEvent.h>
#import <AppKit/NSFont.h>
#import <AppKit/NSImage.h>
#import <AppKit/NSNibLoading.h>
#import <AppKit/NSOpenPanel.h>
#import <AppKit/NSSavePanel.h>
#import <AppKit/NSScreen.h>
#import <AppKit/NSSlider.h>
#import <AppKit/NSTextField.h>
#import <Foundation/NSArray.h>
#import <Foundation/NSAutoreleasePool.h>
#import <Foundation/NSDate.h>
#import <Foundation/NSDictionary.h>
#import <Foundation/NSEnumerator.h>
#import <Foundation/NSNotification.h>
#import <Foundation/NSNotificationQueue.h>
#import <Foundation/NSPathUtilities.h>
#import <Foundation/NSValue.h>
#import <float.h>	// FLT_MAX

#define SOKO_ADVANCE_PLAYBACK_NOTIFICATION @"SokoAdvancePlaybackNotification"
#define SOKO_RECORD_SCORE_NOTIFICATION     @"SokoRecordScoreNotification"

//-----------------------------------------------------------------------------
// SokoBoardCell
//	Prevent MacOS/X from drawing focus rectangle on the top-left cell.
//	-[SokoBoard createCell] attempts to disable the focus rectangle by
//	setting the appropriate attributes of the prototype cell, but OSX seems
//	to ignore them.
//-----------------------------------------------------------------------------
@interface SokoBoardCell : NSButtonCell
- (BOOL)acceptsFirstResponder;
@end
@implementation SokoBoardCell
- (BOOL)acceptsFirstResponder { return NO; }
@end


//-----------------------------------------------------------------------------
// SokoPuzzleDelegate Callback Functions
//-----------------------------------------------------------------------------
#define DEL_BOARD ((SokoBoard*)(x1->info))
#define DEL_FUNC_0(RET,NAME) \
    static RET NAME(SokoPuzzle x1, SokoPuzzleDelegate x2)
#define DEL_FUNC_1(RET,NAME,T1,A1) \
    static RET NAME(SokoPuzzle x1, SokoPuzzleDelegate x2, T1 A1)
#define DEL_FUNC_2(RET,NAME,T1,A1,T2,A2) \
    static RET NAME(SokoPuzzle x1, SokoPuzzleDelegate x2, T1 A1, T2 A2)
#define DEL_FUNC_3(RET,NAME,T1,A1,T2,A2,T3,A3) \
    static RET NAME(SokoPuzzle x1, SokoPuzzleDelegate x2, T1 A1, T2 A2, T3 A3)
#define DEL_FUNC_4(RET,NAME,T1,A1,T2,A2,T3,A3,T4,A4) \
    static RET NAME(SokoPuzzle x1, SokoPuzzleDelegate x2, \
	T1 A1, T2 A2, T3 A3, T4 A4)

DEL_FUNC_3(void, puzzle_alert, SokoAlert,s, char const*,t, char const*,m)
    { SokoSendAlert( s, t, m ); }
DEL_FUNC_4(void, record_score, int,m, int,p, int,r, int,f)
    { [DEL_BOARD recordMoves:m pushes:p runs:r focus:f]; }
DEL_FUNC_1(void, record_level, int,level)
    { [DEL_BOARD recordLevel:level]; }
DEL_FUNC_1(void, set_solved, soko_bool,solved)
    { [DEL_BOARD setSolved:solved]; }
DEL_FUNC_1(void, set_dirty, soko_bool,dirty)
    { [DEL_BOARD setDirty:dirty]; }
DEL_FUNC_0(void, refresh_controls)
    { [DEL_BOARD refreshControls]; }
DEL_FUNC_0(void, refresh_all_cells)
    { [DEL_BOARD refreshAllCells]; }
DEL_FUNC_2(void, refresh_cells, SokoCell const*,cells, int,ncells)
    { [DEL_BOARD refreshCells:cells count:ncells]; }
DEL_FUNC_0(soko_bool, should_animate_playback)
    { return [DEL_BOARD shouldAnimatePlayback]; }
DEL_FUNC_0(int, playback_threshold)
    { return [DEL_BOARD playbackThreshold]; }
DEL_FUNC_0(void, begin_playback)
    { [DEL_BOARD beginPlayback]; }
DEL_FUNC_0(void, end_playback)
    { [DEL_BOARD endPlayback]; }
DEL_FUNC_0(void, begin_animation)
    { [DEL_BOARD beginAnimation]; }
DEL_FUNC_0(void, end_animation)
    { [DEL_BOARD endAnimation]; }
DEL_FUNC_0(void, begin_drag)
    { [DEL_BOARD beginDrag]; }
DEL_FUNC_0(void, end_drag)
    { [DEL_BOARD endDrag]; }
DEL_FUNC_1(void, process_events, soko_bool,await_event)
    { [DEL_BOARD processEvents:await_event]; }

#undef DEL_FUNC_2
#undef DEL_FUNC_1
#undef DEL_FUNC_0
#undef DEL_BOARD


//=============================================================================
// IMPLEMENTATION
//=============================================================================
@implementation SokoBoard

//-----------------------------------------------------------------------------
// validateMenuItem:
//-----------------------------------------------------------------------------
- (BOOL)validateMenuItem:(id)item
    {
    BOOL valid = YES;
    SEL const a = [item action];
    if (a == @selector(openNextPuzzle:))
	valid = nextPuzzleExists;
    else
	{
	int const pos = SokoPuzzle_history_position( puzzle );
	int const len = SokoPuzzle_history_length( puzzle );
	BOOL const active = SokoPuzzle_playback_active( puzzle );
	if (a == @selector(undoMove:) || a == @selector(undoPush:))
	    valid = !active && (pos > 0);
	else if (a == @selector(redoMove:) || a == @selector(redoPush:))
	    valid = !active && (pos < len);
	}
    return valid;
    }


//-----------------------------------------------------------------------------
// performPrint:
//-----------------------------------------------------------------------------
- (void)performPrint:(id)sender
    {
    [window print:self];
    }


//=============================================================================
// SokoPuzzle Delegate Support
//=============================================================================
//-----------------------------------------------------------------------------
// puzzleDelegate
//-----------------------------------------------------------------------------
+ (SokoPuzzleDelegate)puzzleDelegate
    {
    static struct _SokoPuzzleDelegate delegate;
    static BOOL initialized = NO;
    if (!initialized)
	{
	delegate.alert_callback = puzzle_alert;
	delegate.record_score_callback = record_score;
	delegate.record_level_callback = record_level;
	delegate.set_solved_callback = set_solved;
	delegate.set_dirty_callback = set_dirty;
	delegate.refresh_controls_callback = refresh_controls;
	delegate.refresh_all_cells_callback = refresh_all_cells;
	delegate.refresh_cells_callback = refresh_cells;
	delegate.should_animate_playback_callback = should_animate_playback;
	delegate.playback_threshold_callback = playback_threshold;
	delegate.begin_playback_callback = begin_playback;
	delegate.end_playback_callback = end_playback;
	delegate.begin_animation_callback = begin_animation;
	delegate.end_animation_callback = end_animation;
	delegate.begin_drag_callback = begin_drag;
	delegate.end_drag_callback = end_drag;
	delegate.process_events_callback = process_events;
	delegate.info = 0;
	initialized = YES;
	}
    return &delegate;
    }


//-----------------------------------------------------------------------------
// recordLevel:
//-----------------------------------------------------------------------------
- (void)recordLevel:(int)level
    {
    if (level >= [SokoPref getNextLevel])
	[SokoPref setNextLevel:level + 1];
    }


//-----------------------------------------------------------------------------
// recordScore:
//-----------------------------------------------------------------------------
- (void)recordScore:(NSNotification*)n
    {
    NSDictionary* info = [n userInfo];
    [[NSNotificationCenter defaultCenter] removeObserver:self
	name:SOKO_RECORD_SCORE_NOTIFICATION object:self];
    [SokoNewScore solved:puzzle
	moves:[[info objectForKey:@"moves"] intValue]
	pushes:[[info objectForKey:@"pushes"] intValue]
	runs:[[info objectForKey:@"runs"] intValue]
	focus:[[info objectForKey:@"focus"] intValue]];
    }


//-----------------------------------------------------------------------------
// recordMoves:pushes:runs:focus:
//	Delay actual recording of score until application is idle, since this
//	method might otherwise be called during a mouse down, a drag session,
//	a key down, etc, and it would be unfriendly to launch the modal "New
//	Score" panel before those activities ceased.
//-----------------------------------------------------------------------------
- (void)recordMoves:(int)m pushes:(int)p runs:(int)r focus:(int)f
    {
    [[NSNotificationCenter defaultCenter] addObserver:self
	selector:@selector(recordScore:) name:SOKO_RECORD_SCORE_NOTIFICATION
	object:self];
    [[NSNotificationQueue defaultQueue] enqueueNotification:
	[NSNotification notificationWithName:SOKO_RECORD_SCORE_NOTIFICATION
	object:self userInfo:[NSDictionary dictionaryWithObjectsAndKeys:
	[NSNumber numberWithInt:m], @"moves",
	[NSNumber numberWithInt:p], @"pushes",
	[NSNumber numberWithInt:r], @"runs",
	[NSNumber numberWithInt:f], @"focus", 0]]
	postingStyle:NSPostWhenIdle];
    }


//-----------------------------------------------------------------------------
// setSolved:
//-----------------------------------------------------------------------------
- (void)setSolved:(BOOL)solved
    {
    [solvedFld setStringValue:(solved ? @"SOLVED" : @"")];
    }


//-----------------------------------------------------------------------------
// setDirty:
//-----------------------------------------------------------------------------
- (void)setDirty:(BOOL)dirty
    {
    [window setDocumentEdited:dirty];
    }


//-----------------------------------------------------------------------------
// refreshControls
//	Set enabled status of various controls.  Also ensure that the cell
//	matrix is always first-responder so that keyboard movement controls
//	work even if some other control on the window stole first-responder
//	status.
//-----------------------------------------------------------------------------
- (void)refreshControls
    {
    int  const hpos = SokoPuzzle_history_position( puzzle );
    int  const hlen = SokoPuzzle_history_length( puzzle );
    BOOL const undo = (hpos > 0);
    BOOL const redo = (hpos < hlen);

#define ENABLE_DIR(B,D) [move##B##Btn \
    setEnabled:SokoPuzzle_can_move_constrained(puzzle,SOKO_DIR_##D)]
#define ENABLE_DIAG(B,D) [move##B##Btn \
    setEnabled:SokoPuzzle_can_move_diagonal_constrained(puzzle,SOKO_DIAG_##D)]
    ENABLE_DIR(Up,UP);
    ENABLE_DIR(Left,LEFT);
    ENABLE_DIR(Right,RIGHT);
    ENABLE_DIR(Down,DOWN);
    ENABLE_DIAG(UpLeft,UP_LEFT);
    ENABLE_DIAG(UpRight,UP_RIGHT);
    ENABLE_DIAG(DownLeft,DOWN_LEFT);
    ENABLE_DIAG(DownRight,DOWN_RIGHT);
#undef ENABLE_DIAG
#undef ENABLE_DIR

    [undoMoveBtn setEnabled:undo];
    [redoMoveBtn setEnabled:redo];
    [undoPushBtn setEnabled:undo];
    [redoPushBtn setEnabled:redo];
    [playbackSlider setMaxValue:hlen];
    [playbackSlider setIntValue:hpos];
    [playbackSlider setEnabled:(hlen > 0)];

    [numMovesFld  setIntValue:SokoPuzzle_move_count ( puzzle )];
    [numPushesFld setIntValue:SokoPuzzle_push_count ( puzzle )];
    [numRunsFld   setIntValue:SokoPuzzle_run_count  ( puzzle )];
    [numFocusFld  setIntValue:SokoPuzzle_focus_count( puzzle )];

    if ([window firstResponder] != cellMatrix)
	[window makeFirstResponder:cellMatrix];
    }


//-----------------------------------------------------------------------------
// refreshAllCells
//-----------------------------------------------------------------------------
- (void)refreshAllCells
    {
    int r, c;
    int nr = SokoPuzzle_row_count(puzzle), nc = SokoPuzzle_col_count(puzzle);
    [window disableFlushWindow];
    for (r = 0; r < nr; r++)
	for (c = 0; c < nc; c++)
	    [cellMatrix setCellType:SokoPuzzle_cell_type(puzzle, r, c)
		atRow:r column:c];
    [window enableFlushWindow];
    }


//-----------------------------------------------------------------------------
// refreshCells:count:
//-----------------------------------------------------------------------------
- (void)refreshCells:(SokoCell const*)cells count:(int)ncells
    {
    int i;
    [window disableFlushWindow];
    for (i = 0; i < ncells; i++)
	[cellMatrix setCellType:cells[i].type
	    atRow:cells[i].row column:cells[i].col];
    [window enableFlushWindow];
    [window flushWindowIfNeeded];
    }


//=============================================================================
// Board Creation & Destruction Support
//=============================================================================
//-----------------------------------------------------------------------------
// cascadeWindow:
//-----------------------------------------------------------------------------
+ (void)cascadeWindow:(NSWindow*)w
    {
    static BOOL initialized = NO;
    static float baseTop = 0;
    static float baseLeft = 0;
    static int cascadeCounter = -1;
    float top,left;

    if (!initialized)
	{
	NSSize sz = [[NSScreen mainScreen] frame].size;
	baseLeft = floor(sz.width / 10);
	baseTop = sz.height - floor(sz.height / 10);
	initialized = YES;
	}

    if (++cascadeCounter == 10)
	cascadeCounter = 0;

    top  = baseTop  - (cascadeCounter * 20);
    left = baseLeft + (cascadeCounter * 20);
    [w setFrameTopLeftPoint:NSMakePoint(left, top)];
    }


//-----------------------------------------------------------------------------
// adjustWindowSize
//	Resize window to contain puzzle matrix.  However, consider designed
//	control box width as minimum width limit, so do not shrink window
//	horizontally beyond that limit.  If puzzle matrix is smaller than
//	minimum limit, then center matrix in window.
// *MIN/MAX*
//	There is a bug in the AppKit on Windows 9x where the window size of
//	non-resizeable windows becomes corrupted when miniaturized.  Rather
//	than deminiaturizing to normal size, the window deminiaturizes only to
//	the height of its title bar, and to a width just large enough to
//	contain the title bar buttons and a small section of the title.
//	Sizeable windows do not suffer from this problem.  Therefore, to work
//	around this problem, on Windows 9x, SokoWindow automatically overrides
//	the non-resizeable flag for such windows and makes them sizeable.  To
//	compensate for the fact that windows which should not have been
//	sizeable are now sizeable, we must set the minimum and maximum extents
//	of these windows to ensure that the user can not actually resize them.
//	This problem does not affect Windows NT/2000 or Mach.
//-----------------------------------------------------------------------------
- (void)adjustWindowSize
    {
    NSSize windowSize;
    NSRect boxFrame;
    NSRect matrixFrame;

    boxFrame = [controlBox frame];
    matrixFrame = [cellMatrix frame];
    windowSize.width = minControlBoxWidth;

    matrixFrame.origin.y = boxFrame.size.height + 4;
    if (matrixFrame.size.width + 8 > windowSize.width)
	{
	matrixFrame.origin.x = 4;
	windowSize.width = matrixFrame.size.width + 8;
	}
    else
	{
	matrixFrame.origin.x =
	    floor((windowSize.width - matrixFrame.size.width) / 2);
	}

    windowSize.height = matrixFrame.size.height + boxFrame.size.height + 8;

    [window setContentSize:windowSize];
    [window setMinSize:windowSize];		// *MIN/MAX*
    [window setMaxSize:windowSize];

    [controlBox setFrameSize:
	NSMakeSize(windowSize.width, boxFrame.size.height)];
    [controlBox setFrameOrigin:NSMakePoint(0, 0)];
    [cellMatrix setFrameOrigin:matrixFrame.origin];
    }


//-----------------------------------------------------------------------------
// windowWillMiniaturize:
//	See the *MIN/MAX* discussion in -adjustWindowSize.  However, the
//	minimum and maximum sizes must be temporarily disabled during
//	miniaturization.  If this is not done, then the window size also
//	becomes corrupted at deminiaturization time.
//-----------------------------------------------------------------------------
- (void)windowWillMiniaturize:(NSNotification*)n
    {
    [window setMinSize:NSMakeSize(0,0)];
    [window setMaxSize:NSMakeSize(FLT_MAX,FLT_MAX)];
    }


//-----------------------------------------------------------------------------
// windowDidDeminiaturize:
//	See the *MIN/MAX* discussion in -adjustWindowSize.  However, the
//	minimum and maximum sizes must be temporarily disabled during
//	miniaturization.  If this is not done, then the window size also
//	becomes corrupted at deminiaturization time.
//-----------------------------------------------------------------------------
- (void)windowDidDeminiaturize:(NSNotification*)n
    {
    NSRect boxFrame = [controlBox frame];
    NSRect matrixFrame = [cellMatrix frame];
    NSSize windowSize;
    windowSize.width = minControlBoxWidth;
    if (matrixFrame.size.width >= windowSize.width)
	windowSize.width = matrixFrame.size.width;
    windowSize.width += 8;
    windowSize.height = matrixFrame.size.height + boxFrame.size.height + 12;
    [window setMinSize:windowSize];
    [window setMaxSize:windowSize];
    }


//-----------------------------------------------------------------------------
// windowMetricsChanged:
//	This method is called whenever the user changes settings which affect
//	the window metrics, such as menu font, menu height, window border
//	thickness, etc.  On Microsoft Windows, we need to respond to the such
//	changes to ensure that the window's client area is still large enough
//	to display the entire puzzle since the AppKit does not adjust the
//	client size for us.  For instance, if the user increases the menu
//	height, or if a new menu font size causes the menu to wrap, then the
//	menu might begin to occlude the top of the puzzle.
//-----------------------------------------------------------------------------
- (void)windowMetricsChanged:(NSNotification*)n
    {
    [self adjustWindowSize];
    }


//-----------------------------------------------------------------------------
// captureWindowConstraints
//-----------------------------------------------------------------------------
- (void)captureWindowConstraints
    {
    minControlBoxWidth = [controlBox frame].size.width;
    }


//-----------------------------------------------------------------------------
// createMatrix
//-----------------------------------------------------------------------------
- (void)createMatrix
    {
    int const numRows = SokoPuzzle_row_count( puzzle );
    int const numCols = SokoPuzzle_col_count( puzzle );
    SokoPuzzleStyle const style = SokoPuzzle_puzzle_style( puzzle );
    id matrixClass = 0;

    if (style == SOKO_STYLE_SQUARE)
	matrixClass = [SokoMatrixSquare class];
    else if (style == SOKO_STYLE_HEXAGON)
	matrixClass = [SokoMatrixHexagon class];
    else if (style == SOKO_STYLE_TRIANGLE)
	matrixClass = [SokoMatrixTriangle class];
    SOKO_ASSERT( matrixClass != 0 );
	
    cellMatrix = [[matrixClass allocWithZone:[self zone]]
	initWithRows:numRows columns:numCols];
    [cellMatrix setDelegate:self];
    [[window contentView] addSubview:cellMatrix];
    [window makeFirstResponder:cellMatrix];
    [cellMatrix setNextKeyView:moveUpLeftBtn];
    [stopBtn setNextKeyView:cellMatrix];
    }


//-----------------------------------------------------------------------------
// configureControls
//-----------------------------------------------------------------------------
- (void)configureControls
    {
#define BUTTON_TAG(X,Y) [move##X##Btn setTag:SOKO_##Y];DISABLE_FR(move##X##Btn)
#define DISABLE_FR(X) \
    [X setRefusesFirstResponder:YES]; [[X cell] setShowsFirstResponder:NO]
    BUTTON_TAG(Up,DIR_UP);
    BUTTON_TAG(Down,DIR_DOWN);
    BUTTON_TAG(Left,DIR_LEFT);
    BUTTON_TAG(Right,DIR_RIGHT);
    BUTTON_TAG(UpLeft,DIAG_UP_LEFT);
    BUTTON_TAG(UpRight,DIAG_UP_RIGHT);
    BUTTON_TAG(DownLeft,DIAG_DOWN_LEFT);
    BUTTON_TAG(DownRight,DIAG_DOWN_RIGHT);
    DISABLE_FR(undoMoveBtn);
    DISABLE_FR(redoMoveBtn);
    DISABLE_FR(undoPushBtn);
    DISABLE_FR(redoPushBtn);
    DISABLE_FR(playbackSlider);
    DISABLE_FR(animateSwitch);
    DISABLE_FR(stopBtn);
#undef BUTTON_TAG
#undef DISABLE_FR

#if defined(SOKO_UI_AQUA)
#define SQUARE(X) [[X cell] setBezelStyle:NSShadowlessSquareBezelStyle]
#define SMALL(X) [[X cell] setControlSize:NSSmallControlSize]; \
    [X setFont:[NSFont systemFontOfSize:[NSFont smallSystemFontSize]]]
    SMALL(animateSwitch);
    SMALL(playbackSlider);
    SMALL(stopBtn);
    SQUARE(moveDownBtn);
    SQUARE(moveDownLeftBtn);
    SQUARE(moveDownRightBtn);
    SQUARE(moveLeftBtn);
    SQUARE(moveRightBtn);
    SQUARE(moveUpBtn);
    SQUARE(moveUpLeftBtn);
    SQUARE(moveUpRightBtn);
    SQUARE(redoMoveBtn);
    SQUARE(redoPushBtn);
    SQUARE(undoMoveBtn);
    SQUARE(undoPushBtn);
#undef SMALL
#undef SQUARE
#endif
    }


//-----------------------------------------------------------------------------
// init
//-----------------------------------------------------------------------------
- (id)init
    {
    [super init];
    [NSBundle loadNibNamed:@"SokoBoard" owner:self];
    [window setMiniwindowImage:[NSImage imageNamed:@"SokoSave"]];
    [self configureControls];
    puzzle = 0;
    nextPuzzleExists = NO;
    return self;
    }


//-----------------------------------------------------------------------------
// dealloc
//-----------------------------------------------------------------------------
- (void)dealloc
    {
    [[NSNotificationCenter defaultCenter] removeObserver:self];
    if (puzzle != 0)
	SokoPuzzle_destroy( puzzle );
    [cellMatrix release];
    [window setDelegate:0];
    [window close];
    [window release];
    [super dealloc];
    }


//-----------------------------------------------------------------------------
// refreshWindowTitle
//-----------------------------------------------------------------------------
- (void)refreshWindowTitle
    {
    SokoPool pool = SokoPool_new(0);
    char const* path = SokoPuzzle_get_save_file_name( puzzle, pool );
    [window setTitleWithRepresentedFilename:[NSString stringWithCString:path]];
    SokoPool_destroy( pool );
    }


//-----------------------------------------------------------------------------
// checkForNextPuzzle
//-----------------------------------------------------------------------------
- (void)checkForNextPuzzle
    {
    SokoPool pool = SokoPool_new(0);
    int const level = SokoPuzzle_level_for_puzzle_name(
	SokoPuzzle_get_puzzle_file_name(puzzle, pool) );
    nextPuzzleExists =
	soko_path_exists(pool,SokoPuzzle_puzzle_name_for_level(level+1, pool));
    SokoPool_destroy( pool );
    }


//-----------------------------------------------------------------------------
// configureBoard:
//-----------------------------------------------------------------------------
- (void)configureBoard:(SokoPuzzle)p
    {
    puzzle = p;
    [self createMatrix];
    [self captureWindowConstraints];
    [self adjustWindowSize];
    [self setDirty:SokoPuzzle_puzzle_dirty( puzzle )];
    [self setSolved:SokoPuzzle_puzzle_solved( puzzle )];
    [self checkForNextPuzzle];
    [self refreshControls];
    [self refreshAllCells];
    [self refreshWindowTitle];
    [stopBtn setEnabled:NO];
    [[self class] cascadeWindow:window];
    [[NSNotificationCenter defaultCenter] addObserver:self
	selector:@selector(windowMetricsChanged:)
	name:SokoWindowMetricsChangedNotification object:window];
    }


//-----------------------------------------------------------------------------
// activate
//-----------------------------------------------------------------------------
- (void)activate
    {
    [window makeKeyAndOrderFront:self];
    [window finalize];
    }


//-----------------------------------------------------------------------------
// windowShouldClose:
//-----------------------------------------------------------------------------
- (BOOL)windowShouldClose:(id)sender
    {
    SokoPuzzle_abort_playback( puzzle );
    [self autorelease];
    return YES;
    }


//-----------------------------------------------------------------------------
// openPuzzle:
//-----------------------------------------------------------------------------
+ (BOOL)openPuzzle:(NSString*)path
    {
    SokoBoard* board;
    char const* cpath = [path fileSystemRepresentation];
    SokoPuzzle p = SokoPuzzle_find_open_puzzle( cpath );
    if (p != 0)
	board = (SokoBoard*)p->info;
    else
	{
	board = [[SokoBoard alloc] init];
	p = SokoPuzzle_new( [[self class] puzzleDelegate], cpath, board );
	if (p != 0)
	    [board configureBoard:p];
	else
	    {
	    [board release];
	    board = 0;
	    }
	}
    if (board != 0)
	[board activate];
    return (board != 0);
    }


//-----------------------------------------------------------------------------
// openPuzzles:
//-----------------------------------------------------------------------------
+ (BOOL)openPuzzles:(NSArray*)paths
    {
    BOOL ok = NO;
    NSString* path;
    NSEnumerator* enumerator = [paths objectEnumerator];
    while ((path = [enumerator nextObject]) != 0)
	if ([self openPuzzle:path])
	    ok = YES;
    return ok;
    }


//-----------------------------------------------------------------------------
// getOpenTypesWithSeed:
//	Return a list of all puzzle extensions.  Also include an additional
//	extension (seed) if provided.
//-----------------------------------------------------------------------------
+ (NSArray*)getOpenTypesWithSeed:(char const*)seed
    {
    NSMutableArray* types = [NSMutableArray array];
    SokoPuzzleExtension const* e = SokoPuzzle_get_puzzle_extensions();
    for ( ; e->extension != 0; e++ )
	[types addObject:[NSString stringWithCString:e->extension]];
    if (seed != 0)
	[types addObject:[NSString stringWithCString:seed]];
    return types;
    }


//-----------------------------------------------------------------------------
// newPuzzle
//	Only seed the dialog with the name of "next" puzzle if the puzzle
//	exists and if the directory last used by the panel is the same the
//	puzzle's path.  Also, only seed the panel's directory when first
//	created, but never after that.  This way, if the user decides to launch
//	new games from some other directory (containing, for instance, puzzles
//	not distributed with SokoSave), we won't keep jumping back to the
//	factory puzzle directory.
//-----------------------------------------------------------------------------
+ (void)newPuzzle
    {
    SokoPool pool = SokoPool_new(0);
    NSString* file = @"";
    char const* seed;

    static NSOpenPanel* openPanel = 0;
    static NSString* openDir = 0;
    static NSArray* openTypes = 0;
    if (openPanel == 0)
	{
	openPanel = [[NSOpenPanel openPanel] retain];
	[openPanel setTitle:@"New Puzzle"];
	[openPanel setAllowsMultipleSelection:YES];
	[openPanel setTreatsFilePackagesAsDirectories:YES];
	openTypes = [[self getOpenTypesWithSeed:0] retain];
	openDir = [[NSString stringWithCString:
	    soko_expand_path(pool, soko_get_puzzle_directory(pool))] retain];
	}

    seed = SokoPuzzle_puzzle_name_for_level( [SokoPref getNextLevel], pool );
    if (soko_path_exists( pool, seed ))
	{
	char const* dir = soko_directory_part( pool, seed );
	if (soko_stricmp( dir, [openDir fileSystemRepresentation] ) == 0)
	    file = [NSString stringWithCString:soko_filename_part(pool, seed)];
	}

    if ([openPanel runModalForDirectory:openDir file:file types:openTypes] ==
	NSOKButton)
	{
	[self openPuzzles:[openPanel filenames]];
	[openDir release];
	openDir = [[openPanel directory] copy];
	}

    SokoPool_destroy( pool );
    }


//-----------------------------------------------------------------------------
// choosePuzzleFromDirectory:
//	Allow both puzzle and saved-game files to be opened.  If "directory" is
//	non-nil, then use the specified directory, else use last visited
//	directory.
//-----------------------------------------------------------------------------
+ (BOOL)choosePuzzleFromDirectory:(NSString*)dir
    {
    BOOL ok = NO;
    SokoPool pool = SokoPool_new(0);

    static NSOpenPanel* openPanel = 0;
    static NSString* openDir = 0;
    static NSArray* openTypes = 0;
    if (openPanel == 0)
	{
	openPanel = [[NSOpenPanel openPanel] retain];
	[openPanel setTitle:@"Open Puzzle"];
	[openPanel setAllowsMultipleSelection:YES];
	[openPanel setTreatsFilePackagesAsDirectories:YES];
	openTypes = [[self getOpenTypesWithSeed:
	    SokoPuzzle_get_save_extension()] retain];
	openDir = [[NSString stringWithCString:
	    soko_expand_path(pool, soko_get_save_directory(pool))] retain];
	}

    if (dir == 0 || [dir isEqualToString:@""])
	dir = openDir;

    if ([openPanel runModalForDirectory:dir file:@"" types:openTypes] ==
	NSOKButton)
	{
	ok = [self openPuzzles:[openPanel filenames]];
	[openDir release];
	openDir = [[openPanel directory] copy];
	}

    SokoPool_destroy( pool );
    return ok;
    }


//-----------------------------------------------------------------------------
// openPuzzle
//-----------------------------------------------------------------------------
+ (void)openPuzzle
    {
    [self choosePuzzleFromDirectory:0];
    }


//-----------------------------------------------------------------------------
// openPuzzleForLevel:
//	This method is called at program launch time to automatically open the
//	next puzzle which the user has not yet solved.  On the assumption that
//	the user may have begun working on the puzzle and then saved it with
//	the intention of continuing work on it later, we first try to open a
//	saved game with the given level number.  If a saved game does not
//	exist, then we try starting a new game.
//-----------------------------------------------------------------------------
+ (BOOL)openPuzzleForLevel:(int)level
    {
    BOOL ok = NO;
    SokoPool pool = SokoPool_new(0);
    char const* newGame = SokoPuzzle_puzzle_name_for_level( level, pool );
    char const* saveGame = SokoPuzzle_save_name_for_puzzle_name(newGame, pool);
    if (soko_path_exists( pool, saveGame ))
	ok = [self openPuzzle:[NSString stringWithCString:saveGame]];
    if (!ok && soko_path_exists( pool, newGame ))
	ok = [self openPuzzle:[NSString stringWithCString:newGame]];
    SokoPool_destroy( pool );
    return ok;
    }


//-----------------------------------------------------------------------------
// openDefaultPuzzle
//-----------------------------------------------------------------------------
+ (BOOL)openDefaultPuzzle
    {
    return [self openPuzzleForLevel:[SokoPref getNextLevel]] ||
	   [self openPuzzleForLevel:1];
    }


//-----------------------------------------------------------------------------
// openNextPuzzle:
//	Invoked by the "Next Puzzle" user interface element.  Opens the puzzle
//	or saved game which is "next" after the puzzle in represented by this
//	SokoBoard instance.
//-----------------------------------------------------------------------------
- (void)openNextPuzzle:(id)sender
    {
    SokoPool pool = SokoPool_new(0);
    int const level = SokoPuzzle_level_for_puzzle_name(
	SokoPuzzle_get_puzzle_file_name(puzzle, pool) );
    if (level >= 0)
	[[self class] openPuzzleForLevel:level + 1];
    SokoPool_destroy( pool );
    }


//-----------------------------------------------------------------------------
// save:
//-----------------------------------------------------------------------------
- (void)save:(id)sender
    {
    SokoPuzzle_abort_playback( puzzle );
    SokoPuzzle_save( puzzle );
    }


//-----------------------------------------------------------------------------
// saveAs:
//-----------------------------------------------------------------------------
- (void)saveAs:(id)sender
    {
    SokoPool pool = SokoPool_new(0);
    NSString* file = [NSString stringWithCString:
	soko_filename_part(pool, SokoPuzzle_get_save_file_name(puzzle, pool))];

    static NSSavePanel* savePanel = 0;
    static NSString* saveDir = 0;
    if (savePanel == 0)
	{
	savePanel = [[NSSavePanel savePanel] retain];
	[savePanel setTitle:@"Save Puzzle"];
	[savePanel setRequiredFileType:
	    [NSString stringWithCString:SokoPuzzle_get_save_extension()]];
	saveDir = [[NSString stringWithCString:
	    soko_expand_path(pool, soko_get_save_directory(pool))] retain];
	}

    SokoPuzzle_abort_playback( puzzle );
    if ([savePanel runModalForDirectory:saveDir file:file] == NSOKButton)
	{
	NSString* path = [savePanel filename];
	if (SokoPuzzle_save_as( puzzle, [path fileSystemRepresentation] ))
	    {
	    [self refreshWindowTitle];
	    [saveDir release];
	    saveDir = [[savePanel directory] copy];
	    }
	}

    SokoPool_destroy( pool );
    }


//-----------------------------------------------------------------------------
// saveAllPuzzles
//-----------------------------------------------------------------------------
+ (void)saveAllPuzzles
    {
    SokoPuzzle_save_all_puzzles();
    }


//=============================================================================
// Busy Operation Support
//=============================================================================
//-----------------------------------------------------------------------------
// processEvents:
//	Whenever SokoPuzzle needs to perform a lengthy operation which may
//	cause the application to become unresponsive, or when it needs to
//	perform some internal looping while still receiving events (such as
//	during player or crate dragging), it calls -processEvents: to ensure
//	that the user interface remains responsive.  If `awaitEvent' is YES
//	then this method returns only after processing at least one event.  If
//	`awaitEvent' is NO, and if there are no pending events, then this
//	method returns immediately.  In either case, if any events are pending,
//	those events are processed and -processEvents: then returns.
//-----------------------------------------------------------------------------
- (void)processEvents:(BOOL)awaitEvent
    {
    BOOL done = NO;
    NSAutoreleasePool* pool1 = [[NSAutoreleasePool alloc] init];
    NSDate* past = [NSDate distantPast];
    NSDate* await = (awaitEvent ? [NSDate distantFuture] : past);
    while (!done)
	{
	NSAutoreleasePool* pool2 = [[NSAutoreleasePool alloc] init];
	NSEvent* e = [window nextEventMatchingMask:NSAnyEventMask
	    untilDate:await inMode:NSEventTrackingRunLoopMode dequeue:YES];
	if (e != 0)
	    [NSApp sendEvent:e];
	else
	    done = YES;
	await = past;
	[pool2 release];
	}
    [pool1 release];
    }


//=============================================================================
// History Playback Session Support
//=============================================================================
- (BOOL)shouldAnimatePlayback { return [animateSwitch state]; }
- (int)playbackThreshold { return 40; }

- (void)stopPressed:(id)sender { SokoPuzzle_abort_playback( puzzle ); }
- (void)animateToggled:(id)sender {}

- (void)windowDidResignKey:(NSNotification*)n
    { SokoPuzzle_abort_playback( puzzle ); }
- (void)windowDidResignMain:(NSNotification*)n
    { SokoPuzzle_abort_playback( puzzle ); }

//-----------------------------------------------------------------------------
// beginPlayback
//	When SokoPuzzle wants to animate a particularly lengthy sequence of
//	moves, it invokes -beginPlayback (assuming that -shouldAnimatePlayback
//	returned YES).  It is the responsibility of -beginPlayback to arrange
//	for SokoPuzzle_advance_playback() to be called on a periodic basis
//	until SokoPuzzle invokes -endPlayback.  In order to keep the user
//	interface responsive (so that the user can cancel playback, for
//	example), the application should be allowed to continue processing and
//	dispatch events.  Both requirements are met via use of an
//	NSNotificationQueue with posting-style NSPostWhenIdle.  Finally, ensure
//	that the cellMatrix is still first-responder so that the user can
//	cancel playback via the Escape key, if desired.
//-----------------------------------------------------------------------------
- (void)beginPlayback
    {
    [stopBtn          setEnabled:YES];
    [animateSwitch    setEnabled:NO];
    [playbackSlider   setEnabled:NO];
    [undoMoveBtn      setEnabled:NO];
    [redoMoveBtn      setEnabled:NO];
    [undoPushBtn      setEnabled:NO];
    [redoPushBtn      setEnabled:NO];
    [moveUpBtn        setEnabled:NO];
    [moveDownBtn      setEnabled:NO];
    [moveLeftBtn      setEnabled:NO];
    [moveRightBtn     setEnabled:NO];
    [moveUpLeftBtn    setEnabled:NO];
    [moveUpRightBtn   setEnabled:NO];
    [moveDownLeftBtn  setEnabled:NO];
    [moveDownRightBtn setEnabled:NO];

    if ([window firstResponder] != cellMatrix)
	[window makeFirstResponder:cellMatrix];

    [[NSNotificationCenter defaultCenter] addObserver:self
	selector:@selector(advancePlayback:)
	name:SOKO_ADVANCE_PLAYBACK_NOTIFICATION object:self];
    [[NSNotificationQueue defaultQueue] enqueueNotification:
	[NSNotification notificationWithName:SOKO_ADVANCE_PLAYBACK_NOTIFICATION
	object:self] postingStyle:NSPostWhenIdle];
    }


//-----------------------------------------------------------------------------
// endPlayback
//	When SokoPuzzle wants to animate a particularly lengthy sequence of
//	moves, it invokes -endPlayback.
// *EVENT*
//	Place a dummy event into the queue to force menu-item validation to
//	fire, thus re-enabling menu items disabled during playback.  Without
//	this, those items would not be re-enabled until the next user event.
// *CONTROLS*
//	Most controls will be re-enabled by normal control-update machinery
//	following conclusion of playback, so we do not need to manually
//	re-enable all of the ones disabled by -beginPlayback.
//-----------------------------------------------------------------------------
- (void)endPlayback
    {
    [NSApp postEvent:[NSEvent otherEventWithType:NSApplicationDefined //*EVENT*
	location:NSMakePoint(0,0) modifierFlags:0 timestamp:0 windowNumber:0
	context:[NSApp context] subtype:0 data1:0 data2:0] atStart:NO];

    [stopBtn        setEnabled:NO ];	// *CONTROLS*
    [animateSwitch  setEnabled:YES];
    [playbackSlider setEnabled:YES];

    [[NSNotificationCenter defaultCenter] removeObserver:self
	name:SOKO_ADVANCE_PLAYBACK_NOTIFICATION object:self];
    }


//-----------------------------------------------------------------------------
// advancePlayback:
//	Each time the application's run-loop becomes idle while history
//	playback is active, this method is invoked in order to advance the
//	playback.  If, after advancing the playback by one step, playback is
//	still active, then another notification is queued for delivery at the
//	next application-idle time.
// *SLIDER*
//	There is a bug in the AppKit where sending -setEnabled:NO to an
//	NSSlider fails to work correctly if it sent in response to an action
//	message from the slider itself (i.e. the slider just stopped tracking
//	the mouse and has just dispatched its action message).  Although, upon
//	receipt of -setEnabled:NO, even though the slider is redrawn with a
//	disabled appearance, it is never-the-less still enabled and still
//	responds to mouse events.  To work around this problem, it is necessary
//	to disable the slider the first time -advancePlayback: is called (even
//	though the slider was supposed to have been disabled by
//	-beginPlayback).
//-----------------------------------------------------------------------------
- (void)advancePlayback:(NSNotification*)n
    {
    if (SokoPuzzle_playback_active( puzzle ))	// Ignore stray notifications.
	{
	if ([playbackSlider isEnabled])		// *SLIDER*
	    [playbackSlider setEnabled:NO];
	SokoPuzzle_advance_playback( puzzle );
	if (SokoPuzzle_playback_active( puzzle ))
	    [[NSNotificationQueue defaultQueue]
		enqueueNotification:[NSNotification
		notificationWithName:SOKO_ADVANCE_PLAYBACK_NOTIFICATION
		object:self] postingStyle:NSPostWhenIdle];
	}
    }


//=============================================================================
// Animation Sequence Support
//=============================================================================
//-----------------------------------------------------------------------------
// beginAnimation
//	When SokoPuzzle is about to animate a series of moves, it invokes
//	-beginAnimation.  Although Sokoboard, itself, need not update any
//	internal state when animation begins, never-the-less, it must deal with
//	an issue specific to Cocoa/OpenStep for Mach.
//
//	In particular, an inactive application typically comes to the
//	foreground when a mouse click is received in one of its views.
//	However, the AppKit delays actual activation until one of the two
//	following conditions arise:
//
//	* The view's -mouseDown: method returns.
//	* The appication's -nextEventMatchingMask:untilDate:inMode:dequeue:
//	  method is invoked from the view's -mouseDown: method.
//
//	This behavior is a problem for several of SokoSave's movement
//	mechanisms because they may perform quite a bit of drawing and
//	on-screen animation before returning from -mouseDown:, and it is
//	confusing for the user to see the program responding to an event
//	while still in the background.
//
//	A further, and much more serious problem, is that there is an apparent
//	bug in the AppKit which causes it to redraw the entire game window for
//	every single "move" made by the program while the application is in the
//	background.  For instance, if the appilcation is in the background, and
//	the user clicks on an empty square twenty cells distant from the
//	player's current position, as the player is moved over that twenty-cell
//	distance, the game window gets redrawn in its entirety for every single
//	step along the path.  This tends to be very time-consuming and makes
//	the program much less responsive than when it is in the foreground.
//
//	In order to work around these problems, it is necessary to force the
//	AppKit to move the program into the foreground before the animation
//	begins.  Unfortunately, even invoking -[NSApplication
//	activateIgnoringOtherApps:YES] does not actually activate the
//	application.  Since the animation is occurring while -mouseDown: is
//	still active, the only alternative left is to make a dummy query to the
//	event queue before beginning animation.  At the time the event-queue is
//	queried, the AppKit allows the application to become active, and the
//	problems is avoided.
//-----------------------------------------------------------------------------
- (void)beginAnimation
    {
    [NSApp nextEventMatchingMask:NSAnyEventMask untilDate:[NSDate distantPast]
	inMode:NSEventTrackingRunLoopMode dequeue:NO];
    }


//-----------------------------------------------------------------------------
// endAnimation
//-----------------------------------------------------------------------------
- (void)endAnimation
    {
    }


//=============================================================================
// Dragging Support
//=============================================================================
- (void)beginDrag {}
- (void)endDrag   {}


//=============================================================================
// History Support
//=============================================================================
- (void)undoMove:(id)sender { SokoPuzzle_undo_move( puzzle ); }
- (void)redoMove:(id)sender { SokoPuzzle_redo_move( puzzle ); }
- (void)undoPush:(id)sender { SokoPuzzle_undo_push( puzzle ); }
- (void)redoPush:(id)sender { SokoPuzzle_redo_push( puzzle ); }

- (void)inspectSlider:(id)sender
    { SokoPuzzle_reenact_history( puzzle, [playbackSlider intValue] ); }


//=============================================================================
// Movement Support
//=============================================================================
- (void)directionPressed:(id)sender
    {
    SokoDirection const dir = (SokoDirection)[sender tag];
    SOKO_ASSERT( dir >= 0 && dir < SOKO_DIR_MAX );
    SokoPuzzle_move( puzzle, dir );
    }

- (void)diagonalPressed:(id)sender
    {
    SokoDiagonal const dir = (SokoDiagonal)[sender tag];
    SOKO_ASSERT( dir >= 0 && dir < SOKO_DIAG_MAX );
    SokoPuzzle_move_diagonal( puzzle, dir );
    }


//=============================================================================
// Keyboard Support
//=============================================================================
//-----------------------------------------------------------------------------
// eventFlags:
//-----------------------------------------------------------------------------
- (SokoEventFlags)eventFlags:(unsigned int)nsflags
    {
    SokoEventFlags flags = 0;
    if ((nsflags & NSShiftKeyMask    ) != 0) flags |= SOKO_FLAG_SHIFT;
    if ((nsflags & NSControlKeyMask  ) != 0) flags |= SOKO_FLAG_CONTROL;
    if ((nsflags & NSAlternateKeyMask) != 0) flags |= SOKO_FLAG_ALTERNATE;
    if ((nsflags & NSCommandKeyMask  ) != 0) flags |= SOKO_FLAG_COMMAND;
    return flags;
    }


//-----------------------------------------------------------------------------
// keyAction:isDown:
//-----------------------------------------------------------------------------
- (void)keyAction:(NSEvent*)e isDown:(BOOL)down
    {
    #define ESC_CODE       (0x1b)
    #define KP_UP          ('8')
    #define KP_DOWN        ('2')
    #define KP_LEFT        ('4')
    #define KP_RIGHT       ('6')
    #define KP_HOME        ('7')
    #define KP_PAGE_UP     ('9')
    #define KP_END         ('1')
    #define KP_PAGE_DOWN   ('3')
    #define KP_CENTER      ('5')

    SokoKeyCode code = ((SokoKeyCode)~0);
    unsigned int const flags = [e modifierFlags];
    unichar const c = [[e charactersIgnoringModifiers] characterAtIndex:0];
    if ((flags & NSFunctionKeyMask) != 0)
	{
	switch (c)
	    {
	    case NSUpArrowFunctionKey:    code = SOKO_KEY_UP;         break;
	    case NSDownArrowFunctionKey:  code = SOKO_KEY_DOWN;       break;
	    case NSLeftArrowFunctionKey:  code = SOKO_KEY_LEFT;       break;
	    case NSRightArrowFunctionKey: code = SOKO_KEY_RIGHT;      break;
	    case NSInsertFunctionKey:     code = SOKO_KEY_UP_LEFT;    break;
	    case NSDeleteFunctionKey:     code = SOKO_KEY_DOWN_LEFT;  break;
	    case NSPageUpFunctionKey:     code = SOKO_KEY_UP_RIGHT;   break;
	    case NSPageDownFunctionKey:   code = SOKO_KEY_DOWN_RIGHT; break;
	    };
	}
    else if ((flags & NSNumericPadKeyMask) != 0)
	{
	switch (c)
	    {
	    case KP_UP:        code = SOKO_KEY_UP;         break;
	    case KP_DOWN:      code = SOKO_KEY_DOWN;       break;
	    case KP_LEFT:      code = SOKO_KEY_LEFT;       break;
	    case KP_RIGHT:     code = SOKO_KEY_RIGHT;      break;
	    case KP_HOME:      code = SOKO_KEY_UP_LEFT;    break;
	    case KP_PAGE_UP:   code = SOKO_KEY_UP_RIGHT;   break;
	    case KP_END:       code = SOKO_KEY_DOWN_LEFT;  break;
	    case KP_PAGE_DOWN: code = SOKO_KEY_DOWN_RIGHT; break;
	    case KP_CENTER:    code = SOKO_KEY_DOWN;       break;
	    };
	}
    else
	{
	switch (c)
	    {
	    case ESC_CODE: code = SOKO_KEY_ESCAPE; break;
	    };
	}

    if (code != ((SokoKeyCode)~0))
	SokoPuzzle_key_action( puzzle, code, down, [self eventFlags:flags] );
    }

- (void)keyDown:(NSEvent*)e { [self keyAction:e isDown:YES]; }
- (void)keyUp:  (NSEvent*)e { [self keyAction:e isDown:NO ]; }


//=============================================================================
// Mouse Support
//=============================================================================
//-----------------------------------------------------------------------------
// getRow:column:forEvent:
//-----------------------------------------------------------------------------
- (BOOL)getRow:(int*)r column:(int*)c forEvent:(NSEvent*)e
    {
    return [cellMatrix getRow:r column:c
	forPoint:[cellMatrix convertPoint:[e locationInWindow] fromView:0]];
    }


//-----------------------------------------------------------------------------
// mouseAction:button:isDown:
//-----------------------------------------------------------------------------
- (void)mouseAction:(NSEvent*)e button:(int)button isDown:(BOOL)down
    {
    int row, col;
    if ([self getRow:&row column:&col forEvent:e])
	SokoPuzzle_mouse_action( puzzle, button, down,
	    [self eventFlags:[e modifierFlags]], row, col );
    }

- (void)mouseDown:(NSEvent*)e
    { [self mouseAction:e button:0 isDown:YES]; }
- (void)mouseUp:(NSEvent*)e
    { [self mouseAction:e button:0 isDown:NO ]; }
- (void)rightMouseDown:(NSEvent*)e
    { [self mouseAction:e button:1 isDown:YES]; }
- (void)rightMouseUp:(NSEvent*)e
    { [self mouseAction:e button:1 isDown:NO ]; }


//-----------------------------------------------------------------------------
// mouseDragged:
//-----------------------------------------------------------------------------
- (void)mouseDragged:(NSEvent*)e
    {
    int row, col;
    if ([self getRow:&row column:&col forEvent:e])
	SokoPuzzle_mouse_drag( puzzle, row, col );
    }

- (void)rightMouseDragged:(NSEvent*)e { [self mouseDragged:e]; }

@end
