//-----------------------------------------------------------------------------
// SokoApp.m
//
//	Application delegate for SokoSave.
//
// Copyright (c), 1997,2001,2002, Eric Sunshine <sunshine@sunshineco.com>
// Copyright (c), 1997, Paul McCarthy <zarnuk@high-speed-software.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoApp.m,v 1.7 2002/02/19 09:24:07 sunshine Exp $
// $Log: SokoApp.m,v $
// Revision 1.7  2002/02/19 09:24:07  sunshine
// v18
// -*- Now notifies Windows shell after all file associations have been made,
//     rather than after each registration.
//
// Revision 1.6  2002-01-29 15:31:25-05  sunshine
// v17
// -*- Added descriptions for new .sokohex, .xsb, and .hsb file associations
//     to CustomInfo.plist.  Also removed hard-coded .ico extension from icon
//     names in this list, since the extensions were meaningful only for
//     Windows.
//
// Revision 1.5  2001-12-23 13:10:42-05  sunshine
// v15
// -*- Extracted core game logic out of GUI code and generalized it so that
//     the same core code can be used by any platform.  Logic from
//     SokoBoard.m now resides in SokoPuzzle.c, etc.
//
// -*- Rewrote the OpenStep port so that it utilizes the new "sokocore" game
//     library rather than implementing that logic directly.
//
// -*- Replaced the terminology "maze" with "puzzle" throughout the project,
//     including source code, documentation, and all user-visible UI
//     elements.  The only remaining place where "maze" is still used is in
//     the file extension ".sokomaze".  I haven't decided what, if anything,
//     to do about that, yet.
//
// -*- Converted the documentation from RTFD format to HTML format.
//     Documentation is no longer embedded in a scrollable text object on the
//     Info panel.  Instead, a "Help" menu item now displays the help file
//     via a web browser.
//
// -*- Added a "Help" menu item which opens the HTML help file in a web
//     browser.  Previously, the help information was contained in an RTFD
//     file and loaded directly into a text object on the Info panel.
//
// -*- Added SokoSetting implementation which provides a platform-independent
//     API for accessing user settings and well-known paths, such as
//     $(SokoSave) and $(SokoUser).
//
// -*- Changed the logic used to locate the puzzle which is opened
//     automatically at program launch time.  Previously, it simply opened a
//     .sokomaze file with the level number recorded as the "highest" level.
//     Unfortunately, if the user then tried to open a saved game file with
//     the same level number, the program would not actually open the saved
//     game because it thought that the game was already open.  This made it
//     very difficult for the user to continue a saved game which matched the
//     "highest" level.  The user first has to close the "new" game and then
//     open the "saved" game.  This problem was even further complicated on
//     the Windows platform where closing the "new" game would cause the
//     program to terminate if that was the only menu-bearing window.  Since
//     it is probable that a user may save a game before actually arriving at
//     the solution with the intention of returning to that game later, the
//     program now makes it much easier to return to the saved game.  Now, at
//     launch time, the program first checks for a "saved" game matching the
//     "highest" level and open that if present.  If not, then it falls back
//     to opening a new game.
//
// -*- For safety, user defaults (including score panel column order and
//     widths) are now committed immediately, rather than at termination
//     time.
//
// -*- Renamed old SokoWindows.c to SokoApp-windows.c since this
//     Windows-specific module is related to SokoApp.  (Also, this helps to
//     disambiguate it from the new SokoWindow class and implementation
//     file.)
//
// -*- When registering file associations on Windows, now uses an appropriate
//     description for each association, rather than the generic "SokoSave
//     Document".  Added CustomInfo.plist which contains an appropriate
//     description for each file extension.
//
// -*- Created icons for .sokomaze and .sokosave so that they can be
//     distinguished from the application and from each other on platforms
//     (like Windows) which hide the file extension by default.  Created both
//     ICO and TIFF formats.
//-----------------------------------------------------------------------------
#import "SokoApp.h"
#import "SokoPool.h"
#import	"SokoBoard.h"
#import	"SokoConfig.h"
#import "SokoFile.h"
#import "SokoInfo.h"
#import "SokoPref.h"
#import "SokoPuzzle.h"
#import	"SokoScores.h"
#import "SokoSetting.h"
#import "SokoUtil.h"
#import <AppKit/NSApplication.h>
#import <AppKit/NSMenu.h>
#import <AppKit/NSWorkspace.h>
#import <Foundation/NSBundle.h>
#import <Foundation/NSDictionary.h>
#import <Foundation/NSFileManager.h>
#import <Foundation/NSPathUtilities.h>

#define NEW_PLAYER_SETTING "NewPlayer"
#define SOKO_HELP_DIR @"help"
#define SOKO_HELP_FILE @"SokoHelp.html"

#if defined(SOKO_REGISTER_FILE_ASSOCIATIONS)
# if !defined(SOKO_FILE_ICON_EXTENSION)
#  error SOKO_FILE_ICON_EXTENSION not defined
#else
void SokoBeginFileAssociationRegistration();
int SokoRegisterFileAssociation(
    char const* extension, char const* iconpath, char const* description );
void SokoEndFileAssociationRegistration( int changed );
# endif
#endif

@implementation SokoApp

- (void)new:(id)sender               { [SokoBoard  newPuzzle]; }
- (void)open:(id)sender              { [SokoBoard  openPuzzle]; }
- (void)saveAll:(id)sender           { [SokoBoard  saveAllPuzzles]; }
- (void)launchInfo:(id)sender        { [SokoInfo   launch]; }
- (void)launchPreferences:(id)sender { [SokoPref   launch]; }
- (void)launchScores:(id)sender      { [SokoScores launch]; }

//-----------------------------------------------------------------------------
// init
//-----------------------------------------------------------------------------
- (id)init
    {
    [super init];
    autoOpenPuzzle = YES;
    return self;
    }


//-----------------------------------------------------------------------------
// registerFileAssociations
//	On platforms which do not automatically glean file associations (such
//	as .sokosave and .sokomaze) directly from the application wrapper, they
//	must be programmatically registered.
//-----------------------------------------------------------------------------
- (void)registerFileAssociations
    {
#if defined(SOKO_REGISTER_FILE_ASSOCIATIONS)
    NSBundle* bundle = [NSBundle mainBundle];
    NSDictionary* extensions =
	[[bundle infoDictionary] objectForKey:@"SokoExtensions"];
    if (extensions != 0)
	{
	int changed = 0;
	NSString* ext;
	NSEnumerator* keys = [extensions keyEnumerator];
	SokoBeginFileAssociationRegistration();
	while ((ext = [keys nextObject]) != 0)
	    {
	    NSDictionary* dict = [extensions objectForKey:ext];
	    if (dict != 0)
		{
		NSString* icon = [dict objectForKey:@"icon"];
		if (icon != 0)
		    {
		    NSString* path = [bundle
			pathForResource:[icon stringByDeletingPathExtension]
			ofType:SOKO_FILE_ICON_EXTENSION];
		    if (path != 0)
			{
			NSString* desc = [dict objectForKey:@"description"];
			char const* cdesc =
			    (desc != 0 ? [desc cString] : "SokoSave document");
			changed += SokoRegisterFileAssociation(
			    [ext cString], [path cString], cdesc );
			}
		    }
		}
	    }
	SokoEndFileAssociationRegistration( changed );
	}
#endif
    }


//-----------------------------------------------------------------------------
// applicationWillFinishLaunching:
//-----------------------------------------------------------------------------
- (void)applicationWillFinishLaunching:(NSNotification*)n
    {
    [SokoPref startup];
    [self registerFileAssociations];
    }


//-----------------------------------------------------------------------------
// choosePuzzle
//	At application launch time, give user a chance to explicitly open a
//	puzzle or saved game if SokoBoard was unable to automatically open
//	the "default" puzzle.
//-----------------------------------------------------------------------------
- (BOOL)choosePuzzle
    {
    SokoPool pool = SokoPool_new(0);
    BOOL ok = [SokoBoard choosePuzzleFromDirectory:[NSString stringWithCString:
	soko_expand_path(pool, soko_get_puzzle_directory(pool))]];
    SokoPool_destroy( pool );
    return ok;
    }


//-----------------------------------------------------------------------------
// activate:
//-----------------------------------------------------------------------------
- (void)activate:(BOOL)puzzleRequired
    {
    BOOL ok = (SokoPuzzle_open_puzzle_count() > 0);
    if (!ok)
	ok = [SokoBoard openDefaultPuzzle];

    if (!ok && puzzleRequired)
	ok = [self choosePuzzle];

    if (!ok && puzzleRequired)
	[NSApp terminate:self];
    else
	{
	SokoSetting settings = SokoSetting_new(0);
	if (SokoSetting_get_bool( settings, NEW_PLAYER_SETTING, soko_true ))
	    {
	    SokoSetting_set_bool( settings, NEW_PLAYER_SETTING, soko_false );
	    [self launchInfo:self];
	    }
	SokoSetting_destroy( settings );
	}
    }


//-----------------------------------------------------------------------------
// applicationDidFinishLaunching:
//-----------------------------------------------------------------------------
#ifndef SOKO_DEFERRED_LAUNCH
- (void)applicationDidFinishLaunching:(NSNotification*)n
    {
    [self activate:YES];
    }
#endif


//-----------------------------------------------------------------------------
// applicationDidBecomeActive:
//-----------------------------------------------------------------------------
#ifdef SOKO_DEFERRED_LAUNCH
- (void)applicationDidBecomeActive:(NSNotification*)n
    {
    if (autoOpenPuzzle)		// Only do this the first time the 
	{			// application becomes active.
	autoOpenPuzzle = NO;
	[self activate:NO];
	}
    }
#endif


//-----------------------------------------------------------------------------
// applicationWillTerminate:
//-----------------------------------------------------------------------------
- (void)applicationWillTerminate:(NSNotification*)n
    {
    [SokoPref shutdown];
    }


//-----------------------------------------------------------------------------
// application:openFile:
//-----------------------------------------------------------------------------
- (BOOL)application:(id)sender openFile:(NSString*)path
    {
    return [SokoBoard openPuzzle:path];
    }


//-----------------------------------------------------------------------------
// launchInstructions:
//-----------------------------------------------------------------------------
- (void)launchInstructions:(id)sender
    {
    NSString* path =
	[[NSBundle mainBundle] pathForResource:SOKO_HELP_DIR ofType:@""];
    if (path != 0)
	{
	path = [path stringByAppendingPathComponent:SOKO_HELP_FILE];
	if ([[NSFileManager defaultManager] fileExistsAtPath:path])
	    [[NSWorkspace sharedWorkspace] openFile:path];
	}
    }


//-----------------------------------------------------------------------------
// validateMenuItem:
//-----------------------------------------------------------------------------
- (BOOL)validateMenuItem:(NSMenuItem*)item
    {
    if ([item action] == @selector(saveAll:))
	return (SokoPuzzle_open_puzzle_count() > 0);
    return YES;
    }

@end
