//-----------------------------------------------------------------------------
// SokoApp-windows.c
//
//	Microsoft Windows-specific functionality.
//
// Copyright (c), 2001,2002, Eric Sunshine <sunshine@sunshineco.com>
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoApp-windows.c,v 1.3 2002/02/19 09:24:02 sunshine Exp $
// $Log: SokoApp-windows.c,v $
// Revision 1.3  2002/02/19 09:24:02  sunshine
// v18
// -*- Now notifies Windows shell after all file associations have been made,
//     rather than after each registration.
//
// Revision 1.2  2001-12-23 11:01:42-05  sunshine
// v15
// -*- Renamed old SokoWindows.c to SokoApp-windows.c since this
//     Windows-specific module is related to SokoApp.  (Also, this helps to
//     disambiguate it from the new SokoWindow class and implementation
//     file.)
//
// -*- When registering file associations on Windows, now uses an appropriate
//     description for each association, rather than the generic "SokoSave
//     Document".  Added CustomInfo.plist which contains an appropriate
//     description for each file extension.
//
// Revision 1.1  2001/08/19 12:47:06  sunshine
// v11.1
// -*- On Windows, file associations for the .sokomaze and .sokosave
//     extensions are programmatically added to the registry if absent or
//     incorrect.  This allows users to double-click on these files and have
//     SokoSave automatically launched as a result.
//-----------------------------------------------------------------------------
struct IShellView; // Pacify compiler when including shlobj.h.
#include <shlobj.h>
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <wtypes.h>
#include <winreg.h>

#define SOKO_DEBUG

//-----------------------------------------------------------------------------
// Report a Windows-specific error
//-----------------------------------------------------------------------------
static void win32_error(char const* msg, long err)
    {
#if defined(SOKO_DEBUG)
    LPVOID buff;
    if (FormatMessage(
	FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
	0, err, 0, (LPTSTR)&buff, 0, 0 ) != 0)
	{
	fprintf(stderr, "Win32 Error: %s (%ld): %s\n", msg, err, (char*)buff);
	LocalFree( buff );
	}
    else
	fprintf( stderr, "Win32 Error: %s (%ld)\n", msg, err );
#endif
    (void)msg; (void)err;
    }


//-----------------------------------------------------------------------------
// Read a registry value.  Caller is responsible for invoking free() on result.
//-----------------------------------------------------------------------------
static char* reg_get_value(HKEY key, char const* subkey, char const* name, ...)
    {
    char* value = 0;
    HKEY h;
    char keybuff[ 1024 ];
    va_list args;
    va_start( args, name );
    vsprintf( keybuff, subkey, args );
    va_end( args );

    if (RegOpenKeyEx( key, keybuff, 0, KEY_READ, &h ) == ERROR_SUCCESS)
	{
	DWORD len;
	if (RegQueryValueEx(h, name, 0,0,0, &len) == ERROR_SUCCESS && len != 0)
	    {
	    char* buff = (char*)malloc( len * sizeof(buff[0]) );
	    if (RegQueryValueEx( h, name, 0, 0, buff, &len ) == ERROR_SUCCESS)
		value = buff;
	    else
		free( buff );
	    }
	RegCloseKey(h);
	}
    return value;
    }


//-----------------------------------------------------------------------------
// Set the value of a registry key; creating the key if necessary.
//-----------------------------------------------------------------------------
static void reg_set_value(HKEY key, char const* subkey, char const* arg,
    char const* name, char const* value, ...)
    {
    HKEY h;
    long rc;
    DWORD disp;
    char keybuff[ 1024 ];
    char valbuff[ 1024 ];
    va_list args;
    va_start( args, value );
    vsprintf( valbuff, value, args );
    va_end( args );
    sprintf( keybuff, subkey, arg );

    rc = RegCreateKeyEx(
	key, keybuff, 0, 0, REG_OPTION_NON_VOLATILE, KEY_WRITE, 0, &h, &disp );
    if (rc != ERROR_SUCCESS)
	win32_error( "Failed to create registry key", rc );
    else
	{
	rc = RegSetValueEx( h, name, 0, REG_SZ, valbuff, strlen(valbuff) + 1 );
	if (rc != ERROR_SUCCESS)
	    win32_error( "Failed to write registry key", rc );
	RegCloseKey(h);
	}
    }


//-----------------------------------------------------------------------------
// Register a file type association with Windows.  The file type is registered
// if it does not already exist or if an existing registration has the wrong
// path.
//-----------------------------------------------------------------------------
int SokoRegisterFileAssociation(
    char const* ext, char const* icon, char const* desc )
    {
    int do_registration = 0;
    char exe_path[ FILENAME_MAX ];
    if (GetModuleFileName( 0, exe_path, sizeof(exe_path) ) == 0)
	win32_error( "Unable to determine SokoSave path", GetLastError() );
    else
	{
	char* s;
	do_registration = 1;
	s = reg_get_value(HKEY_CLASSES_ROOT,"SokoSave.%s\\DefaultIcon",0,ext);
	if (s != 0)
	    {
	    do_registration = (stricmp( s, icon ) != 0);
	    free(s);
	    }
	if (do_registration)
	    {
	    reg_set_value(HKEY_CLASSES_ROOT,".%s", ext, 0, "SokoSave.%s", ext);
	    reg_set_value(HKEY_CLASSES_ROOT, "SokoSave.%s", ext, 0, desc);
	    reg_set_value(HKEY_CLASSES_ROOT,
		"SokoSave.%s\\DefaultIcon", ext, 0, "%s", icon );
	    reg_set_value(HKEY_CLASSES_ROOT,
		"SokoSave.%s\\shell\\Open\\command", ext, 0,
		"%s -NSUseRunningCopy YES -NSOpen \"%%1\"", exe_path);
	    }
	}
    return do_registration;
    }


//-----------------------------------------------------------------------------
// Begin file association registration process.
//-----------------------------------------------------------------------------
void SokoBeginFileAssociationRegistration()
    {
    }


//-----------------------------------------------------------------------------
// Complete file association registration process.
//-----------------------------------------------------------------------------
void SokoEndFileAssociationRegistration( int changed )
    {
    if (changed != 0)
	SHChangeNotify( SHCNE_ASSOCCHANGED, SHCNF_IDLIST, 0, 0 );
    }
