//-----------------------------------------------------------------------------
// SokoURL.m
//
//	Subclass of TextField which displays and dispatches a URL.
//
// Copyright (c), 2001, Eric Sunshine <sunshine@sunshineco.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoURL.m,v 1.1 2001/12/24 20:16:14 sunshine Exp $
// $Log: SokoURL.m,v $
// Revision 1.1  2001/12/24 20:16:14  sunshine
// v15
// -*- Added the SokoURL class for use by the Info panel.  This class
//     displays fly-over, clickable URLs.  It handles mailto: URLs directly;
//     all others are handed off to OmniWeb's "Open URL" service.
//
// -*- Email addresses (and names) on Info panel are now active links which
//     launch the mail program.  The email addresses appears underlined as
//     the mouse flies over it.
//
// -*- Added a SokoSave URL to the Info panel.  The URL is an active link
//     which launches the web browser when clicked.  The URL appears
//     underlined as the mouse flies over it.
//
//-----------------------------------------------------------------------------
#import "SokoURL.h"
#import "SokoPool.h"
#import "SokoUtil.h"
#import <appkit/Application.h>
#import <appkit/Cell.h>
#import <appkit/color.h>
#import <appkit/Font.h>
#import <appkit/NXCursor.h>
#import <appkit/NXImage.h>
#import <appkit/Text.h>
#import <appkit/Window.h>
#import <dpsclient/event.h>
#import <math.h>
#import <libc.h>

BOOL soko_dispatch_url( char const* url );

//=============================================================================
// SokoURL
//=============================================================================
@implementation SokoURL

- (BOOL)acceptsFirstMouse { return YES; }
- (char const*)url { return url; }

//-----------------------------------------------------------------------------
// handCursor
//-----------------------------------------------------------------------------
+ (NXCursor*)handCursor
    {
    static NXCursor* cursor = 0;
    if (cursor == 0)
	{
	NXImage* image = [NXImage findImageNamed:"SokoHand"];
	if (image == 0)
	    cursor = NXArrow;
	else
	    {
	    NXPoint const hot_spot = { 6, 0 };
	    cursor = [[NXCursor alloc] initFromImage:image];
	    [cursor setHotSpot:&hot_spot];
	    [cursor setOnMouseEntered:YES];
	    }
	}
    return cursor;
    }


//-----------------------------------------------------------------------------
// setColor:underline:
//-----------------------------------------------------------------------------
- (void)setColor:(NXColor)c underline:(BOOL)u
    {
    underlined = u;
    [self setTextColor:c];
    [self display];
    }


//-----------------------------------------------------------------------------
// adjustAttributes
//-----------------------------------------------------------------------------
- (void)adjustAttributes
    {
    NXColor c;
    if (!urlValid)
	c = NX_COLORBLACK;
    else if (!mouseInside)
	c = NX_COLORPURPLE;
    else if (trackingMouse) // && mouseInside
	c = NX_COLORRED;
    else // (mouseInside && !trackingMouse)
	c = NX_COLORBLUE;
    [self setColor:c underline:(urlValid && mouseInside)];
    }


//-----------------------------------------------------------------------------
// setStringValue:
//-----------------------------------------------------------------------------
- (id)setStringValue:(char const*)s
    {
    [super setStringValue:s];
    [self adjustAttributes];
    return self;
    }


//-----------------------------------------------------------------------------
// validateUrl:
//-----------------------------------------------------------------------------
- (BOOL)validateUrl:(char const*)u
    {
    return (u != 0 && *u != '\0');
    }


//-----------------------------------------------------------------------------
// setUrl:
//-----------------------------------------------------------------------------
- (id)setUrl:(char const*)u
    {
    if (u != url)
	{
	free( url );
	url = soko_strdup(u);
	urlValid = [self validateUrl:url];
	if (!urlValid)
	    {
	    mouseInside = trackingMouse = NO;
	    if (trackingRectEstablished)
		[[self window] discardTrackingRect:(int)self];
	    }
	[[self window] invalidateCursorRectsForView:self];
	[self adjustAttributes];
	}
    return self;
    }


//-----------------------------------------------------------------------------
// setStringValue:url:
//-----------------------------------------------------------------------------
- (id)setStringValue:(char const*)s url:(char const*)u
    {
    [self setStringValue:s];
    [self setUrl:u];
    return self;
    }


//-----------------------------------------------------------------------------
// dispatchUrl
//-----------------------------------------------------------------------------
- (BOOL)dispatchUrl
    {
    return (urlValid && soko_dispatch_url( url ));
    }


//-----------------------------------------------------------------------------
// textRect
//-----------------------------------------------------------------------------
- (NXRect)textRect
    {
    NXRect b;
    NXSize s;
    Cell* c = [self cell];
    [self getBounds:&b];
    [c getDrawRect:&b];
    [c calcCellSize:&s inRect:&b];
    if (s.width < b.size.width)
	{
	int const a = [self alignment];
	switch (a)
	    {
	    case NX_LEFTALIGNED:
		b.size = s;
		break;
	    case NX_RIGHTALIGNED:
		b.origin.x += b.size.width - s.width;
		b.size = s;
		break;
	    case NX_CENTERED:
		b.origin.x += floor((b.size.width - s.width) / 2.0);
		b.size = s;
		break;
	    }
	}
    return b;
    }


//-----------------------------------------------------------------------------
// visibleTextRect
//-----------------------------------------------------------------------------
- (NXRect)visibleTextRect
    {
    NXRect t, v;
    t = [self textRect];
    [self getVisibleRect:&v];
    NXIntersectionRect( &t, &v );
    return v;
    }


//-----------------------------------------------------------------------------
// resetCursorRects
//-----------------------------------------------------------------------------
- (id)resetCursorRects
    {
    if (urlValid)
	{
	NXRect r = [self visibleTextRect];
	[self addCursorRect:&r cursor:[[self class] handCursor]];
	}
    return self;
    }


//-----------------------------------------------------------------------------
// drawUnderline
//-----------------------------------------------------------------------------
- (void)drawUnderline
    {
    NXRect r = [self visibleTextRect];
    NXCoord ascender, descender, line_height;
    NXTextFontInfo( [self font], &ascender, &descender, &line_height );
    r.origin.y += floor(line_height - descender + 1);
    r.size.height = 1;
    NXSetColor( [self textColor] );
    NXRectFill(&r);
    }


//-----------------------------------------------------------------------------
// drawCellInside:
//-----------------------------------------------------------------------------
- (id)drawCellInside:(id)c
    {
    [super drawCellInside:c];
    if (underlined && c == [self cell] && [self canDraw])
	{
	[self lockFocus];
	[self drawUnderline];
	[self unlockFocus];
	}
    return self;
    }


//-----------------------------------------------------------------------------
// drawSelf::
//-----------------------------------------------------------------------------
- (id)drawSelf:(NXRect const*)r :(int)n
    {
    [super drawSelf:r:n];
    if (underlined)
	[self drawUnderline];
    return self;
    }


//-----------------------------------------------------------------------------
// mouseEntered:
//-----------------------------------------------------------------------------
- (id)mouseEntered:(NXEvent*)e
    {
    if (urlValid)
	{
	mouseInside = YES;
	[self adjustAttributes];
	}
    return self;
    }


//-----------------------------------------------------------------------------
// mouseExited:
//-----------------------------------------------------------------------------
- (id)mouseExited:(NXEvent*)e
    {
    if (urlValid)
	{
	mouseInside = NO;
	[self adjustAttributes];
	}
    return self;
    }


//-----------------------------------------------------------------------------
// mouseDown:
//	Track the mouse.  If button is released while the mouse is within our
//	URL area, then dispatch the URL.
// *INSIDE*
//	It is possible for -mouseDown: to be invoked when the tracking
//	rectangle is not established.  This can happen, for instance, if the
//	window is not the key window, and the window's delegate has invoked
//	-stopTracking.  In this case, since -mouseEnter: will not have been
//	received, the `mouseInside' flag will be NO even though the mouse may
//	really be "inside".  Therefore, we must manually check the mouse's
//	position in order to ensure that `mouseInside' is correctly set.
// *DISCARD*
//	Upon mouse-up, discard other pending events which piled up while
//	tracking the mouse.  This prevents mouse-entered and mouse-exited
//	events to this URL field and others on the window from being processed
//	unnecessarily.  (They don't need to be processed because we were
//	manually tracking the mouse.)
//-----------------------------------------------------------------------------
- (id)mouseDown:(NXEvent*)e
    {
    if (urlValid)
	{
	NXPoint p = e->location;
	NXRect r = [self visibleTextRect];
	Window* w = [self window];
	int mask = [w addToEventMask:NX_LMOUSEDRAGGEDMASK];
	[self convertPoint:&p fromView:0];
	mouseInside = [self mouse:&p inRect:&r];		// *INSIDE*
	trackingMouse = YES;
	[self adjustAttributes];
	while (trackingMouse)
	    {
	    NXEvent* e = [NXApp getNextEvent:
		NX_LMOUSEUPMASK | NX_LMOUSEDRAGGEDMASK];
	    if (e->type == NX_LMOUSEUP)
		{
		trackingMouse = NO;
		do { e = [NXApp peekAndGetNextEvent:		// *DISCARD*
		    NX_MOUSEENTEREDMASK|NX_MOUSEEXITEDMASK]; } while (e != 0);
		}
	    else
		{
		BOOL inside;
		p = e->location;
		[self convertPoint:&p fromView:0];
		inside = [self mouse:&p inRect:&r];
		if (inside != mouseInside)
		    {
		    mouseInside = inside;
		    [self adjustAttributes];
		    }
		}
	    }
	[w setEventMask:mask];
	if (mouseInside)
	    [self dispatchUrl];
	[self adjustAttributes];
	}
    return self;
    }


//-----------------------------------------------------------------------------
// startTracking
//-----------------------------------------------------------------------------
- (void)startTracking
    {
    if (!trackingRectEstablished && urlValid)
	{
	NXRect r = [self visibleTextRect];
	[self convertRect:&r toView:0];
	[[self window] setTrackingRect:&r inside:NO owner:self tag:(int)self
	    left:NO right:NO];
	trackingRectEstablished = YES;
	}
    }


//-----------------------------------------------------------------------------
// stopTracking
//-----------------------------------------------------------------------------
- (void)stopTracking
    {
    if (trackingRectEstablished)
	{
	[[self window] discardTrackingRect:(int)self];
	trackingRectEstablished = NO;
	mouseInside = NO;
	trackingMouse = NO;
	[self adjustAttributes];
	}
    }


//-----------------------------------------------------------------------------
// urlInit:
//-----------------------------------------------------------------------------
- (void)urlInit:(NXRect const*)r
    {
    [super initFrame:r];
    [self setFont:[Font newFont:"Helvetica" size:12]];
    [self setBackgroundTransparent:YES];
    [self setBezeled:NO];
    [self setBordered:NO];
    [self setEditable:NO];
    [self setSelectable:NO];
    url = soko_strdup("");
    urlValid = NO;
    trackingRectEstablished = NO;
    mouseInside = NO;
    trackingMouse = NO;
    underlined = NO;
    }


//-----------------------------------------------------------------------------
// init
//-----------------------------------------------------------------------------
- (id)init
    {
    NXRect const r = {{ 0, 0 }, { 10, 10 }};
    [self urlInit:&r];
    return self;
    }


//-----------------------------------------------------------------------------
// initFrame:
//-----------------------------------------------------------------------------
- (id)initFrame:(NXRect const*)r
    {
    [self urlInit:r];
    return self;
    }


//-----------------------------------------------------------------------------
// free
//-----------------------------------------------------------------------------
- (id)free
    {
    [self stopTracking];
    free( url );
    return [super free];
    }

@end
