//-----------------------------------------------------------------------------
// SokoURL-nextstep.m
//
//	URL dispatcher for NextStep.
//
// Copyright (c), 2001, Eric Sunshine <sunshine@sunshineco.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoURL-nextstep.m,v 1.1 2001/12/23 15:31:21 sunshine Exp $
// $Log: SokoURL-nextstep.m,v $
// Revision 1.1  2001/12/23 15:31:21  sunshine
// v15
// -*- Added the SokoURL class for use by the Info panel.  This class
//     displays fly-over, clickable URLs.  It handles mailto: URLs directly;
//     all others are handed off to OmniWeb's "Open URL" service.
//
// -*- Email addresses (and names) on Info panel are now active links which
//     launch the mail program.  The email addresses appears underlined as
//     the mouse flies over it.
//
// -*- Added a SokoSave URL to the Info panel.  The URL is an active link
//     which launches the web browser when clicked.  The URL appears
//     underlined as the mouse flies over it.
//
//-----------------------------------------------------------------------------
#import <appkit/Application.h>
#import <appkit/Listener.h>
#import <appkit/Panel.h>
#import <appkit/Speaker.h>
#import <appkit/workspaceRequest.h>
#import <defaults/defaults.h>
#import <mach/mach_init.h>
#import <mach/mach_interface.h>
#import "SokoPool.h"
#import "SokoBuild.h"
#import "SokoRelease.h"
#import "SokoUtil.h"

#define SOKO_URL_SERVICE "OmniWeb/Open URL"

//=============================================================================
// Dispatcher for mailto: URLs.
//=============================================================================
@interface SokoMailSpeaker : Speaker
    {
    port_t port;
    }
- (id)init;
- (BOOL)sendMailTo:(char const*)to subject:(char const*)subject;
@end

@implementation SokoMailSpeaker

//-----------------------------------------------------------------------------
// getMailerPort
//	Get the Mach port advertised by the GUI mailer program.  If no program
//	is advertising that port, try launching the mailer program specified
//	by the user default "Mailer".  If that fails, try launching Mail.app.
//-----------------------------------------------------------------------------
+ (port_t)getMailerPort
    {
    port_t p = NXPortNameLookup( "MailSendDemo", 0 );
    if (p == PORT_NULL)
	{ // Mailer not running, try "Mailer" from defaults, then "Mail.app".
	BOOL launched = NO;
	id<NXWorkspaceRequestProtocol> w = [Application workspace];
	char const* mailer = NXGetDefaultValue( "SokoSave", "Mailer" );
	if (mailer != 0 && *mailer != '\0')
	    launched = [w launchApplication:mailer];
	if (!launched)
	    launched = [w launchApplication:"Mail"];
	if (launched)
	    p = NXPortFromName( "MailSendDemo", 0 );
	}
    return p;
    }


//-----------------------------------------------------------------------------
// init
//-----------------------------------------------------------------------------
- (id)init
    {
    [super init];
    port = [[self class] getMailerPort];
    if (port != PORT_NULL)
	[self setSendPort:port];
    return self;
    }


//-----------------------------------------------------------------------------
// free
//-----------------------------------------------------------------------------
- (id)free
    {
    if (port != PORT_NULL)
	port_deallocate( task_self(), port );
    return [super free];
    }


//-----------------------------------------------------------------------------
// sendMailTo:subject:
//	Open a compose window in the GUI mailer program and fill in the To:
//	and Subject: fields.
//-----------------------------------------------------------------------------
- (BOOL)sendMailTo:(char const*)to subject:(char const*)subject
    {
    int composer;
    return (port != PORT_NULL &&
	[self selectorRPC:"openSend:" paramTypes:"I", &composer]==0 &&
	[self selectorRPC:"setTo:inWindow:" paramTypes:"ci",to,composer]==0 &&
	[self selectorRPC:"setSubject:inWindow:" paramTypes:"ci",
	    subject, composer]==0);
    }

@end


//=============================================================================
// soko_dispatch_url
//	Figure out the correct mechanism for dispatching a particular URL
//	and invoke that mechanism.
//
// *SERVICE*
//	The pasteboard is cached, rather than being destroyed after invoking
//	NXPerformService(), since the program to which the service request is
//	sent may not be able to handle the request immediately.  The pasteboard
//	must still exist by the time the receiving application finally gets
//	around to handling the service request.
//=============================================================================
BOOL soko_dispatch_url( char const* url )
    {
    BOOL ok = NO;
    char const mailto[] = "mailto:";
    if (soko_has_prefix( url, mailto, soko_true ))
	{
	SokoMailSpeaker* speaker;
	char subject[ 256 ];
	char const* to = url + sizeof(mailto) - 1;
	sprintf( subject,
	    "SokoSave feedback (release %s; build %s; NextStep/Mach)",
	    SOKO_RELEASE, SOKO_BUILD );
	speaker = [[SokoMailSpeaker alloc] init];
	ok = [speaker sendMailTo:to subject:subject];
	[speaker free];
	}
    else
	{
	NX_DURING
	    static Pasteboard* pb = 0;		// *SERVICE*
	    if (pb == 0)
		pb = [Pasteboard newUnique];
	    [pb declareTypes:&NXAsciiPboardType num:1 owner:0];
	    [pb writeType:NXAsciiPboardType data:url length:strlen(url) + 1];
	    ok = NXPerformService( SOKO_URL_SERVICE, pb );
	NX_HANDLER
	    NXRunAlertPanel( "SokoSave",
		"Communication error while dispatching URL: %s", 0,0,0, url );
	NX_ENDHANDLER
	}
    return ok;
    }
