//-----------------------------------------------------------------------------
// SokoScores.m
//
//	"Top-Scores" object for SokoSave.
//
// Copyright (c), 1997,2001,2002, Eric Sunshine <sunshine@sunshineco.com>
// Copyright (c), 1997, Paul McCarthy <zarnuk@high-speed-software.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoScores.m,v 1.6 2002/05/06 05:44:16 sunshine Exp $
// $Log: SokoScores.m,v $
// Revision 1.6  2002/05/06 05:44:16  sunshine
// v19
// -*- Fixed v15 bug: -[SokoScores formatInt:] was formatting number into one
//     buffer but then returning pointer to a different buffer.
//
// -*- Fixed v15 bug: SokoScores was reporting a "buffer count" of two to
//     MiscTableScroll, but -formatInt: only maintained two buffers, though
//     it really needed eight (two each for four numeric columns).  Rather
//     than allocating 512 bytes for these static buffers, which are very
//     rarely used, I instead eliminated the "buffer count" optimization
//     altogether.
//
// Revision 1.5  2002/01/29 20:08:57  sunshine
// v17
// -*- SokoPuzzle now calculates two new scores in addition to "moves" and
//     "pushes".  The "runs" score is the number of straight lines in which
//     crates have been pushed.  Looked at another way, it is the number of
//     turns crates have made while being pushed.  The "focus" score is the
//     number of times the player's focus has changed from one crate to
//     another.
//
// -*- Added "runs" and "focus" scores columns to scores panel and to "New
//     Score" panel.
//-----------------------------------------------------------------------------
#import "SokoPool.h"
#import "SokoScores.h"
#import "SokoAlerter.h"
#import "SokoFile.h"
#import "SokoSetting.h"
#import "SokoWindow.h"
#import <appkit/Application.h>
#import <appkit/Cell.h>
#import <appkit/Text.h>
#import <appkit/TextField.h>
#import <defaults/defaults.h>
#import <misckit/MiscTableCell.h>
#import <misckit/MiscTableScroll.h>
#import <objc/List.h>

enum
    {
    PUZZLE_COL,
    MOVE_COL,
    PUSH_COL,
    RUN_COL,
    FOCUS_COL,
    NAME_COL,
    DATE_COL,
    NOTE_COL,
    MAX_COL
    };

static struct _SokoScoreDelegate SCORE_DELEGATE;

static char const* get_def( char const* name )
    { return NXGetDefaultValue( "SokoSave", name ); }

static void set_def( char const* name, char const* value )
    { NXWriteDefault( "SokoSave", name, value ); }

static void score_alert( SokoScore score, SokoScoreDelegate delegate,
    SokoAlert severity, char const* title, char const* msg )
    { SokoSendAlert( severity, title, msg ); }


//=============================================================================
// IMPLEMENTATION
//=============================================================================
@implementation SokoScores

//-----------------------------------------------------------------------------
// MiscTableScroll delegate & data source methods
//-----------------------------------------------------------------------------
- (id)tableScroll:(MiscTableScroll*)ts border:(MiscBorderType)b
    slotResized:(int)n
    {
    char s[ 256 ];
    if ([tableScroll colSizesAsString:s size:sizeof(s) canExpand:NO] != 0)
	set_def( "ScoreColSizes", s );
    return self;
    }

- (void)saveColumnOrder
    {
    char s[ 256 ];
    if ([tableScroll colOrderAsString:s size:sizeof(s) canExpand:NO] != 0)
	set_def( "ScoreColOrder", s );
    }

- (id)tableScroll:(MiscTableScroll*)ts border:(MiscBorderType)b
    slotDraggedFrom:(int)from_pos to:(int)to_pos
    {
    [self saveColumnOrder];
    return self;
    }

- (id)tableScroll:(MiscTableScroll*)ts border:(MiscBorderType)b
    slotSortReversed:(int)n
    {
    [self saveColumnOrder];
    return self;
    }

- (char const*)formatInt:(int)i
    {
    static char buff[32];
    if (i >= 0)
	sprintf( buff, "%d", i );
    else
	buff[0] = '\0';
    return buff;
    }

- (char const*)tableScroll:(MiscTableScroll*)ts stringValueAt:(int)row:(int)col
    {
    char const* s = "";
    SokoHighScore const* r = SokoScore_get_score( scores, row );
    switch (col)
	{
	case PUZZLE_COL: s = r->puzzle;                  break;
	case MOVE_COL:   s = [self formatInt:r->moves ]; break;
	case PUSH_COL:   s = [self formatInt:r->pushes]; break;
	case RUN_COL:    s = [self formatInt:r->runs  ]; break;
	case FOCUS_COL:  s = [self formatInt:r->focus ]; break;
	case NAME_COL:   s = r->player;                  break;
	case DATE_COL:   s = r->date_formatted;          break;
	case NOTE_COL:   s = r->notes;                   break;
	}
    return s;
    }

- (int)tableScroll:(MiscTableScroll*)ts tagAt:(int)row:(int)col
    {
    int n = 0;
    SokoHighScore const* r = SokoScore_get_score( scores, row );
    switch (col)
	{
	case MOVE_COL:  n = r->moves;  break;
	case PUSH_COL:  n = r->pushes; break;
	case RUN_COL:   n = r->runs;   break;
	case FOCUS_COL: n = r->focus;  break;
	case DATE_COL:  n = r->date;   break;
	}
    return n;
    }

- (id)tableScroll:(MiscTableScroll*)ts cellAt:(int)row :(int)col
    {
    id cell = [cells objectAt:col];
    SokoHighScore const* r = SokoScore_get_score( scores, row );
    switch (col)
	{
	case PUZZLE_COL:
	    [cell setStringValueNoCopy:r->puzzle];
	    break;
	case MOVE_COL:
	    [cell setIntValue:r->moves];
	    [cell setTag:r->moves];
	    break;
	case PUSH_COL:
	    [cell setIntValue:r->pushes];
	    [cell setTag:r->pushes];
	    break;
	case RUN_COL:
	    if (r->runs >= 0)
		[cell setIntValue:r->runs];
	    else
		[cell setStringValue:""];
	    [cell setTag:r->runs];
	    break;
	case FOCUS_COL:
	    if (r->focus >= 0)
		[cell setIntValue:r->focus];
	    else
		[cell setStringValue:""];
	    [cell setTag:r->focus];
	    break;
	case NAME_COL:
	    [cell setStringValueNoCopy:r->player];
	    break;
	case DATE_COL:
	    [cell setStringValueNoCopy:r->date_formatted];
	    [cell setTag:r->date];
	    break;
	case NOTE_COL:
	    [cell setStringValueNoCopy:r->notes];
	    break;
	}
    return cell;
    }


//-----------------------------------------------------------------------------
// configureCells
//-----------------------------------------------------------------------------
- (void)configureCells
    {
    int i;
    NXZone* z = [cells zone];
    [cells setAvailableCapacity:MAX_COL];
    for (i = 0; i < MAX_COL; i++)
	{
	id cell = [[tableScroll colCellPrototype:i] copyFromZone:z];
	if ([cell respondsTo:@selector(setOwner:)])
	    [cell setOwner:tableScroll];
	[cells addObject:cell];
	}
    }


//-----------------------------------------------------------------------------
// init
//-----------------------------------------------------------------------------
- (id)init
    {
    char const* s;
    [super init];

    SCORE_DELEGATE.alert_callback = score_alert;
    SCORE_DELEGATE.info = 0;
    scores = SokoScore_new( &SCORE_DELEGATE, 0 );

    [NXApp loadNibSection:"SokoScores.nib" owner:self withNames:NO];
    [panel setMiniwindowIcon:"SokoSave"];
    [panel setFrameAutosaveName:"SokoScore"];

    cells = [[List allocFromZone:[self zone]] init];
    [self configureCells];

    if ((s = get_def( "ScoreColSizes" )) != 0)
	[tableScroll setColSizesFromString:s];
    if ((s = get_def( "ScoreColOrder" )) != 0)
	[tableScroll setColOrderFromString:s];

    [tableScroll renewRows:SokoScore_score_count( scores )];
    [tableScroll sortRows];
    return self;
    }


//-----------------------------------------------------------------------------
// free
//-----------------------------------------------------------------------------
- (id)free
    {
    [panel setDelegate:0];
    [panel close];
    [panel free];
    [cells freeObjects];
    [cells free];
    SokoScore_destroy( scores );
    return [super free];
    }


//-----------------------------------------------------------------------------
// activate
//-----------------------------------------------------------------------------
- (void)activate
    {
    [panel makeKeyAndOrderFront:self];
    }


//-----------------------------------------------------------------------------
// print:
//-----------------------------------------------------------------------------
- (id)print:(id)sender
    {
    [tableScroll printPSCode:self];
    return self;
    }


//-----------------------------------------------------------------------------
// solved:moves:pushes:runs:focus:name:notes:
//-----------------------------------------------------------------------------
- (void)solved:(SokoPuzzle)puzzle
    moves:(int)moves
    pushes:(int)pushes
    runs:(int)runs
    focus:(int)focus
    name:(char const*)name
    notes:(char const*)notes
    {
    int const row =
	SokoScore_add(scores, puzzle, moves, pushes, runs, focus, name, notes);
    [panel disableFlushWindow];
    [tableScroll renewRows:SokoScore_score_count( scores )];
    [tableScroll sortRows];
    [tableScroll selectRow:row];
    [tableScroll setCursorRow:row];
    [tableScroll scrollRowToVisible:row];
    [tableScroll display];
    [panel reenableFlushWindow];
    [panel flushWindow];
    }


//-----------------------------------------------------------------------------
// globalInstance
//-----------------------------------------------------------------------------
+ (SokoScores*)globalInstance
    {
    static SokoScores* instance = 0;
    if (instance == 0)
	instance = [[self alloc] init];
    return instance;
    }


//-----------------------------------------------------------------------------
// launch
//-----------------------------------------------------------------------------
+ (void)launch
    {
    [[self globalInstance] activate];
    }


//-----------------------------------------------------------------------------
// solved:moves:pushes:runs:focus:name:notes:
//-----------------------------------------------------------------------------
+ (void)solved:(SokoPuzzle)puzzle
    moves:(int)moves
    pushes:(int)pushes
    runs:(int)runs
    focus:(int)focus
    name:(char const*)name
    notes:(char const*)notes
    {
    SokoScores* p = [self globalInstance];
    [p solved:puzzle
	moves:moves pushes:pushes runs:runs focus:focus name:name notes:notes];
    [p activate];
    }

@end
