//-----------------------------------------------------------------------------
// SokoPref.m
//
//	User preferences module for SokoSave.
//
// Copyright (c), 1997,2001, Eric Sunshine <sunshine@sunshineco.com>
// Copyright (c), 1997, Paul McCarthy <zarnuk@high-speed-software.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoPref.m,v 1.4 2001/12/23 17:54:32 sunshine Exp $
// $Log: SokoPref.m,v $
// Revision 1.4  2001/12/23 17:54:32  sunshine
// v15
// -*- Extracted core game logic out of GUI code and generalized it so that
//     the same core code can be used by any platform.  Logic from
//     SokoBoard.m now resides in SokoPuzzle.c, etc.
//
// -*- Rewrote the NextStep port so that it utilizes the new "sokocore" game
//     library rather than implementing that logic directly.
//
// -*- Replaced the terminology "maze" with "puzzle" throughout the project,
//     including source code, documentation, and all user-visible UI
//     elements.  The only remaining place where "maze" is still used is in
//     the file extension ".sokomaze".  I haven't decided what, if anything,
//     to do about that, yet.
//
// -*- Added the new pseudo-variable $(SokoUser) which points at the user's
//     "personal" file space.  This is where user-specific SokoSave files are
//     stored by default.  This variable complements the existing $(SokoSave)
//     variable.
//
// -*- Renamed soko_collapse() to soko_collapse_path().  Renamed
//     soko_expand() to soko_expand_path().
//
// -*- The path setting functions in SokoFile now invoke soko_collapse_path()
//     on all set operations.  This ensures that all paths displayed on the
//     preferences panel are properly collapsed, and removes the onus of this
//     task from the UI code.  Previously, soko_collapse_path() was only
//     called by the UI code for the puzzle path.
//
// -*- Added soko_normalize_path() and soko_denormalize_path() to SokoUtil.
//     All path manipulation functions now utilize these functions in order
//     to ensure correctness of path parsing and composition on all
//     platforms.
//
// -*- Added soko_set_puzzle_directory(), soko_set_save_directory(), and
//     soko_set_score_file() to SokoFile in order to complement the existing
//     "get" functions and to centralize control over these settings.
//
// -*- Added soko_get_default_puzzle_directory(),
//     get_default_save_directory(), and soko_get_default_score_file() to
//     SokoFile.  These functions return values appropriate for the
//     corresponding fields on the Preferences panel of each port when the
//     "Defaults" button is pressed.
//
// -*- Added SokoSetting implementation which provides a platform-independent
//     API for accessing user settings and well-known paths, such as
//     $(SokoSave) and $(SokoUser).
//
// -*- For safety, user defaults (including score panel column order and
//     widths) are now committed immediately, rather than at termination
//     time.
//
// -*- Added SokoWindow and SokoPanel which are subclasses of Window and
//     Panel, respectively.  All windows and panels throughout the
//     application are now instances of these classes.  Although these
//     classes currently implement no extra functionality, they provide
//     convenient hooks for the addition of new functionality in the future,
//     if needed.
//
// -*- Renamed "window" instance variable to "panel" for all classes and nibs
//     in which a panel is actually used (everything but SokoBoard, in fact).
//
// Revision 1.3  2001/08/19 10:00:40  sunshine
// v11
// -*- Converted from Objective-C++ to pure Objective-C.  Changed file
//     extensions from .M to .m, and .cc to .c.  This change makes it easier
//     to support both GNUstep and MacOS/X, neither of which feature an
//     Objective-C++ compiler (though MacOS/X will reportedly support
//     Objective-C++ at some point in the future).
//
// -*- Eliminated all inclusions of <appkit/appkit.h> throughout the project;
//     replaced with inclusion of individual header files.
//
// -*- Upgraded formatting of source and header files which I had not updated
//     in the last version.  Added explicit (id) in several places which had
//     not previously been updated.
//
// -*- Renamed the following functions in order to avoid potential symbolic
//     collisions in the target environment:
//
//     basename --> soko_basename
//     collapse --> soko_collapse
//     directoryPart --> soko_directory_part
//     ends_with --> soko_ends_with
//     expand --> soko_expand
//     filenamePart --> soko_filename_part
//     getFactoryScoresFile --> soko_get_factory_scores_file
//     getMazeDirectory --> soko_get_maze_directory
//     getMazeExtension --> soko_get_maze_extension
//     getSaveDirectory --> soko_get_save_directory
//     getSaveExtension --> soko_get_save_extension
//     getScoresFile --> soko_get_scores_file
//     load_nib --> soko_load_nib
//     mazeNameForLevel --> soko_maze_name_for_level
//     mkdirs --> soko_mkdirs
//     runLengthDecodeChar --> soko_run_length_decode_char
//     runLengthDecodeString --> soko_run_length_decode_string
//     runLengthEncodeChar --> soko_run_length_encode_char
//     runLengthEncodeString --> soko_run_length_encode_string
//     saveFileNameForMazeName --> soko_save_filename_for_maze_name
//     starts_with --> soko_starts_with
//     strdup --> soko_strdup
//-----------------------------------------------------------------------------
#import	"SokoPool.h"
#import "SokoPref.h"
#import "SokoFile.h"
#import "SokoUtil.h"
#import "SokoWindow.h"
#import <appkit/Application.h>
#import <appkit/Button.h>
#import <appkit/TextField.h>
#import <defaults/defaults.h>
#import <objc/NXBundle.h>
#import <string.h>

#define DEF_OWNER "SokoSave"

static SokoPref* COMMON_INSTANCE = 0;
static struct _SokoSettingDelegate SETTING_DELEGATE;

#define DEL_FUNC_0(RET,NAME) \
    static RET NAME( SokoSetting x1, SokoSettingDelegate x2 )
#define DEL_FUNC_1(RET,NAME,T1,A1) \
    static RET NAME( SokoSetting x1, SokoSettingDelegate x2, T1 A1 )
#define DEL_FUNC_2(RET,NAME,T1,A1,T2,A2) \
    static RET NAME( SokoSetting x1, SokoSettingDelegate x2, T1 A1, T2 A2 )

DEL_FUNC_1(soko_bool, setting_exists, char const*,name)
    {
    return (NXGetDefaultValue( DEF_OWNER, name ) != 0);
    }

DEL_FUNC_2(char const*, get_setting, char const*,name, cSokoPool,pool)
    {
    return NXGetDefaultValue( DEF_OWNER, name );
    }

DEL_FUNC_2(void, set_setting, char const*,name, char const*,value)
    {
    NXWriteDefault( DEF_OWNER, name, value );
    }

DEL_FUNC_1(void, remove_setting, char const*,name)
    {
    if (NXGetDefaultValue( DEF_OWNER, name ) != 0)
	NXRemoveDefault( DEF_OWNER, name );
    }

DEL_FUNC_1(char const*, factory_path, cSokoPool,pool)
    {
    static char* path = 0;
    if (path == 0)
	path = soko_strdup( [[NXBundle mainBundle] directory] );
    return path;
    }

DEL_FUNC_1(char const*, user_path, cSokoPool,pool)
    {
    static char* path = 0;
    if (path == 0)
	{
	char const* home = NXHomeDirectory();
	char const local[] = "/Library/SokoSave";
	path = (char*)malloc( strlen(home) + sizeof(local) );
	strcpy( path, home );
	strcat( path, local );
	}
    return path;
    }

#undef DEL_FUNC_2
#undef DEL_FUNC_1
#undef DEL_FUNC_0
#undef STD_DEF

//=============================================================================
// SokoPref Implementation
//=============================================================================
@implementation SokoPref

//-----------------------------------------------------------------------------
// loadDefaults
//-----------------------------------------------------------------------------
- (void)loadDefaults
    {
    SokoPool pool = SokoPool_new(0);
    [saveDirField setStringValue:soko_get_default_save_directory(pool)];
    [autoSaveSwitch setState:1];
    [scoresFileField setStringValue:soko_get_default_score_file(pool)];
    [puzzleDirField setStringValue:soko_get_default_puzzle_directory(pool)];
    [levelField setIntValue:1];
    SokoPool_destroy( pool );
    }


//-----------------------------------------------------------------------------
// loadSettings
//	*NOTE* Ensure that editing is not in progress, since (under some AppKit
//	versions) setting a field while it is being edited is a no-op.
//-----------------------------------------------------------------------------
- (void)loadSettings
    {
    SokoPool pool = SokoPool_new(0);
    [panel makeFirstResponder:panel];	// *NOTE*
    [saveDirField setStringValue:soko_get_save_directory( pool )];
    [autoSaveSwitch setState:
	SokoSetting_get_bool( setting, SOKO_SETTING_AUTO_SAVE, soko_true )];
    [scoresFileField setStringValue:soko_get_score_file( pool )];
    [puzzleDirField setStringValue:soko_get_puzzle_directory( pool )];
    [levelField setIntValue:SokoSetting_get_int(setting,SOKO_SETTING_LEVEL,1)];
    [panel setDocEdited:NO];
    SokoPool_destroy( pool );
    }


//-----------------------------------------------------------------------------
// saveSettings
//	*POST-SAVE* Load settings in case some settings were rejected by "save"
//	logic.  Loading settings also clears "dirty" flag as a side-effect.
//-----------------------------------------------------------------------------
- (void)saveSettings
    {
    if ([panel makeFirstResponder:panel])
	{
	SokoPool pool = SokoPool_new(0);
	int level;
	char const* s;
	[panel endEditingFor:0];
	s = [saveDirField stringValue];
	if (s != 0 && *s != '\0')
	    soko_set_save_directory( pool, s );
	SokoSetting_set_bool(
	    setting, SOKO_SETTING_AUTO_SAVE, [autoSaveSwitch state] );
	s = [scoresFileField stringValue];
	if (s != 0 && *s != '\0')
	    soko_set_score_file( pool, s );
	s = [puzzleDirField stringValue];
	if (s != 0 && *s != '\0')
	    soko_set_puzzle_directory( pool, s );
	level = [levelField intValue];
	if (level >= 1)
	    SokoSetting_set_int( setting, SOKO_SETTING_LEVEL, level );
	[self loadSettings];	// *POST-SAVE*
	SokoPool_destroy( pool );
	}
    }


//-----------------------------------------------------------------------------
// madeDirty:
//-----------------------------------------------------------------------------
- (id)madeDirty:(id)sender
    {
    [panel setDocEdited:YES];
    return self;
    }


//-----------------------------------------------------------------------------
// doSet:
//-----------------------------------------------------------------------------
- (id)doSet:(id)sender
    {
    [panel close];
    [self saveSettings];
    return self;
    }


//-----------------------------------------------------------------------------
// doCancel:
//-----------------------------------------------------------------------------
- (id)doCancel:(id)sender
    {
    [panel close];
    [self loadSettings];
    return self;
    }


//-----------------------------------------------------------------------------
// doDefaults:
//-----------------------------------------------------------------------------
- (id)doDefaults:(id)sender
    {
    [self loadDefaults];
    [panel setDocEdited:YES];
    [saveDirField selectText:self];
    return self;
    }


//-----------------------------------------------------------------------------
// textDidChange:
//-----------------------------------------------------------------------------
- (id)textDidChange:(id)sender
    {
    [self madeDirty:self];
    return self;
    }


//-----------------------------------------------------------------------------
// windowWillClose:
//-----------------------------------------------------------------------------
- (id)windowWillClose:(id)sender
    {
    BOOL ok = YES;
    if ([panel isDocEdited])
	{
	int const rc = NXRunAlertPanel( "Preferences",
	    "Save your changes to the preferences?",
	    "Save", "Don't Save", "Cancel" );
	if (rc == NX_ALERTDEFAULT)
	    [self saveSettings];
	else if (rc == NX_ALERTALTERNATE)
	    [self loadSettings];
	else // (rc == NX_ALERTOTHER)
	    ok = NO;
	}
    return (ok ? self : 0);
    }


//-----------------------------------------------------------------------------
// setNextLevel:
//-----------------------------------------------------------------------------
- (void)setNextLevel:(int)level
    {
    [levelField setIntValue:level];
    }


//-----------------------------------------------------------------------------
// activate
//-----------------------------------------------------------------------------
- (void)activate
    {
    id fr = [panel firstResponder];
    if (fr == 0 || fr == panel)
	[saveDirField selectText:self];
    [panel makeKeyAndOrderFront:self];
    }


//-----------------------------------------------------------------------------
// init
//-----------------------------------------------------------------------------
- (id)init
    {
    [super init];
    [NXApp loadNibSection:"SokoPref.nib" owner:self withNames:NO];
    [panel setFrameAutosaveName:"SokoPref"];
    setting = SokoSetting_new(0);
    [self loadSettings];
    return self;
    }


//-----------------------------------------------------------------------------
// free
//-----------------------------------------------------------------------------
- (id)free
    {
    [panel setDelegate:0];
    [panel close];
    [panel free];
    SokoSetting_destroy( setting );
    return [super free];
    }


//-----------------------------------------------------------------------------
// startup
//-----------------------------------------------------------------------------
+ (void)startup
    {
    SETTING_DELEGATE.exists_callback = setting_exists;
    SETTING_DELEGATE.get_callback = get_setting;
    SETTING_DELEGATE.set_callback = set_setting;
    SETTING_DELEGATE.remove_callback = remove_setting;
    SETTING_DELEGATE.factory_path_callback = factory_path;
    SETTING_DELEGATE.user_path_callback = user_path;
    SETTING_DELEGATE.info = 0;
    SokoSetting_set_delegate( &SETTING_DELEGATE );
    }


//-----------------------------------------------------------------------------
// shutdown
//-----------------------------------------------------------------------------
+ (void)shutdown
    {
    if (COMMON_INSTANCE != 0)
	{
	[COMMON_INSTANCE free];
	COMMON_INSTANCE = 0;
	}
    SokoSetting_set_delegate(0);
    }


//-----------------------------------------------------------------------------
// getNextLevel
//-----------------------------------------------------------------------------
+ (int)getNextLevel
    {
    SokoSetting s = SokoSetting_new(0);
    int level = SokoSetting_get_int( s, SOKO_SETTING_LEVEL, 1 );
    SokoSetting_destroy(s);
    return level;
    }


//-----------------------------------------------------------------------------
// setNextLevel:
//	*NOTE* If the preference panel has been instantiated, then also set the
//	value of the "next level" field.  This works correctly whether or not
//	the panel is currently visible.  Even if it is visible, just setting
//	the value of the field is sufficient; there is no need to set the
//	"dirty" flag (since the new setting is already saved).
//-----------------------------------------------------------------------------
+ (void)setNextLevel:(int)n
    {
    if (n >= 0)
	{
	SokoSetting s = SokoSetting_new(0);
	SokoSetting_set_int( s, SOKO_SETTING_LEVEL, n );
	if (COMMON_INSTANCE != 0)
	    [COMMON_INSTANCE setNextLevel:n];
	SokoSetting_destroy(s);
	}
    }


//-----------------------------------------------------------------------------
// launch
//-----------------------------------------------------------------------------
+ (void)launch
    {
    if (COMMON_INSTANCE == 0)
	COMMON_INSTANCE = [[self alloc] init];
    [COMMON_INSTANCE activate];
    }

@end
