//-----------------------------------------------------------------------------
// SokoNewScore.m
//
//	Panel allowing user input upon new high-score for SokoSave.
//
// Copyright (c), 2001,2002, Eric Sunshine <sunshine@sunshineco.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoNewScore.m,v 1.2 2002/01/29 20:07:49 sunshine Exp $
// $Log: SokoNewScore.m,v $
// Revision 1.2  2002/01/29 20:07:49  sunshine
// v17
// -*- SokoPuzzle now calculates two new scores in addition to "moves" and
//     "pushes".  The "runs" score is the number of straight lines in which
//     crates have been pushed.  Looked at another way, it is the number of
//     turns crates have made while being pushed.  The "focus" score is the
//     number of times the player's focus has changed from one crate to
//     another.
//
// -*- Added "runs" and "focus" scores columns to scores panel and to "New
//     Score" panel.
//
// -*- Now strips the file extension from the puzzle name on the new-score
//     panel.
//
// Revision 1.1  2001/12/23 16:33:39  sunshine
// v15
// -*- Extracted core game logic out of GUI code and generalized it so that
//     the same core code can be used by any platform.  Logic from
//     SokoBoard.m now resides in SokoPuzzle.c, etc.
//
// -*- Rewrote the NextStep port so that it utilizes the new "sokocore" game
//     library rather than implementing that logic directly.
//
// -*- Split new-score entry panel and functionality out of SokoScores and
//     into its own module, SokoNewScore.
//
// -*- Now remembers username from New Score panel as default for that panel
//     as a convenience to users rather than always seeding it with the
//     username.
//
// -*- Replaced the terminology "maze" with "puzzle" throughout the project,
//     including source code, documentation, and all user-visible UI
//     elements.  The only remaining place where "maze" is still used is in
//     the file extension ".sokomaze".  I haven't decided what, if anything,
//     to do about that, yet.
//
// -*- Added SokoWindow and SokoPanel which are subclasses of Window and
//     Panel, respectively.  All windows and panels throughout the
//     application are now instances of these classes.  Although these
//     classes currently implement no extra functionality, they provide
//     convenient hooks for the addition of new functionality in the future,
//     if needed.
//
// -*- Renamed "window" instance variable to "panel" for all classes and nibs
//     in which a panel is actually used (everything but SokoBoard, in fact).
//
// -*- Added SokoSetting implementation which provides a platform-independent
//     API for accessing user settings and well-known paths, such as
//     $(SokoSave) and $(SokoUser).
//-----------------------------------------------------------------------------
#import "SokoPool.h"
#import "SokoNewScore.h"
#import "SokoScores.h"
#import "SokoSetting.h"
#import "SokoUtil.h"
#import "SokoWindow.h"
#import <appkit/Application.h>
#import <appkit/Button.h>
#import <appkit/TextField.h>
#import <libc.h>
#import <string.h>

#define SOKO_PLAYER_NAME "PlayerName"

//=============================================================================
// SokoNewScore
//=============================================================================
@implementation SokoNewScore

- (id)okayPressed:(id)sender { [NXApp stopModal]; return self; }

//-----------------------------------------------------------------------------
// init
//-----------------------------------------------------------------------------
- (id)init
    {
    SokoPool pool = SokoPool_new(0);
    SokoSetting setting = SokoSetting_new(0);
    storedName = soko_strdup(
	SokoSetting_get_string(setting, SOKO_PLAYER_NAME, "", pool) );
    SokoSetting_destroy( setting );
    SokoPool_destroy( pool );

    [super init];
    [NXApp loadNibSection:"SokoNewScore.nib" owner:self withNames:NO];
    [panel setFrameAutosaveName:"SokoNewScore"];
    if (storedName == 0 || *storedName == '\0')
	[nameField setStringValue:NXUserName()];
    else
	[nameField setStringValue:storedName];
    return self;
    }


//-----------------------------------------------------------------------------
// free
//-----------------------------------------------------------------------------
- (id)free
    {
    [panel setDelegate:0];
    [panel close];
    [panel free];
    free( storedName );
    return [super free];
    }


//-----------------------------------------------------------------------------
// refreshControls
//-----------------------------------------------------------------------------
- (void)refreshControls
    {
    SokoPool pool = SokoPool_new(0);
    char const* name = [nameField stringValue];
    BOOL badName = (strlen(name) == 0);
    if (!badName)
	badName = (strlen(soko_collapse(pool, name)) == 0);
    if (badName == [okayButton isEnabled])
	[okayButton setEnabled:!badName];
    SokoPool_destroy( pool );
    }


//-----------------------------------------------------------------------------
// textDidGetKeys:isEmpty:
//-----------------------------------------------------------------------------
- (id)textDidGetKeys:(id)sender isEmpty:(BOOL)empty
    {
    if (nameField == [sender delegate])
	[self refreshControls];
    return self;
    }


//-----------------------------------------------------------------------------
// windowWillResize:toSize:
//-----------------------------------------------------------------------------
- (id)windowWillResize:(id)sender toSize:(NXSize*)size
    {
    NXRect r;
    [panel getFrame:&r];
    size->height = r.size.height;
    return self;
    }


//-----------------------------------------------------------------------------
// populateWithPuzzle:moves:pushes:runs:focus:
//-----------------------------------------------------------------------------
- (void)populateWithPuzzle:(SokoPuzzle) p moves:(int)moves pushes:(int)pushes
    runs:(int)runs focus:(int)focus
    {
    SokoPool pool = SokoPool_new(0);
    [puzzleField setStringValue:
	soko_remove_path_extension(pool, soko_filename_part(pool,
	SokoPuzzle_get_puzzle_file_name(p, pool)))];
    [movesField setIntValue:moves];
    [pushesField setIntValue:pushes];
    [runsField setIntValue:runs];
    [focusField setIntValue:focus];
    [notesField setStringValue:""];
    [self refreshControls];
    [nameField selectText:self];
    SokoPool_destroy( pool );
    }


//-----------------------------------------------------------------------------
// normalizePlayerName
//-----------------------------------------------------------------------------
- (void)normalizePlayerName
    {
    SokoPool pool = SokoPool_new(0);
    char const* name = soko_collapse( pool, [nameField stringValue] );
    [nameField setStringValue:name];
    if (storedName == 0 || strcmp( name, storedName ) != 0)
	{
	SokoSetting setting = SokoSetting_new(0);
	SokoSetting_set_string( setting, SOKO_PLAYER_NAME, name );
	SokoSetting_destroy( setting );
	free( storedName );
	storedName = soko_strdup( name );
	}
    }


//-----------------------------------------------------------------------------
// solved:moves:pushes:runs:focus:
//-----------------------------------------------------------------------------
- (void)solved:(SokoPuzzle)puzzle moves:(int)moves pushes:(int)pushes
    runs:(int)runs focus:(int)focus
    {
    [self populateWithPuzzle:puzzle
	moves:moves pushes:pushes runs:runs focus:focus];
    [panel makeKeyAndOrderFront:self];
    [NXApp runModalFor:panel];
    [panel close];
    [self normalizePlayerName];
    [SokoScores solved:puzzle moves:moves pushes:pushes runs:runs focus:focus
	name:[nameField stringValue] notes:[notesField stringValue]];
    }


//-----------------------------------------------------------------------------
// solved:moves:pushes:runs:focus:
//-----------------------------------------------------------------------------
+ (void)solved:(SokoPuzzle)puzzle moves:(int)moves pushes:(int)pushes
    runs:(int)runs focus:(int)focus
    {
    static SokoNewScore* instance = 0;
    if (instance == 0)
	instance = [[SokoNewScore alloc] init];
    [instance solved:puzzle moves:moves pushes:pushes runs:runs focus:focus];
    }

@end
