//-----------------------------------------------------------------------------
// SokoMatrixTriangle.m
//
//	A hexagonal-celled matrix for SokoSave.
//
// Copyright (c), 2002, Eric Sunshine <sunshine@sunshineco.com>
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoMatrixTriangle.m,v 1.1 2002/02/19 08:57:04 sunshine Exp $
// $Log: SokoMatrixTriangle.m,v $
// Revision 1.1  2002/02/19 08:57:04  sunshine
// v18
// -*- Added support for new triangular-style Trioban puzzles.
//
// -*- Consolidated the user-interface movement constraints into SokoPuzzle
//     rather than having the same code repeated in each port.  This code
//     decides whether or not the direction movement buttons in the
//     user-interface should be enabled depending upon cell type,
//     orientation, etc.  Added the SokoPuzzle methods can_move_constrained()
//     and can_move_diagonal_constrained() to complement the existing
//     can_move() and can_move_diagonal().
//
//-----------------------------------------------------------------------------
#import "SokoPool.h"
#import "SokoMatrixTriangle.h"
#import "SokoGeometry.h"
#import <appkit/NXImage.h>

#define CELL_WIDTH      (24)
#define CELL_HEIGHT     (22)
#define CELL_HALF_WIDTH (CELL_WIDTH / 2)

static char const* const CELL_NAMES[ SOKO_CELL_MAX * 2 ] =
    {
    "SokoEmptyTriS",		 // SOKO_CELL_EMPTY
    "SokoEmptyTriN",		 // SOKO_CELL_EMPTY
    "SokoEmptySafeTriS",	 // SOKO_CELL_EMPTY_SAFE
    "SokoEmptySafeTriN",	 // SOKO_CELL_EMPTY_SAFE
    "SokoWallTriS",		 // SOKO_CELL_WALL
    "SokoWallTriN",		 // SOKO_CELL_WALL
    "SokoPlayerTriS",		 // SOKO_CELL_PLAYER
    "SokoPlayerTriN",		 // SOKO_CELL_PLAYER
    "SokoPlayerSafeTriS",	 // SOKO_CELL_PLAYER_SAFE
    "SokoPlayerSafeTriN",	 // SOKO_CELL_PLAYER_SAFE
    "SokoCrateTriS",		 // SOKO_CELL_CRATE
    "SokoCrateTriN",		 // SOKO_CELL_CRATE
    "SokoCrateSafeTriS",	 // SOKO_CELL_CRATE_SAFE
    "SokoCrateSafeTriN",	 // SOKO_CELL_CRATE_SAFE
    "SokoCrateSelectedTriS",	 // SOKO_CELL_CRATE_SELECTED
    "SokoCrateSelectedTriN",	 // SOKO_CELL_CRATE_SELECTED
    "SokoCrateSafeSelectedTriS", // SOKO_CELL_CRATE_SELECTED_SAFE
    "SokoCrateSafeSelectedTriN", // SOKO_CELL_CRATE_SELECTED_SAFE
    0,				 // SOKO_CELL_NULL
    0                            // SOKO_CELL_NULL
    };

@implementation SokoMatrixTriangle

//-----------------------------------------------------------------------------
// imageForCellType:withRow:column:
//-----------------------------------------------------------------------------
+ (NXImage*)imageForCellType:(SokoCellType)type withRow:(int)r column:(int)c;
    {
    int const north_south = (r & 1) ^ (c & 1);
    char const* n = CELL_NAMES[ type * 2 + north_south ];
    return (n == 0 ? 0 : [NXImage findImageNamed:n]);
    }


//-----------------------------------------------------------------------------
// initWithRows:columns:
//-----------------------------------------------------------------------------
- (id)initWithRows:(int)r columns:(int)c
    {
    [super initWithRows:r columns:c];
    [self setClipping:NO];
    [self sizeTo:(c + 1) * CELL_HALF_WIDTH:r * CELL_HEIGHT];
    return self;
    }


//-----------------------------------------------------------------------------
// drawSelf::
//-----------------------------------------------------------------------------
- (id)drawSelf:(NXRect const*)f :(int)n
    {
    int r, c, rs, cs, nr, nc, rlim, clim;
    NXRect b;
    [self getBounds:&b];
    soko_coverage_triangle( (int)NX_X(f), (int)NX_Y(f), (int)NX_WIDTH(f),
	(int)NX_HEIGHT(f), rows, cols, CELL_WIDTH, CELL_HEIGHT,
	CELL_HALF_WIDTH, NX_X(&b), NX_Y(&b), &rs, &cs, &nr, &nc );
    for (r = rs, rlim = rs + nr; r < rlim; r++)
	for (c = cs, clim = cs + nc; c < clim; c++)
	    [self drawCellAtRow:r column:c];
    return self;
    }


//-----------------------------------------------------------------------------
// drawCellAtRow:column:
//-----------------------------------------------------------------------------
- (void)drawCellAtRow:(int)r column:(int)c
    {
    NXImage* i = [self imageAtRow:r column:c];
    if (i != 0)
	{
	NXPoint p;
	p.x = c * CELL_HALF_WIDTH;
	p.y = r * CELL_HEIGHT + ([self isFlipped] ? CELL_HEIGHT : 0);
	[i composite:NX_SOVER toPoint:&p];
	}
    }


//-----------------------------------------------------------------------------
// getRow:column:forPoint:
//-----------------------------------------------------------------------------
- (BOOL)getRow:(int*)r column:(int*)c forPoint:(NXPoint const*)p
    {
    NXRect b;
    [self getBounds:&b];
    return soko_hit_test_triangle( (int)p->x, (int)p->y, rows, cols,
	CELL_WIDTH, CELL_HEIGHT, CELL_HALF_WIDTH, NX_X(&b), NX_Y(&b), r, c );
    }

@end
