//-----------------------------------------------------------------------------
// SokoMatrixHexagon.m
//
//	A hexagonal-celled matrix for SokoSave.
//
// Copyright (c), 2002, Eric Sunshine <sunshine@sunshineco.com>
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoMatrixHexagon.m,v 1.2 2002/02/19 08:57:10 sunshine Exp $
// $Log: SokoMatrixHexagon.m,v $
// Revision 1.2  2002/02/19 08:57:10  sunshine
// v18
// -*- Added support for new triangular-style Trioban puzzles.
//
// -*- Consolidated the user-interface movement constraints into SokoPuzzle
//     rather than having the same code repeated in each port.  This code
//     decides whether or not the direction movement buttons in the
//     user-interface should be enabled depending upon cell type,
//     orientation, etc.  Added the SokoPuzzle methods can_move_constrained()
//     and can_move_diagonal_constrained() to complement the existing
//     can_move() and can_move_diagonal().
//
// Revision 1.1  2002/01/29 20:11:24  sunshine
// v17
// -*- Added support for the new hexagonal-style puzzles.
//
// -*- Rewrote SokoMatrix.  It is now a generic base class for rendering
//     grids.  No longer based upon Matrix (which was only capable of
//     supporting rectangular cells).  Custom rendering is now done, rather
//     than relying upon Matrix.  Consequently, animation speed has increased
//     by a factor of five or more.
//
// -*- Added SokoMatrixSquare, a subclass of SokoMatrix, which knows how to
//     draw square-tiled puzzles, and perform appropriate hit-testing.
//
// -*- Added SokoMatrixHexagon, a subclass of SokoMatrix, which knows how to
//     draw hexagon-tiled puzzles, and perform appropriate hit-testing.
//-----------------------------------------------------------------------------
#import "SokoPool.h"
#import "SokoMatrixHexagon.h"
#import "SokoGeometry.h"
#import <appkit/NXImage.h>

#define CELL_WIDTH   (18)
#define CELL_HEIGHT  (22)
#define CELL_X_INSET (9)
#define CELL_Y_INSET (6)

static char const* const CELL_NAMES[ SOKO_CELL_MAX ] =
    {
    "SokoEmptyHex",		// SOKO_CELL_EMPTY
    "SokoEmptySafeHex",		// SOKO_CELL_EMPTY_SAFE
    "SokoWallHex",		// SOKO_CELL_WALL
    "SokoPlayerHex",		// SOKO_CELL_PLAYER
    "SokoPlayerSafeHex",	// SOKO_CELL_PLAYER_SAFE
    "SokoCrateHex",		// SOKO_CELL_CRATE
    "SokoCrateSafeHex",		// SOKO_CELL_CRATE_SAFE
    "SokoCrateSelectedHex",	// SOKO_CELL_CRATE_SELECTED
    "SokoCrateSafeSelectedHex",	// SOKO_CELL_CRATE_SELECTED_SAFE
    0				// SOKO_CELL_NULL
    };

@implementation SokoMatrixHexagon

//-----------------------------------------------------------------------------
// imageForCellType:withRow:column:
//-----------------------------------------------------------------------------
+ (NXImage*)imageForCellType:(SokoCellType)type withRow:(int)r column:(int)c;
    {
    char const* n = CELL_NAMES[ type ];
    return (n == 0 ? 0 : [NXImage findImageNamed:n]);
    }


//-----------------------------------------------------------------------------
// initWithRows:columns:
//-----------------------------------------------------------------------------
- (id)initWithRows:(int)r columns:(int)c
    {
    [super initWithRows:r columns:c];
    [self setClipping:NO];
    [self sizeTo:(c * CELL_WIDTH) + (r <= 1 ? 0 : CELL_X_INSET):
		 r * (CELL_HEIGHT - CELL_Y_INSET) + CELL_Y_INSET];
    return self;
    }


//-----------------------------------------------------------------------------
// drawSelf::
//-----------------------------------------------------------------------------
- (id)drawSelf:(NXRect const*)f :(int)n
    {
    int r, c, rs, cs, nr, nc, rlim, clim;
    NXRect b;
    [self getBounds:&b];
    soko_coverage_hexagon( (int)NX_X(f), (int)NX_Y(f), (int)NX_WIDTH(f),
	(int)NX_HEIGHT(f), rows, cols, CELL_WIDTH, CELL_HEIGHT, CELL_X_INSET,
	CELL_Y_INSET, NX_X(&b), NX_Y(&b), &rs, &cs, &nr, &nc );
    for (r = rs, rlim = rs + nr; r < rlim; r++)
	for (c = cs, clim = cs + nc; c < clim; c++)
	    [self drawCellAtRow:r column:c];
    return self;
    }


//-----------------------------------------------------------------------------
// drawCellAtRow:column:
//-----------------------------------------------------------------------------
- (void)drawCellAtRow:(int)r column:(int)c
    {
    NXImage* i = [self imageAtRow:r column:c];
    if (i != 0)
	{
	NXPoint p;
	p.x = c * CELL_WIDTH + ((r & 1) == 0 ? 0 : CELL_X_INSET);
	p.y = r * (CELL_HEIGHT - CELL_Y_INSET) +
	    ([self isFlipped] ? CELL_HEIGHT : 0);
	[i composite:NX_SOVER toPoint:&p];
	}
    }


//-----------------------------------------------------------------------------
// getRow:column:forPoint:
//-----------------------------------------------------------------------------
- (BOOL)getRow:(int*)r column:(int*)c forPoint:(NXPoint const*)p
    {
    NXRect b;
    [self getBounds:&b];
    return soko_hit_test_hexagon( (int)p->x, (int)p->y, rows, cols, CELL_WIDTH,
	CELL_HEIGHT, CELL_X_INSET, CELL_Y_INSET, NX_X(&b), NX_Y(&b), r, c );
    }

@end
