//-----------------------------------------------------------------------------
// SokoMatrix.m
//
//	A custom Matrix for SokoSave.
//
// Copyright (c), 1997,2001,2002, Eric Sunshine <sunshine@sunshineco.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoMatrix.m,v 1.5 2002/02/19 08:56:59 sunshine Exp $
// $Log: SokoMatrix.m,v $
// Revision 1.5  2002/02/19 08:56:59  sunshine
// v18
// -*- Added support for new triangular-style Trioban puzzles.
//
// -*- Consolidated the user-interface movement constraints into SokoPuzzle
//     rather than having the same code repeated in each port.  This code
//     decides whether or not the direction movement buttons in the
//     user-interface should be enabled depending upon cell type,
//     orientation, etc.  Added the SokoPuzzle methods can_move_constrained()
//     and can_move_diagonal_constrained() to complement the existing
//     can_move() and can_move_diagonal().
//
// Revision 1.4  2002/01/29 20:11:28  sunshine
// v17
// -*- Added support for the new hexagonal-style puzzles.
//
// -*- Rewrote SokoMatrix.  It is now a generic base class for rendering
//     grids.  No longer based upon Matrix (which was only capable of
//     supporting rectangular cells).  Custom rendering is now done, rather
//     than relying upon Matrix.  Consequently, animation speed has increased
//     by a factor of five or more.
//
// -*- Added SokoMatrixSquare, a subclass of SokoMatrix, which knows how to
//     draw square-tiled puzzles, and perform appropriate hit-testing.
//
// -*- Added SokoMatrixHexagon, a subclass of SokoMatrix, which knows how to
//     draw hexagon-tiled puzzles, and perform appropriate hit-testing.
//-----------------------------------------------------------------------------
#import "SokoPool.h"
#import "SokoMatrix.h"
#import "SokoAssert.h"
#import <appkit/NXImage.h>
#import <stdlib.h>
#import <string.h>

@implementation SokoMatrix

//-----------------------------------------------------------------------------
// initWithRows:columns:
//-----------------------------------------------------------------------------
- (id)initWithRows:(int)r columns:(int)c
    {
    NXRect const f = {{ 0, 0 }, { 1, 1 }};
    int const sz = r * c * sizeof(matrix[0]);
    [super initFrame:&f];
    [self setFlipped:YES];
    rows = r;
    cols = c;
    matrix = (NXImage**)malloc(sz);
    memset( matrix, 0, sz );
    return self;
    }


//-----------------------------------------------------------------------------
// free
//-----------------------------------------------------------------------------
- (id)free
    {
    free( matrix );
    return [super free];
    }


//-----------------------------------------------------------------------------
// setCellType:atRow:column:
//-----------------------------------------------------------------------------
- (void)setCellType:(SokoCellType)type atRow:(int)r column:(int)c
    {
    SOKO_ASSERT( r >= 0 && r < rows );
    SOKO_ASSERT( c >= 0 && c < cols );
    matrix[ r * cols + c ] =
	[[self class] imageForCellType:type withRow:r column:c];
    if ([self canDraw])
	{
	[self lockFocus];
	[self drawCellAtRow:r column:c];
	[self unlockFocus];
	[[self window] flushWindow];
	}
    }


//-----------------------------------------------------------------------------
// imageAtRow:column:
//-----------------------------------------------------------------------------
- (NXImage*)imageAtRow:(int)r column:(int)c
    {
    SOKO_ASSERT( r >= 0 && r < rows );
    SOKO_ASSERT( c >= 0 && c < cols );
    return *(matrix + r * cols + c);
    }


//-----------------------------------------------------------------------------
// Stubs
//-----------------------------------------------------------------------------
- (BOOL)acceptsFirstResponder { return YES; }
- (BOOL)acceptsFirstMouse     { return YES; }

- (id)setDelegate:(id)d { delegate = d; return self; }
- (id)delegate { return delegate; }

+ (NXImage*)imageForCellType:(SokoCellType)type withRow:(int)r column:(int)c
    { [self subclassResponsibility:_cmd]; return 0; }
- (void)drawCellAtRow:(int)r column:(int)c
    { [self subclassResponsibility:_cmd]; }
- (BOOL)getRow:(int*)r column:(int*)c forPoint:(NXPoint const*)p
    { [self subclassResponsibility:_cmd]; return NO; }

#define FWD(M) - (id)M:(NXEvent*)e { return [delegate M:e]; }

FWD(mouseDown)
FWD(mouseUp)
FWD(mouseDragged)
FWD(rightMouseDown)
FWD(rightMouseUp)
FWD(rightMouseDragged)
FWD(keyDown)
FWD(keyUp)

#undef FWD

@end
