//-----------------------------------------------------------------------------
// SokoBoard.m
//
//	Implementation of a Sokoban puzzle board for SokoSave.
//
// Copyright (c), 1997,2001,2002, Eric Sunshine <sunshine@sunshineco.com>
// Copyright (c), 1997, Paul McCarthy <zarnuk@high-speed-software.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoBoard.m,v 1.9 2002/02/19 08:55:40 sunshine Exp $
// $Log: SokoBoard.m,v $
// Revision 1.9  2002/02/19 08:55:40  sunshine
// v18
// -*- Added support for new triangular-style Trioban puzzles.
//
// -*- Consolidated the user-interface movement constraints into SokoPuzzle
//     rather than having the same code repeated in each port.  This code
//     decides whether or not the direction movement buttons in the
//     user-interface should be enabled depending upon cell type,
//     orientation, etc.  Added the SokoPuzzle methods can_move_constrained()
//     and can_move_diagonal_constrained() to complement the existing
//     can_move() and can_move_diagonal().
//
// Revision 1.8  2002/01/29 20:14:51  sunshine
// v17
// -*- Added support for the new hexagonal-style puzzles.
//
// -*- No longer fills areas outside the puzzle with wall segments.  Doing so
//     causes some "artistic" effect to be lost from some puzzles.  Added
//     SokoCellType SOKO_CELL_NULL to support of this.
//
// -*- Added SokoPuzzleStyle enumeration to SokoPuzzle.  Styles are
//     SOKO_STYLE_SQUARE and SOKO_STYLE_HEXAGON.
//
// -*- Now recognize these additional file extensions: .xsb, .hsb, .sokohex.
//
// -*- Removed soko_get_puzzle_extension() and soko_get_save_extension() from
//     SokoFile.
//
// -*- Relocated functionality of soko_save_filename_for_puzzle(),
//     soko_puzzle_name_for_level(), and soko_level_for_puzzle_name() from
//     SokoFile to SokoPuzzle, where they are class methods of SokoPuzzle.
//
// -*- Added get_puzzle_extensions(), get_puzzle_extensions_count(), and
//     get_save_extension() as class methods to SokoPuzzle.  There are now
//     multiple recognized puzzle extensions, rather than just the one
//     (.sokomaze).
//
// -*- Added file_is_puzzle(), file_is_save_game(), and get_file_style()
//     class methods to SokoPuzzle.
//
// -*- Extended keyboard movement support.  Numeric keypad can now be used,
//     as well as standard arrow keys.  Added key equivalents for hexagonal
//     puzzles.
//
// -*- SokoPuzzle now calculates two new scores in addition to "moves" and
//     "pushes".  The "runs" score is the number of straight lines in which
//     crates have been pushed.  Looked at another way, it is the number of
//     turns crates have made while being pushed.  The "focus" score is the
//     number of times the player's focus has changed from one crate to
//     another.
//
// -*- Added "runs" and "focus" scores controls to SokoBoard.
//
// -*- Rewrote SokoMatrix.  It is now a generic base class for rendering
//     grids.  No longer based upon Matrix (which was only capable of
//     supporting rectangular cells).  Custom rendering is now done, rather
//     than relying upon Matrix.  Consequently, animation speed has increased
//     by a factor of five or more.
//
// -*- Added SokoMatrixSquare, a subclass of SokoMatrix, which knows how to
//     draw square-tiled puzzles, and perform appropriate hit-testing.
//
// -*- Added SokoMatrixHexagon, a subclass of SokoMatrix, which knows how to
//     draw hexagon-tiled puzzles, and perform appropriate hit-testing.
//-----------------------------------------------------------------------------
#import "SokoPool.h"
#import "SokoBoard.h"
#import "SokoAlerter.h"
#import "SokoAssert.h"
#import "SokoFile.h"
#import "SokoMatrixHexagon.h"
#import "SokoMatrixSquare.h"
#import "SokoMatrixTriangle.h"
#import "SokoNewScore.h"
#import "SokoPref.h"
#import "SokoSetting.h"
#import "SokoUtil.h"
#import "SokoWindow.h"
#import <appkit/Application.h>
#import <appkit/Box.h>
#import <appkit/Button.h>
#import <appkit/ButtonCell.h>
#import <appkit/MenuCell.h>
#import <appkit/NXImage.h>
#import <appkit/OpenPanel.h>
#import <appkit/SavePanel.h>
#import <appkit/Slider.h>
#import <appkit/TextField.h>
#import <math.h>

//-----------------------------------------------------------------------------
// SokoPuzzleDelegate Callback Functions
//-----------------------------------------------------------------------------
#define DEL_BOARD ((SokoBoard*)(x1->info))
#define DEL_FUNC_0(RET,NAME) \
    static RET NAME(SokoPuzzle x1, SokoPuzzleDelegate x2)
#define DEL_FUNC_1(RET,NAME,T1,A1) \
    static RET NAME(SokoPuzzle x1, SokoPuzzleDelegate x2, T1 A1)
#define DEL_FUNC_2(RET,NAME,T1,A1,T2,A2) \
    static RET NAME(SokoPuzzle x1, SokoPuzzleDelegate x2, T1 A1, T2 A2)
#define DEL_FUNC_3(RET,NAME,T1,A1,T2,A2,T3,A3) \
    static RET NAME(SokoPuzzle x1, SokoPuzzleDelegate x2, T1 A1, T2 A2, T3 A3)
#define DEL_FUNC_4(RET,NAME,T1,A1,T2,A2,T3,A3,T4,A4) \
    static RET NAME(SokoPuzzle x1, SokoPuzzleDelegate x2, \
	T1 A1, T2 A2, T3 A3, T4 A4)

DEL_FUNC_3(void, puzzle_alert, SokoAlert,s, char const*,t, char const*,m)
    { SokoSendAlert( s, t, m ); }
DEL_FUNC_4(void, record_score, int,m, int,p, int,r, int,f)
    { [DEL_BOARD recordMoves:m pushes:p runs:r focus:f]; }
DEL_FUNC_1(void, record_level, int,level)
    { [DEL_BOARD recordLevel:level]; }
DEL_FUNC_1(void, set_solved, soko_bool,solved)
    { [DEL_BOARD setSolved:solved]; }
DEL_FUNC_1(void, set_dirty, soko_bool,dirty)
    { [DEL_BOARD setDirty:dirty]; }
DEL_FUNC_0(void, refresh_controls)
    { [DEL_BOARD refreshControls]; }
DEL_FUNC_0(void, refresh_all_cells)
    { [DEL_BOARD refreshAllCells]; }
DEL_FUNC_2(void, refresh_cells, SokoCell const*,cells, int,ncells)
    { [DEL_BOARD refreshCells:cells count:ncells]; }
DEL_FUNC_0(soko_bool, should_animate_playback)
    { return [DEL_BOARD shouldAnimatePlayback]; }
DEL_FUNC_0(int, playback_threshold)
    { return [DEL_BOARD playbackThreshold]; }
DEL_FUNC_0(void, begin_playback)
    { [DEL_BOARD beginPlayback]; }
DEL_FUNC_0(void, end_playback)
    { [DEL_BOARD endPlayback]; }
DEL_FUNC_0(void, begin_animation)
    { [DEL_BOARD beginAnimation]; }
DEL_FUNC_0(void, end_animation)
    { [DEL_BOARD endAnimation]; }
DEL_FUNC_0(void, begin_drag)
    { [DEL_BOARD beginDrag]; }
DEL_FUNC_0(void, end_drag)
    { [DEL_BOARD endDrag]; }
DEL_FUNC_1(void, process_events, soko_bool,await_event)
    { [DEL_BOARD processEvents:await_event]; }

#undef DEL_FUNC_2
#undef DEL_FUNC_1
#undef DEL_FUNC_0
#undef DEL_BOARD


//=============================================================================
// IMPLEMENTATION
//=============================================================================
@implementation SokoBoard

//-----------------------------------------------------------------------------
// validateMenuItem:
//-----------------------------------------------------------------------------
- (BOOL)validateMenuItem:(MenuCell*)item
    {
    BOOL valid = YES;
    SEL const a = [item action];
    if (a == @selector(openNextPuzzle:))
	valid = nextPuzzleExists;
    else
	{
	int const pos = SokoPuzzle_history_position( puzzle );
	int const len = SokoPuzzle_history_length( puzzle );
	BOOL const active = SokoPuzzle_playback_active( puzzle );
	if (a == @selector(undoMove:) || a == @selector(undoPush:))
	    valid = !active && (pos > 0);
	else if (a == @selector(redoMove:) || a == @selector(redoPush:))
	    valid = !active && (pos < len);
	}
    return valid;
    }


//-----------------------------------------------------------------------------
// print:
//-----------------------------------------------------------------------------
- (id)print:(id)sender
    {
    [window smartPrintPSCode:self];
    return self;
    }


//=============================================================================
// SokoPuzzle Delegate Support
//=============================================================================
//-----------------------------------------------------------------------------
// puzzleDelegate
//-----------------------------------------------------------------------------
+ (SokoPuzzleDelegate)puzzleDelegate
    {
    static struct _SokoPuzzleDelegate delegate;
    static BOOL initialized = NO;
    if (!initialized)
	{
	delegate.alert_callback = puzzle_alert;
	delegate.record_score_callback = record_score;
	delegate.record_level_callback = record_level;
	delegate.set_solved_callback = set_solved;
	delegate.set_dirty_callback = set_dirty;
	delegate.refresh_controls_callback = refresh_controls;
	delegate.refresh_all_cells_callback = refresh_all_cells;
	delegate.refresh_cells_callback = refresh_cells;
	delegate.should_animate_playback_callback = should_animate_playback;
	delegate.playback_threshold_callback = playback_threshold;
	delegate.begin_playback_callback = begin_playback;
	delegate.end_playback_callback = end_playback;
	delegate.begin_animation_callback = begin_animation;
	delegate.end_animation_callback = end_animation;
	delegate.begin_drag_callback = begin_drag;
	delegate.end_drag_callback = end_drag;
	delegate.process_events_callback = process_events;
	delegate.info = 0;
	initialized = YES;
	}
    return &delegate;
    }


//-----------------------------------------------------------------------------
// recordLevel:
//-----------------------------------------------------------------------------
- (void)recordLevel:(int)level
    {
    if (level >= [SokoPref getNextLevel])
	[SokoPref setNextLevel:level + 1];
    }


//-----------------------------------------------------------------------------
// recordScore:
//-----------------------------------------------------------------------------
- (id)recordScore:(id)sender
    {
    [SokoNewScore solved:puzzle moves:recordMoves pushes:recordPushes
	runs:recordRuns focus:recordFocus];
    return self;
    }


//-----------------------------------------------------------------------------
// recordMoves:pushes:runs:focus:
//	Delay actual recording of score until application is idle, since this
//	method might otherwise be called during a mouse down, a drag session,
//	a key down, etc, and it would be unfriendly to launch the modal "New
//	Score" panel before those activities ceased.
//-----------------------------------------------------------------------------
- (void)recordMoves:(int)m pushes:(int)p runs:(int)r focus:(int)f
    {
    recordMoves  = m;
    recordPushes = p;
    recordRuns   = r;
    recordFocus  = f;
    [self perform:@selector(recordScore:) with:0 afterDelay:1
	cancelPrevious:YES];
    }


//-----------------------------------------------------------------------------
// setSolved:
//-----------------------------------------------------------------------------
- (void)setSolved:(BOOL)solved
    {
    [solvedFld setStringValue:(solved ? "SOLVED" : "")];
    }


//-----------------------------------------------------------------------------
// setDirty:
//-----------------------------------------------------------------------------
- (void)setDirty:(BOOL)dirty
    {
    [window setDocEdited:dirty];
    }


//-----------------------------------------------------------------------------
// refreshControls
//	Set enabled status of various controls.  Also ensure that the cell
//	matrix is always first-responder so that keyboard movement controls
//	work even if some other control on the window stole first-responder
//	status.
//-----------------------------------------------------------------------------
- (void)refreshControls
    {
    int  const hpos = SokoPuzzle_history_position( puzzle );
    int  const hlen = SokoPuzzle_history_length( puzzle );
    BOOL const undo = (hpos > 0);
    BOOL const redo = (hpos < hlen);

#define ENABLE_DIR(B,D) [move##B##Btn \
    setEnabled:SokoPuzzle_can_move_constrained(puzzle,SOKO_DIR_##D)]
#define ENABLE_DIAG(B,D) [move##B##Btn \
    setEnabled:SokoPuzzle_can_move_diagonal_constrained(puzzle,SOKO_DIAG_##D)]
    ENABLE_DIR(Up,UP);
    ENABLE_DIR(Left,LEFT);
    ENABLE_DIR(Right,RIGHT);
    ENABLE_DIR(Down,DOWN);
    ENABLE_DIAG(UpLeft,UP_LEFT);
    ENABLE_DIAG(UpRight,UP_RIGHT);
    ENABLE_DIAG(DownLeft,DOWN_LEFT);
    ENABLE_DIAG(DownRight,DOWN_RIGHT);
#undef ENABLE_DIAG
#undef ENABLE_DIR

    [undoMoveBtn setEnabled:undo];
    [redoMoveBtn setEnabled:redo];
    [undoPushBtn setEnabled:undo];
    [redoPushBtn setEnabled:redo];
    [playbackSlider setMaxValue:hlen];
    [playbackSlider setIntValue:hpos];
    [playbackSlider setEnabled:(hlen > 0)];

    [numMovesFld  setIntValue:SokoPuzzle_move_count ( puzzle )];
    [numPushesFld setIntValue:SokoPuzzle_push_count ( puzzle )];
    [numRunsFld   setIntValue:SokoPuzzle_run_count  ( puzzle )];
    [numFocusFld  setIntValue:SokoPuzzle_focus_count( puzzle )];

    if ([window firstResponder] != cellMatrix)
	[window makeFirstResponder:cellMatrix];
    }


//-----------------------------------------------------------------------------
// refreshAllCells
//-----------------------------------------------------------------------------
- (void)refreshAllCells
    {
    int r, c;
    int nr = SokoPuzzle_row_count(puzzle), nc = SokoPuzzle_col_count(puzzle);
    [window disableDisplay];
    for (r = 0; r < nr; r++)
	for (c = 0; c < nc; c++)
	    [cellMatrix setCellType:SokoPuzzle_cell_type(puzzle, r, c)
		atRow:r column:c];
    [window reenableDisplay];
    [cellMatrix display];
    }


//-----------------------------------------------------------------------------
// refreshCells:count:
//-----------------------------------------------------------------------------
- (void)refreshCells:(SokoCell const*)cells count:(int)ncells
    {
    int i;
    [window disableFlushWindow];
    for (i = 0; i < ncells; i++)
	[cellMatrix setCellType:cells[i].type
	    atRow:cells[i].row column:cells[i].col];
    [window reenableFlushWindow];
    [window flushWindowIfNeeded];
    }


//=============================================================================
// Board Creation & Destruction Support
//=============================================================================
//-----------------------------------------------------------------------------
// cascadeWindow:
//-----------------------------------------------------------------------------
+ (void)cascadeWindow:(Window*)w
    {
    static BOOL initialized = NO;
    static NXCoord baseTop = 0;
    static NXCoord baseLeft = 0;
    static int cascadeCounter = -1;
    NXCoord top,left;

    if (!initialized)
	{
	NXSize sz;
	[NXApp getScreenSize:&sz];
	baseLeft = floor(sz.width / 10);
	baseTop = sz.height - floor(sz.height / 10);
	initialized = YES;
	}

    if (++cascadeCounter == 10)
	cascadeCounter = 0;

    top  = baseTop  - (cascadeCounter * 20);
    left = baseLeft + (cascadeCounter * 20);
    [w moveTopLeftTo:left:top];
    }


//-----------------------------------------------------------------------------
// adjustWindowSize
//	Resize window to contain puzzle matrix.  However, consider designed
//	control box width as minimum width limit, so do not shrink window
//	horizontally beyond that limit.  If puzzle matrix is smaller than
//	minimum limit, then center matrix in window.
// *BOX-WIDTH*
//	The purpose of this expression is to grab the design-time width of the
//	control box.  The expression is only valid when -adjustWindowSize is
//	invoked immediately after the nib is loaded.  If, in the future,
//	-adjustWindowSize is invoked more than once (as is the case with the
//	Cocoa/OpenStep port), then it will be necessary to latch and remember
//	the design-time width, rather than accessing it directly from the box
//	frame.
//-----------------------------------------------------------------------------
- (void)adjustWindowSize
    {
    NXSize windowSize;
    NXRect boxFrame;
    NXRect matrixFrame;

    [controlBox getFrame:&boxFrame];
    [cellMatrix getFrame:&matrixFrame];
    windowSize.width = boxFrame.size.width;	// *BOX-WIDTH*

    matrixFrame.origin.y = boxFrame.size.height + 4;
    if (matrixFrame.size.width + 8 >= windowSize.width)
	{
	matrixFrame.origin.x = 4;
	windowSize.width = matrixFrame.size.width + 8;
	}
    else
	{
	matrixFrame.origin.x =
	    floor((windowSize.width - matrixFrame.size.width) / 2);
	}

    windowSize.height = matrixFrame.size.height + boxFrame.size.height + 8;

    [window sizeWindow:windowSize.width:windowSize.height];

    [controlBox sizeTo:windowSize.width:boxFrame.size.height];
    [controlBox moveTo:0:0];
    [cellMatrix moveTo:matrixFrame.origin.x:matrixFrame.origin.y];
    }


//-----------------------------------------------------------------------------
// createMatrix
//-----------------------------------------------------------------------------
- (void)createMatrix
    {
    int const numRows = SokoPuzzle_row_count( puzzle );
    int const numCols = SokoPuzzle_col_count( puzzle );
    SokoPuzzleStyle const style = SokoPuzzle_puzzle_style( puzzle );
    id matrixClass = 0;

    if (style == SOKO_STYLE_SQUARE)
	matrixClass = [SokoMatrixSquare class];
    else if (style == SOKO_STYLE_HEXAGON)
	matrixClass = [SokoMatrixHexagon class];
    else if (style == SOKO_STYLE_TRIANGLE)
	matrixClass = [SokoMatrixTriangle class];
    SOKO_ASSERT( matrixClass != 0 );
	
    cellMatrix = [[matrixClass allocFromZone:[self zone]]
	initWithRows:numRows columns:numCols];
    [cellMatrix setDelegate:self];
    [[window contentView] addSubview:cellMatrix];
    [window makeFirstResponder:cellMatrix];
    }


//-----------------------------------------------------------------------------
// configureControls
//-----------------------------------------------------------------------------
- (void)configureControls
    {
#define BUTTON_TAG(X,Y) [move##X##Btn setTag:SOKO_##Y]
    BUTTON_TAG(Up,DIR_UP);
    BUTTON_TAG(Down,DIR_DOWN);
    BUTTON_TAG(Left,DIR_LEFT);
    BUTTON_TAG(Right,DIR_RIGHT);
    BUTTON_TAG(UpLeft,DIAG_UP_LEFT);
    BUTTON_TAG(UpRight,DIAG_UP_RIGHT);
    BUTTON_TAG(DownLeft,DIAG_DOWN_LEFT);
    BUTTON_TAG(DownRight,DIAG_DOWN_RIGHT);
#undef BUTTON_TAG
#undef DISABLE_FR
    }


//-----------------------------------------------------------------------------
// init
//-----------------------------------------------------------------------------
- (id)init
    {
    [super init];
    [NXApp loadNibSection:"SokoBoard.nib" owner:self withNames:NO];
    [window setMiniwindowIcon:"SokoSave"];
    [self configureControls];
    puzzle = 0;
    nextPuzzleExists = NO;
    recordMoves  = -1;
    recordPushes = -1;
    recordRuns   = -1;
    recordFocus  = -1;
    return self;
    }


//-----------------------------------------------------------------------------
// free
//-----------------------------------------------------------------------------
- (id)free
    {
    [self perform:@selector(recordScore:) with:0 afterDelay:-1
	cancelPrevious:YES]; // Cancel any pending delayed-perform.
    if (puzzle != 0)
	SokoPuzzle_destroy( puzzle );
    [window setDelegate:0];
    [window close];
    [window free];
    return [super free];
    }


//-----------------------------------------------------------------------------
// refreshWindowTitle
//-----------------------------------------------------------------------------
- (void)refreshWindowTitle
    {
    SokoPool pool = SokoPool_new(0);
    char const* path = SokoPuzzle_get_save_file_name( puzzle, pool );
    [window setTitleAsFilename:path];
    SokoPool_destroy( pool );
    }


//-----------------------------------------------------------------------------
// checkForNextPuzzle
//-----------------------------------------------------------------------------
- (void)checkForNextPuzzle
    {
    SokoPool pool = SokoPool_new(0);
    int const level = SokoPuzzle_level_for_puzzle_name(
	 SokoPuzzle_get_puzzle_file_name(puzzle, pool) );
    nextPuzzleExists =
	soko_path_exists(pool,SokoPuzzle_puzzle_name_for_level(level+1, pool));
    SokoPool_destroy( pool );
    }


//-----------------------------------------------------------------------------
// configureBoard:
//-----------------------------------------------------------------------------
- (void)configureBoard:(SokoPuzzle)p
    {
    puzzle = p;
    [self createMatrix];
    [self adjustWindowSize];
    [self setDirty:SokoPuzzle_puzzle_dirty( puzzle )];
    [self setSolved:SokoPuzzle_puzzle_solved( puzzle )];
    [self checkForNextPuzzle];
    [self refreshControls];
    [self refreshAllCells];
    [self refreshWindowTitle];
    [stopBtn setEnabled:NO];
    [[self class] cascadeWindow:window];
    }


//-----------------------------------------------------------------------------
// activate
//-----------------------------------------------------------------------------
- (void)activate
    {
    [window makeKeyAndOrderFront:self];
    }


//-----------------------------------------------------------------------------
// windowWillClose:
//-----------------------------------------------------------------------------
- (id)windowWillClose:(id)sender
    {
    SokoPuzzle_abort_playback( puzzle );
    [NXApp delayedFree:self];
    return self;
    }


//-----------------------------------------------------------------------------
// openPuzzle:
//-----------------------------------------------------------------------------
+ (BOOL)openPuzzle:(char const*)path
    {
    SokoBoard* board;
    SokoPuzzle p = SokoPuzzle_find_open_puzzle( path );
    if (p != 0)
	board = (SokoBoard*)p->info;
    else
	{
	board = [[SokoBoard alloc] init];
	p = SokoPuzzle_new( [[self class] puzzleDelegate], path, board );
	if (p != 0)
	    [board configureBoard:p];
	else
	    {
	    [board free];
	    board = 0;
	    }
	}
    if (board != 0)
	[board activate];
    return (board != 0);
    }


//-----------------------------------------------------------------------------
// openPuzzles:fromDirectory:
//-----------------------------------------------------------------------------
+ (BOOL)openPuzzles:(char const* const*)files fromDirectory:(char const*)dir
    {
    BOOL ok = NO;
    if (files != 0 && dir != 0)
	{
	SokoPool pool = SokoPool_new(0);
	for ( ; *files != 0; files++)
	    {
	    char const* path = soko_add_path_component( pool, dir, *files );
	    if ([self openPuzzle:path])
		ok = YES;
	    }
	SokoPool_destroy( pool );
	}
    return ok;
    }


//-----------------------------------------------------------------------------
// getOpenTypes:seed:
//	Return a null-terminated list of all puzzle extensions.  Also include
//	an additional extension (seed) if provided.
//-----------------------------------------------------------------------------
+ (char const* const*)getOpenTypes:(SokoPool)pool seed:(char const*)seed
    {
    SokoPuzzleExtension const* e = SokoPuzzle_get_puzzle_extensions();
    int n = SokoPuzzle_get_puzzle_extensions_count() + (seed == 0 ? 0 : 1);
    char const** types =
	(char const**)SokoPool_allocate( pool, (n + 1) * sizeof(char const*) );
    types[n--] = 0;
    for ( ; e->extension != 0; e++ )
	types[n--] = e->extension;
    if (seed != 0)
	types[n--] = seed;
    SOKO_ASSERT( n == -1 );
    return types;
    }


//-----------------------------------------------------------------------------
// newPuzzle
//	Only seed the dialog with the name of "next" puzzle if the puzzle
//	exists and if the directory last used by the panel is the same the
//	puzzle's path.  Also, only seed the panel's directory when first
//	created, but never after that.  This way, if the user decides to launch
//	new games from some other directory (containing, for instance, puzzles
//	not distributed with SokoSave), we won't keep jumping back to the
//	factory puzzle directory.
//-----------------------------------------------------------------------------
+ (void)newPuzzle
    {
    SokoPool pool = SokoPool_new(0);
    char const* const* openTypes;
    OpenPanel* openPanel;
    char const* seed;
    char const* file = "";
    int rc;
    static char* openDir = 0;

    openPanel = [OpenPanel new];
    [openPanel setTitle:"New Puzzle"];
    [openPanel allowMultipleFiles:YES];
    [openPanel setTreatsFilePackagesAsDirectories:YES];

    if (openDir == 0)
	openDir = soko_strdup(
	    soko_expand_path(pool, soko_get_puzzle_directory(pool)) );

    seed = SokoPuzzle_puzzle_name_for_level( [SokoPref getNextLevel], pool );
    if (soko_path_exists( pool, seed ))
	{
	char const* dir = soko_directory_part( pool, seed );
	if (strcmp( dir, openDir ) == 0)
	    file = soko_filename_part( pool, seed );
	}

    openTypes = [self getOpenTypes:pool seed:0];
    rc = [openPanel runModalForDirectory:openDir file:file types:openTypes];
    [openPanel close];
    if (rc == NX_OKTAG)
	{
	char const* dir = [openPanel directory];
	[self openPuzzles:[openPanel filenames] fromDirectory:dir];
	free( openDir );
	openDir = soko_strdup( dir );
	}

    SokoPool_destroy( pool );
    }


//-----------------------------------------------------------------------------
// choosePuzzleFromDirectory:
//	Allow both puzzle and saved-game files to be opened.  If "directory" is
//	non-nil, then use the specified directory, else use last visited
//	directory.
//-----------------------------------------------------------------------------
+ (BOOL)choosePuzzleFromDirectory:(char const*)dir
    {
    BOOL ok = NO;
    SokoPool pool = SokoPool_new(0);
    char const* const* openTypes;
    OpenPanel* openPanel;
    int rc;
    static char* openDir = 0;

    openPanel = [OpenPanel new];
    [openPanel setTitle:"Open Puzzle"];
    [openPanel allowMultipleFiles:YES];
    [openPanel setTreatsFilePackagesAsDirectories:YES];

    if (openDir == 0)
	openDir = soko_strdup(
	    soko_expand_path(pool, soko_get_save_directory(pool)) );

    if (dir == 0 || *dir == '\0')
	dir = openDir;

    openTypes = [self getOpenTypes:pool seed:SokoPuzzle_get_save_extension()];
    rc = [openPanel runModalForDirectory:dir file:0 types:openTypes];
    [openPanel close];
    if (rc == NX_OKTAG)
	{
	char const* dir = [openPanel directory];
	ok = [self openPuzzles:[openPanel filenames] fromDirectory:dir];
	free( openDir );
	openDir = soko_strdup( dir );
	}

    SokoPool_destroy( pool );
    return ok;
    }


//-----------------------------------------------------------------------------
// openPuzzle
//-----------------------------------------------------------------------------
+ (void)openPuzzle
    {
    [self choosePuzzleFromDirectory:0];
    }


//-----------------------------------------------------------------------------
// openPuzzleForLevel:
//	This method is called at program launch time to automatically open the
//	next puzzle which the user has not yet solved.  On the assumption that
//	the user may have begun working on the puzzle and then saved it with
//	the intention of continuing work on it later, we first try to open a
//	saved game with the given level number.  If a saved game does not
//	exist, then we try starting a new game.
//-----------------------------------------------------------------------------
+ (BOOL)openPuzzleForLevel:(int)level
    {
    BOOL ok = NO;
    SokoPool pool = SokoPool_new(0);
    char const* newGame = SokoPuzzle_puzzle_name_for_level( level, pool );
    char const* saveGame = SokoPuzzle_save_name_for_puzzle_name(newGame, pool);
    if (soko_path_exists( pool, saveGame ))
	ok = [self openPuzzle:saveGame];
    if (!ok && soko_path_exists( pool, newGame ))
	ok = [self openPuzzle:newGame];
    SokoPool_destroy( pool );
    return ok;
    }


//-----------------------------------------------------------------------------
// openDefaultPuzzle
//-----------------------------------------------------------------------------
+ (BOOL)openDefaultPuzzle
    {
    return [self openPuzzleForLevel:[SokoPref getNextLevel]] ||
	   [self openPuzzleForLevel:1];
    }


//-----------------------------------------------------------------------------
// openNextPuzzle:
//	Invoked by the "Next Puzzle" user interface element.  Opens the puzzle
//	or saved game which is "next" after the puzzle in represented by this
//	SokoBoard instance.
//-----------------------------------------------------------------------------
- (id)openNextPuzzle:(id)sender
    {
    SokoPool pool = SokoPool_new(0);
    int const level = SokoPuzzle_level_for_puzzle_name(
	SokoPuzzle_get_puzzle_file_name(puzzle, pool) );
    if (level >= 0)
	[[self class] openPuzzleForLevel:level + 1];
    SokoPool_destroy( pool );
    return self;
    }


//-----------------------------------------------------------------------------
// save:
//-----------------------------------------------------------------------------
- (id)save:(id)sender
    {
    SokoPuzzle_abort_playback( puzzle );
    SokoPuzzle_save( puzzle );
    return self;
    }


//-----------------------------------------------------------------------------
// saveAs:
//-----------------------------------------------------------------------------
- (id)saveAs:(id)sender
    {
    SokoPool pool = SokoPool_new(0);
    char const* file =
	soko_filename_part(pool, SokoPuzzle_get_save_file_name(puzzle, pool));
    SavePanel* savePanel;
    int rc;
    static char* saveDir = 0;

    savePanel = [SavePanel new];
    [savePanel setTitle:"Save Puzzle"];
    [savePanel setRequiredFileType:SokoPuzzle_get_save_extension()];

    if (saveDir == 0)
	saveDir = soko_strdup(
	    soko_expand_path(pool, soko_get_save_directory(pool)) );

    SokoPuzzle_abort_playback( puzzle );
    rc = [savePanel runModalForDirectory:saveDir file:file];
    if (rc == NX_OKTAG)
	{
	if (SokoPuzzle_save_as( puzzle, [savePanel filename] ))
	    {
	    [self refreshWindowTitle];
	    free( saveDir );
	    saveDir = soko_strdup( [savePanel directory] );
	    }
	}

    SokoPool_destroy( pool );
    return self;
    }


//-----------------------------------------------------------------------------
// saveAllPuzzles
//-----------------------------------------------------------------------------
+ (void)saveAllPuzzles
    {
    SokoPuzzle_save_all_puzzles();
    }


//=============================================================================
// Busy Operation Support
//=============================================================================
//-----------------------------------------------------------------------------
// processEvents:
//	Whenever SokoPuzzle needs to perform a lengthy operation which may
//	cause the application to become unresponsive, or when it needs to
//	perform some internal looping while still receiving events (such as
//	during player or crate dragging), it calls -processEvents: to ensure
//	that the user interface remains responsive.  If `awaitEvent' is YES
//	then this method returns only after processing at least one event.  If
//	`awaitEvent' is NO, and if there are no pending events, then this
//	method returns immediately.  In either case, if any events are pending,
//	those events are processed and -processEvents: then returns.
//-----------------------------------------------------------------------------
- (void)processEvents:(BOOL)awaitEvent
    {
    NXEvent* e;
    if (awaitEvent)
	e = [NXApp getNextEvent:NX_ALLEVENTS];
    else
	e = [NXApp peekAndGetNextEvent:NX_ALLEVENTS];
    while (e != 0)
	{
	[NXApp sendEvent:e];
	e = [NXApp peekAndGetNextEvent:NX_ALLEVENTS];
	}
    }


//=============================================================================
// History Playback Session Support
//=============================================================================
- (BOOL)shouldAnimatePlayback { return [animateSwitch state]; }
- (int)playbackThreshold { return 40; }

- (id)stopPressed:(id)sender
    { SokoPuzzle_abort_playback( puzzle ); return self; }
- (id)animateToggled:(id)sender { return self; }

- (id)windowDidResignKey:(id)sender
    { SokoPuzzle_abort_playback( puzzle ); return self; }
- (id)windowDidResignMain:(id)sender
    { SokoPuzzle_abort_playback( puzzle ); return self; }

//-----------------------------------------------------------------------------
// beginPlayback
//	When SokoPuzzle wants to animate a particularly lengthy sequence of
//	moves, it invokes -beginPlayback (assuming that -shouldAnimatePlayback
//	returned YES).  It is the responsibility of -beginPlayback to arrange
//	for SokoPuzzle_advance_playback() to be called on a periodic basis
//	until SokoPuzzle invokes -endPlayback.  In order to keep the user
//	interface responsive (so that the user can cancel playback, for
//	example), the application should be allowed to continue processing and
//	dispatch events.  Both requirements are met via use of
//	-perform:with:afterDelay:cancelPrevious:, with a delay of zero which
//	causes the action to be performed as soon as possible after control is
//	returned to the Application's run-loop.  Finally, ensure that the
//	cellMatrix is still first-responder so that the user can cancel
//	playback via the Escape key, if desired.
//-----------------------------------------------------------------------------
- (void)beginPlayback
    {
    [stopBtn          setEnabled:YES];
    [animateSwitch    setEnabled:NO];
    [playbackSlider   setEnabled:NO];
    [undoMoveBtn      setEnabled:NO];
    [redoMoveBtn      setEnabled:NO];
    [undoPushBtn      setEnabled:NO];
    [redoPushBtn      setEnabled:NO];
    [moveUpBtn        setEnabled:NO];
    [moveDownBtn      setEnabled:NO];
    [moveLeftBtn      setEnabled:NO];
    [moveRightBtn     setEnabled:NO];
    [moveUpLeftBtn    setEnabled:NO];
    [moveUpRightBtn   setEnabled:NO];
    [moveDownLeftBtn  setEnabled:NO];
    [moveDownRightBtn setEnabled:NO];

    if ([window firstResponder] != cellMatrix)
	[window makeFirstResponder:cellMatrix];

    [self perform:@selector(advancePlayback:) with:0 afterDelay:0
	cancelPrevious:NO];
    }


//-----------------------------------------------------------------------------
// endPlayback
//	When SokoPuzzle wants to animate a particularly lengthy sequence of
//	moves, it invokes -endPlayback.
// *EVENT*
//	Place a dummy event into the queue to force menu-item validation to
//	fire, thus re-enabling menu items disabled during playback.  Without
//	this, those items would not be re-enabled until the next user event.
// *CONTROLS*
//	Most controls will be re-enabled by normal control-update machinery
//	following conclusion of playback, so we do not need to manually
//	re-enable all of the ones disabled by -beginPlayback.
//-----------------------------------------------------------------------------
- (void)endPlayback
    {
    NXEvent e;
    e.type = NX_APPDEFINED;
    e.ctxt = [NXApp context];
    e.time = 0;
    e.flags = 0;
    e.window = 0;
    DPSPostEvent( &e, FALSE );		// *EVENT*

    [stopBtn        setEnabled:NO ];	// *CONTROLS*
    [animateSwitch  setEnabled:YES];
    [playbackSlider setEnabled:YES];

    [self perform:@selector(advancePlayback:) with:0 afterDelay:-1
	cancelPrevious:YES]; // Cancel any pending -advancePlayback: messages.
    }


//-----------------------------------------------------------------------------
// advancePlayback:
//	Each time the application's run-loop becomes idle while history
//	playback is active, this method is invoked in order to advance the
//	playback.  If, after advancing the playback by one step, playback is
//	still active, then another notification is queued for delivery at the
//	next application-idle time.
// *SLIDER*
//	There is a bug in the AppKit where sending -setEnabled:NO to a Slider
//	fails to work correctly if it sent in response to an action message
//	from the slider itself (i.e. the slider just stopped tracking the mouse
//	and has just dispatched its action message).  Although, upon receipt of
//	-setEnabled:NO, even though the slider is redrawn with a disabled
//	appearance, it is never-the-less still enabled and still responds to
//	mouse events.  To work around this problem, it is necessary to disable
//	the slider the first time -advancePlayback: is called (even though the
//	slider was supposed to have been disabled by -beginPlayback).
//-----------------------------------------------------------------------------
- (id)advancePlayback:(id)sender
    {
    if (SokoPuzzle_playback_active( puzzle ))	// Ignore stray notifications.
	{
	if ([playbackSlider isEnabled])		// *SLIDER*
	    [playbackSlider setEnabled:NO];
	SokoPuzzle_advance_playback( puzzle );
	if (SokoPuzzle_playback_active( puzzle ))
	    [self perform:@selector(advancePlayback:) with:0 afterDelay:0
		cancelPrevious:NO];
	}
    return self;
    }


//=============================================================================
// Animation Sequence Support
//=============================================================================
- (void)beginAnimation {}
- (void)endAnimation {}


//=============================================================================
// Dragging Support
//=============================================================================
- (void)beginDrag
    { [window addToEventMask:NX_LMOUSEDRAGGEDMASK|NX_RMOUSEDRAGGEDMASK]; }
- (void)endDrag
    { [window removeFromEventMask:NX_LMOUSEDRAGGEDMASK|NX_RMOUSEDRAGGEDMASK]; }


//=============================================================================
// History Support
//=============================================================================
- (id)undoMove:(id)sender { SokoPuzzle_undo_move( puzzle ); return self; }
- (id)redoMove:(id)sender { SokoPuzzle_redo_move( puzzle ); return self; }
- (id)undoPush:(id)sender { SokoPuzzle_undo_push( puzzle ); return self; }
- (id)redoPush:(id)sender { SokoPuzzle_redo_push( puzzle ); return self; }

- (id)inspectSlider:(id)sender
    {
    SokoPuzzle_reenact_history( puzzle, [playbackSlider intValue] );
    return self;
    }


//=============================================================================
// Movement Support
//=============================================================================
- (id)directionPressed:(id)sender
    {
    SokoDirection const dir = (SokoDirection)[sender tag];
    SOKO_ASSERT( dir < SOKO_DIR_MAX );
    SokoPuzzle_move( puzzle, dir );
    return self;
    }

- (id)diagonalPressed:(id)sender
    {
    SokoDiagonal const dir = (SokoDiagonal)[sender tag];
    SOKO_ASSERT( dir < SOKO_DIAG_MAX );
    SokoPuzzle_move_diagonal( puzzle, dir );
    return self;
    }


//=============================================================================
// Keyboard Support
//=============================================================================
//-----------------------------------------------------------------------------
// eventFlags:
//-----------------------------------------------------------------------------
- (SokoEventFlags)eventFlags:(int)nxflags
    {
    SokoEventFlags flags = 0;
    if ((nxflags & NX_SHIFTMASK    ) != 0) flags |= SOKO_FLAG_SHIFT;
    if ((nxflags & NX_CONTROLMASK  ) != 0) flags |= SOKO_FLAG_CONTROL;
    if ((nxflags & NX_ALTERNATEMASK) != 0) flags |= SOKO_FLAG_ALTERNATE;
    if ((nxflags & NX_COMMANDMASK  ) != 0) flags |= SOKO_FLAG_COMMAND;
    return flags;
    }


//-----------------------------------------------------------------------------
// keyAction:isDown:
//-----------------------------------------------------------------------------
- (void)keyAction:(NXEvent const*)e isDown:(BOOL)down
    {
    #define ESC_CODE       (0x1b)
    #define UP_CODE        (0xad)
    #define DOWN_CODE      (0xaf)
    #define LEFT_CODE      (0xac)
    #define RIGHT_CODE     (0xae)
    #define KP_UP          ('8')
    #define KP_DOWN        ('2')
    #define KP_LEFT        ('4')
    #define KP_RIGHT       ('6')
    #define KP_HOME        ('7')
    #define KP_PAGE_UP     ('9')
    #define KP_END         ('1')
    #define KP_PAGE_DOWN   ('3')
    #define KP_CENTER      ('5')
    #define INSERT_CODE    (0x2c)
    #define DELETE_CODE    (0x2d)
    #define PAGE_UP_CODE   (0x30)
    #define PAGE_DOWN_CODE (0x31)
    #define EDIT_CHAR_SET  (0xfe)

    SokoKeyCode code = ((SokoKeyCode)~0);
    unsigned short const nxcset = e->data.key.charSet;
    unsigned short const nxcode = e->data.key.charCode;
    BOOL const keypad = ((e->flags & NX_NUMERICPADMASK) != 0);

    if (nxcset == NX_ASCIISET && !keypad && nxcode == ESC_CODE)
	{
	code = SOKO_KEY_ESCAPE;
	}
    else if (nxcset == NX_SYMBOLSET)
	{
	switch (nxcode)
	    {
	    case UP_CODE:    code = SOKO_KEY_UP;    break;
	    case DOWN_CODE:  code = SOKO_KEY_DOWN;  break;
	    case LEFT_CODE:  code = SOKO_KEY_LEFT;  break;
	    case RIGHT_CODE: code = SOKO_KEY_RIGHT; break;
	    }
	}
    else if (keypad && nxcset == NX_ASCIISET)
	{
	switch (nxcode)
	    {
	    case KP_UP:        code = SOKO_KEY_UP;         break;
	    case KP_DOWN:      code = SOKO_KEY_DOWN;       break;
	    case KP_LEFT:      code = SOKO_KEY_LEFT;       break;
	    case KP_RIGHT:     code = SOKO_KEY_RIGHT;      break;
	    case KP_HOME:      code = SOKO_KEY_UP_LEFT;    break;
	    case KP_PAGE_UP:   code = SOKO_KEY_UP_RIGHT;   break;
	    case KP_END:       code = SOKO_KEY_DOWN_LEFT;  break;
	    case KP_PAGE_DOWN: code = SOKO_KEY_DOWN_RIGHT; break;
	    case KP_CENTER:    code = SOKO_KEY_DOWN;       break;
	    }
	}
    else if (nxcset == EDIT_CHAR_SET)
	{
	switch (nxcode)
	    {
	    case INSERT_CODE:    code = SOKO_KEY_UP_LEFT;    break;
	    case DELETE_CODE:    code = SOKO_KEY_DOWN_LEFT;  break;
	    case PAGE_UP_CODE:   code = SOKO_KEY_UP_RIGHT;   break;
	    case PAGE_DOWN_CODE: code = SOKO_KEY_DOWN_RIGHT; break;
	    }
	}

    if (code != ((SokoKeyCode)~0))
	SokoPuzzle_key_action(puzzle, code, down, [self eventFlags:e->flags]);
    }

- (id)keyDown:(NXEvent*)e { [self keyAction:e isDown:YES]; return self; }
- (id)keyUp:  (NXEvent*)e { [self keyAction:e isDown:NO ]; return self; }


//=============================================================================
// Mouse Support
//=============================================================================
//-----------------------------------------------------------------------------
// getRow:column:forEvent:
//-----------------------------------------------------------------------------
- (BOOL)getRow:(int*)r column:(int*)c forEvent:(NXEvent const*)e
    {
    NXPoint p = e->location;
    [cellMatrix convertPoint:&p fromView:0];
    return ([cellMatrix getRow:r column:c forPoint:&p] != 0);
    }


//-----------------------------------------------------------------------------
// mouseAction:button:isDown:
//-----------------------------------------------------------------------------
- (void)mouseAction:(NXEvent const*)e button:(int)button isDown:(BOOL)down
    {
    int row, col;
    if ([self getRow:&row column:&col forEvent:e])
	SokoPuzzle_mouse_action(
	    puzzle, button, down, [self eventFlags:e->flags], row, col );
    }

- (id)mouseDown:(NXEvent*)e
    { [self mouseAction:e button:0 isDown:YES]; return self; }
- (id)mouseUp:(NXEvent*)e
    { [self mouseAction:e button:0 isDown:NO ]; return self; }
- (id)rightMouseDown:(NXEvent*)e
    { [self mouseAction:e button:1 isDown:YES]; return self; }
- (id)rightMouseUp:(NXEvent*)e
    { [self mouseAction:e button:1 isDown:NO ]; return self; }


//-----------------------------------------------------------------------------
// mouseDragged:
//-----------------------------------------------------------------------------
- (id)mouseDragged:(NXEvent*)e
    {
    int row, col;
    if ([self getRow:&row column:&col forEvent:e])
	SokoPuzzle_mouse_drag( puzzle, row, col );
    return self;
    }

- (id)rightMouseDragged:(NXEvent*)e { return [self mouseDragged:e]; }

@end
