#ifndef SokoBoard_h
#define SokoBoard_h
//-----------------------------------------------------------------------------
// SokoBoard.h
//
//	Implementation of a Sokoban puzzle board for SokoSave.
//
// Copyright (c), 1997,2001,2002, Eric Sunshine <sunshine@sunshineco.com>
// Copyright (c), 1997, Paul McCarthy <zarnuk@high-speed-software.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoBoard.h,v 1.5 2002/01/29 20:14:37 sunshine Exp $
// $Log: SokoBoard.h,v $
// Revision 1.5  2002/01/29 20:14:37  sunshine
// v17
// -*- Added support for the new hexagonal-style puzzles.
//
// -*- No longer fills areas outside the puzzle with wall segments.  Doing so
//     causes some "artistic" effect to be lost from some puzzles.  Added
//     SokoCellType SOKO_CELL_NULL to support of this.
//
// -*- Added SokoPuzzleStyle enumeration to SokoPuzzle.  Styles are
//     SOKO_STYLE_SQUARE and SOKO_STYLE_HEXAGON.
//
// -*- Now recognize these additional file extensions: .xsb, .hsb, .sokohex.
//
// -*- Removed soko_get_puzzle_extension() and soko_get_save_extension() from
//     SokoFile.
//
// -*- Relocated functionality of soko_save_filename_for_puzzle(),
//     soko_puzzle_name_for_level(), and soko_level_for_puzzle_name() from
//     SokoFile to SokoPuzzle, where they are class methods of SokoPuzzle.
//
// -*- Added get_puzzle_extensions(), get_puzzle_extensions_count(), and
//     get_save_extension() as class methods to SokoPuzzle.  There are now
//     multiple recognized puzzle extensions, rather than just the one
//     (.sokomaze).
//
// -*- Added file_is_puzzle(), file_is_save_game(), and get_file_style()
//     class methods to SokoPuzzle.
//
// -*- Extended keyboard movement support.  Numeric keypad can now be used,
//     as well as standard arrow keys.  Added key equivalents for hexagonal
//     puzzles.
//
// -*- SokoPuzzle now calculates two new scores in addition to "moves" and
//     "pushes".  The "runs" score is the number of straight lines in which
//     crates have been pushed.  Looked at another way, it is the number of
//     turns crates have made while being pushed.  The "focus" score is the
//     number of times the player's focus has changed from one crate to
//     another.
//
// -*- Added "runs" and "focus" scores controls to SokoBoard.
//
// -*- Rewrote SokoMatrix.  It is now a generic base class for rendering
//     grids.  No longer based upon Matrix (which was only capable of
//     supporting rectangular cells).  Custom rendering is now done, rather
//     than relying upon Matrix.  Consequently, animation speed has increased
//     by a factor of five or more.
//
// -*- Added SokoMatrixSquare, a subclass of SokoMatrix, which knows how to
//     draw square-tiled puzzles, and perform appropriate hit-testing.
//
// -*- Added SokoMatrixHexagon, a subclass of SokoMatrix, which knows how to
//     draw hexagon-tiled puzzles, and perform appropriate hit-testing.
//
// Revision 1.4  2001/12/23 18:17:48  sunshine
// v15
// -*- Extracted core game logic out of GUI code and generalized it so that
//     the same core code can be used by any platform.  Logic from
//     SokoBoard.m now resides in SokoPuzzle.c, etc.
//
// -*- Rewrote the NextStep port so that it utilizes the new "sokocore" game
//     library rather than implementing that logic directly.
//
// -*- Augmented all input/output logic so that it now deals gracefully with
//     line terminators from all common platforms; Unix (LF), Macintosh (CR),
//     and Windows/DOS (CRLF).
//
// -*- SokoPuzzle now understands the token "+" as player-safe, since this is
//     what is used by XSokoban, and used in numerous existing add-on
//     puzzles.  The "^" token which was historically understood by SokoSave
//     is still recognized, but is no longer published in the documentation.
//
// -*- SokoPuzzle no longer complains about loading puzzles with unreachable
//     crates, provided that those crates are already on goal squares.  This
//     allows some existing add-on puzzles to be used with SokoSave.
//
// -*- Replaced the terminology "maze" with "puzzle" throughout the project,
//     including source code, documentation, and all user-visible UI
//     elements.  The only remaining place where "maze" is still used is in
//     the file extension ".sokomaze".  I haven't decided what, if anything,
//     to do about that, yet.
//
// -*- Renamed soko_save_filename_for_maze_name() and
//     soko_maze_name_for_level() in SokoFile to to
//     soko_save_filename_for_puzzle_name() and soko_puzzle_name_for_level(),
//     respectively.
//
// -*- Added soko_level_for_puzzle_name() to SokoFile.  This function is used
//     by the new "Next Puzzle" menu item and shortcut.
//
// -*- Added SokoSetting implementation which provides a platform-independent
//     API for accessing user settings and well-known paths, such as
//     $(SokoSave) and $(SokoUser).
//
// -*- Added a "Next Puzzle" menu item (and toolbar shortcut), which opens
//     the puzzle that comes "next" after the puzzle in the active window.
//
// -*- Changed the logic used to locate the puzzle which is opened
//     automatically at program launch time.  Previously, it simply opened a
//     .sokomaze file with the level number recorded as the "highest" level.
//     Unfortunately, if the user then tried to open a saved game file with
//     the same level number, the program would not actually open the saved
//     game because it thought that the game was already open.  This made it
//     very difficult for the user to continue a saved game which matched the
//     "highest" level.  The user first has to close the "new" game and then
//     open the "saved" game.  This problem was even further complicated on
//     the Windows platform where closing the "new" game would cause the
//     program to terminate if that was the only menu-bearing window.  Since
//     it is probable that a user may save a game before actually arriving at
//     the solution with the intention of returning to that game later, the
//     program now makes it much easier to return to the saved game.  Now, at
//     launch time, the program first checks for a "saved" game matching the
//     "highest" level and open that if present.  If not, then it falls back
//     to opening a new game.
//
// -*- Completely re-worked playback animation logic.  The pseudo-modal loop
//     which specially filtered and dispatched events is no longer used.
//     Instead, animation is now driven from the main event loop and occurs
//     in tandem with the receipt and processing of normal user events.
//     Rather than trying to restrict and filter user events, all events are
//     now handled as usual, except that some events may have the side-effect
//     of aborting playback.  There are now several different mechanisms for
//     aborting playback: pressing the stop button, pressing Escape, clicking
//     in the puzzle view, switching to a different window, deactivating the
//     application (for instance, by clicking on the window of a another
//     application, or hiding SokoSave).
//
// -*- Now remembers the directory from which a new game was loaded during a
//     session, and seeds the New Game open panel with that location.  This
//     makes it more convenient for the user to start new games from a
//     directory other than the default New Game directory.  (It was very
//     annoying for the program to revert to the factory directory each time
//     New Game was invoked.)
//
// -*- Added SokoWindow and SokoPanel which are subclasses of Window and
//     Panel, respectively.  All windows and panels throughout the
//     application are now instances of these classes.  Although these
//     classes currently implement no extra functionality, they provide
//     convenient hooks for the addition of new functionality in the future,
//     if needed.
//
// -*- Directional movement buttons on SokoBoard window are now connected to
//     -directionPressed: and diagonalPressed:, rather than each being
//     connected to a dedicated method.
//
// -*- Renamed -[SokoBoard inspectPlaybackSlider:] to -inspectSlider:.
//
// -*- Connected the "Animate" switch on SokoBoard to the new -[SokoBoard
//     animateToggled:] method.
//-----------------------------------------------------------------------------
#import <objc/Object.h>
#import "SokoPuzzle.h"
@class Box, Button, MenuCell, Slider, SokoMatrix, SokoWindow, TextField;

@interface SokoBoard : Object
    {
    Button* animateSwitch;
    SokoMatrix* cellMatrix;
    Box* controlBox;
    Button* moveDownBtn;
    Button* moveDownLeftBtn;
    Button* moveDownRightBtn;
    Button* moveLeftBtn;
    Button* moveRightBtn;
    Button* moveUpBtn;
    Button* moveUpLeftBtn;
    Button* moveUpRightBtn;
    TextField* numMovesFld;
    TextField* numPushesFld;
    TextField* numRunsFld;
    TextField* numFocusFld;
    Slider* playbackSlider;
    Button* redoMoveBtn;
    Button* redoPushBtn;
    TextField* solvedFld;
    Button* stopBtn;
    Button* undoMoveBtn;
    Button* undoPushBtn;
    SokoWindow* window;
    SokoPuzzle puzzle;
    BOOL nextPuzzleExists;
    int recordMoves;
    int recordPushes;
    int recordRuns;
    int recordFocus;
    }

// Public Interface -----------------------------------------------------------
+ (BOOL)openPuzzle:(char const*)path;
+ (BOOL)openDefaultPuzzle;
+ (BOOL)choosePuzzleFromDirectory:(char const*)dir;
+ (void)newPuzzle;
+ (void)openPuzzle;
+ (void)saveAllPuzzles;

// Nib Interface --------------------------------------------------------------
- (id)animateToggled:(id)sender;
- (id)inspectSlider:(id)sender;
- (id)directionPressed:(id)sender;
- (id)diagonalPressed:(id)sender;
- (id)stopPressed:(id)sender;
- (id)redoMove:(id)sender;
- (id)undoMove:(id)sender;
- (id)redoPush:(id)sender;
- (id)undoPush:(id)sender;
- (id)openNextPuzzle:(id)sender;
- (id)save:(id)sender;
- (id)saveAs:(id)sender;
- (BOOL)validateMenuItem:(MenuCell*)item;

// Delegate Interface ---------------------------------------------------------
- (void)recordMoves:(int)m pushes:(int)p runs:(int)r focus:(int)f;
- (void)recordLevel:(int)level;
- (void)setSolved:(BOOL)solved;
- (void)setDirty:(BOOL)dirty;
- (void)refreshControls;
- (void)refreshAllCells;
- (void)refreshCells:(SokoCell const*)cells count:(int)ncells;
- (BOOL)shouldAnimatePlayback;
- (int)playbackThreshold;
- (void)beginPlayback;
- (void)endPlayback;
- (void)beginAnimation;
- (void)endAnimation;
- (void)beginDrag;
- (void)endDrag;
- (void)processEvents:(BOOL)awaitEvent;

@end

#endif // SokoBoard_h
