//-----------------------------------------------------------------------------
// SokoApp.m
//
//	Application delegate for SokoSave.
//
// Copyright (c), 1997,2001, Eric Sunshine <sunshine@sunshineco.com>
// Copyright (c), 1997, Paul McCarthy <zarnuk@high-speed-software.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoApp.m,v 1.4 2001/12/23 18:12:11 sunshine Exp $
// $Log: SokoApp.m,v $
// Revision 1.4  2001/12/23 18:12:11  sunshine
// v15
// -*- Extracted core game logic out of GUI code and generalized it so that
//     the same core code can be used by any platform.  Logic from
//     SokoBoard.m now resides in SokoPuzzle.c, etc.
//
// -*- Rewrote the NextStep port so that it utilizes the new "sokocore" game
//     library rather than implementing that logic directly.
//
// -*- Replaced the terminology "maze" with "puzzle" throughout the project,
//     including source code, documentation, and all user-visible UI
//     elements.  The only remaining place where "maze" is still used is in
//     the file extension ".sokomaze".  I haven't decided what, if anything,
//     to do about that, yet.
//
// -*- Converted the documentation from RTFD format to HTML format.
//     Documentation is no longer embedded in a scrollable text object on the
//     Info panel.  Instead, a "Help" menu item now displays the help file
//     via a web browser.
//
// -*- Added a "Help" menu item which opens the HTML help file in a web
//     browser.  Previously, the help information was contained in an RTFD
//     file and loaded directly into a text object on the Info panel.
//
// -*- Added SokoSetting implementation which provides a platform-independent
//     API for accessing user settings and well-known paths, such as
//     $(SokoSave) and $(SokoUser).
//
// -*- Changed the logic used to locate the puzzle which is opened
//     automatically at program launch time.  Previously, it simply opened a
//     .sokomaze file with the level number recorded as the "highest" level.
//     Unfortunately, if the user then tried to open a saved game file with
//     the same level number, the program would not actually open the saved
//     game because it thought that the game was already open.  This made it
//     very difficult for the user to continue a saved game which matched the
//     "highest" level.  The user first has to close the "new" game and then
//     open the "saved" game.  This problem was even further complicated on
//     the Windows platform where closing the "new" game would cause the
//     program to terminate if that was the only menu-bearing window.  Since
//     it is probable that a user may save a game before actually arriving at
//     the solution with the intention of returning to that game later, the
//     program now makes it much easier to return to the saved game.  Now, at
//     launch time, the program first checks for a "saved" game matching the
//     "highest" level and open that if present.  If not, then it falls back
//     to opening a new game.
//
// -*- For safety, user defaults (including score panel column order and
//     widths) are now committed immediately, rather than at termination
//     time.
//
// Revision 1.3  2001/08/19 10:20:28  sunshine
// v11
// -*- Converted from Objective-C++ to pure Objective-C.  Changed file
//     extensions from .M to .m, and .cc to .c.  This change makes it easier
//     to support both GNUstep and MacOS/X, neither of which feature an
//     Objective-C++ compiler (though MacOS/X will reportedly support
//     Objective-C++ at some point in the future).
//
// -*- Eliminated all inclusions of <appkit/appkit.h> throughout the project;
//     replaced with inclusion of individual header files.
//
// -*- The static `checked' variable in SokoApp.m is now an instance variable
//     of the SokoApp class and has been renamed to `autoOpenPuzzle'.  The
//     static `LAUNCHED_WITH_FILE' variable has been removed.  Its function is
//     now handled by checking if the SokoBoard class has any open mazes.
//
// -*- Implemented menu item validation machinery.  All menu items are now
//     enabled and disabled as appropriate.  The core machinery is implemented
//     by SokoApp.  SokoBoard implements validation logic for menu items which
//     are associated with the board's state.
//
// -*- Upgraded formatting of source and header files which I had not updated
//     in the last version.  Added explicit (id) in several places which had
//     not previously been updated.
//-----------------------------------------------------------------------------
#import "SokoApp.h"
#import	"SokoPool.h"
#import	"SokoBoard.h"
#import "SokoFile.h"
#import "SokoInfo.h"
#import "SokoPref.h"
#import "SokoPuzzle.h"
#import	"SokoScores.h"
#import "SokoSetting.h"
#import "SokoUtil.h"
#import <appkit/Application.h>
#import <appkit/Matrix.h>
#import	<appkit/Menu.h>
#import <appkit/MenuCell.h>
#import <appkit/workspaceRequest.h>
#import <objc/NXBundle.h>
#import	<libc.h>

#define NEW_PLAYER_SETTING "NewPlayer"
#define SOKO_HELP_DIR "help"
#define SOKO_HELP_FILE "SokoHelp.html"

@interface Object(SokoApp)
- (BOOL)validateMenuItem:(MenuCell*)item;
@end

@implementation SokoApp

- (id)new:(id)sender               { [SokoBoard newPuzzle];      return self; }
- (id)open:(id)sender              { [SokoBoard openPuzzle];     return self; }
- (id)saveAll:(id)sender           { [SokoBoard saveAllPuzzles]; return self; }
- (id)launchInfo:(id)sender        { [SokoInfo launch];          return self; }
- (id)launchPreferences:(id)sender { [SokoPref launch];          return self; }
- (id)launchScores:(id)sender      { [SokoScores launch];        return self; }
- (BOOL)appAcceptsAnotherFile:(id)sender { return YES; }

//-----------------------------------------------------------------------------
// init
//-----------------------------------------------------------------------------
- (id)init
    {
    [super init];
    autoOpenPuzzle = YES;
    return self;
    }


//-----------------------------------------------------------------------------
// menuItemValidator:
//	Simulate OpenStep-style menu-item validation.
//
//	This method is installed as "update action" for each menu item in the
//	application's menu (see -[MenuCell updateAction]).
//
//	First attempts to find an object which responds to the item's action
//	message.  This is done by initially checking the item's target.  If the
//	target is non-nil, then the referenced object is used.  Otherwise, the
//	following objects are tested in order to see if they respond to the
//	action message: key window's responder chain (starting with first
//	responder), key window itself, key window's delegate, main window's
//	responder chain, main window itself, main window's delegate, the
//	Application object, and finally the Application object's delegate.
//
//	If none of these objects respond to the action message, then the item
//	is disabled.  If an object does respond to the action message, a check
//	is made to see if it also responds to -validateMenuItem:.  If
//	-validateMenuItem: is not implemented, then the item is enabled.  If
//	it is implemented, then the return value of -validateMenuItem:
//	indicates whether the item should be enabled or disabled.
//
//	The boolean value returned from this method tells the parent menu
//	whether or not the menu cell should be redrawn (see -[MenuCell
//	setUpdateAction:]).
//-----------------------------------------------------------------------------
- (BOOL)menuItemValidator:(id)item
    {
    BOOL enable, redraw;
    id target = [item target];
    if (target == 0)
	target = [NXApp calcTargetForAction:[item action]];

    enable = YES;
    if (target == 0)
	enable = NO;
    else if ([target respondsTo:@selector(validateMenuItem:)])
	enable = [target validateMenuItem:item];

    redraw = (enable != [item isEnabled]);
    if (redraw)
	[item setEnabled:enable];
    return redraw;
    }


//-----------------------------------------------------------------------------
// initializeMenu:
//	Initialize automated enabling/disabling of menu cells.
//-----------------------------------------------------------------------------
- (void)initializeMenu:(Menu*)menu
    {
    int i;
    List* cells = [[menu itemList] cellList];
    for (i = [cells count] - 1; i >= 0; i--)
    	{
	MenuCell* cell = [cells objectAt:i];
	if ([cell hasSubmenu])
	    [self initializeMenu:[cell target]];
	else
	    [cell setUpdateAction:@selector(menuItemValidator:) forMenu:menu];
	}
    }


//-----------------------------------------------------------------------------
// appWillInit:
//-----------------------------------------------------------------------------
- (id)appWillInit:(id)sender
    {
    [self initializeMenu:[sender mainMenu]];
    [sender setAutoupdate:YES];
    [SokoPref startup];
    return self;
    }


//-----------------------------------------------------------------------------
// choosePuzzle
//	At application launch time, give user a chance to explicitly open a
//	puzzle or saved game if SokoBoard was unable to automatically open
//	the "default" puzzle.
//-----------------------------------------------------------------------------
- (BOOL)choosePuzzle
    {
    SokoPool pool = SokoPool_new(0);
    BOOL ok = [SokoBoard choosePuzzleFromDirectory:
	soko_expand_path(pool, soko_get_puzzle_directory(pool))];
    SokoPool_destroy( pool );
    return ok;
    }


//-----------------------------------------------------------------------------
// activate:
//-----------------------------------------------------------------------------
- (void)activate:(BOOL)puzzleRequired
    {
    BOOL ok = (SokoPuzzle_open_puzzle_count() > 0);
    if (!ok)
	ok = [SokoBoard openDefaultPuzzle];

    if (!ok && puzzleRequired)
	ok = [self choosePuzzle];

    if (!ok && puzzleRequired)
	[NXApp terminate:self];
    else
	{
	SokoSetting settings = SokoSetting_new(0);
	if (SokoSetting_get_bool( settings, NEW_PLAYER_SETTING, soko_true ))
	    {
	    SokoSetting_set_bool( settings, NEW_PLAYER_SETTING, soko_false );
	    [self launchInfo:self];
	    }
	SokoSetting_destroy( settings );
	}
    }


//-----------------------------------------------------------------------------
// appDidBecomeActive:
//-----------------------------------------------------------------------------
- (id)appDidBecomeActive:(id)sender
    {
    if (autoOpenPuzzle)		// Only do this the first time the 
	{			// application becomes active.
	autoOpenPuzzle = NO;
	[self activate:NO];
	}
    return self;
    }


//-----------------------------------------------------------------------------
// appWillTerminate:
//-----------------------------------------------------------------------------
- (id)appWillTerminate:(id)sender
    {
    [SokoPref shutdown];
    return self;
    }


//-----------------------------------------------------------------------------
// app:openFile:type:
//-----------------------------------------------------------------------------
- (int)app:(id)sender openFile:(char const*)path type:(char const*)type
    {
    return [SokoBoard openPuzzle:path];
    }


//-----------------------------------------------------------------------------
// launchInstructions:
//-----------------------------------------------------------------------------
- (id)launchInstructions:(id)sender
    {
    SokoPool pool = SokoPool_new(0);
    char* dir = SokoPool_allocate( pool, FILENAME_MAX + 1 );
    if ([[NXBundle mainBundle] getPath:dir forResource:SOKO_HELP_DIR ofType:0])
	{
	char const* path = soko_add_path_component(pool, dir, SOKO_HELP_FILE);
	if (soko_path_exists( pool, path ))
	    [[Application workspace] openFile:path];
	}
    SokoPool_destroy( pool );
    return self;
    }


//-----------------------------------------------------------------------------
// validateMenuItem:
//-----------------------------------------------------------------------------
- (BOOL)validateMenuItem:(MenuCell*)item
    {
    if ([item action] == @selector(saveAll:))
	return (SokoPuzzle_open_puzzle_count() > 0);
    return YES;
    }

@end
