//-----------------------------------------------------------------------------
// SokoUtil.c
//
//	Utility routines for SokoSave.
//
// Copyright (c), 2001,2002, Eric Sunshine <sunshine@sunshineco.com>
// Copyright (c), 1997, Paul McCarthy <zarnuk@high-speed-software.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoUtil.c,v 1.4 2002/01/29 18:20:02 sunshine Exp $
// $Log: SokoUtil.c,v $
// Revision 1.4  2002/01/29 18:20:02  sunshine
// v17:
// -*- Added soko_extension_part() to SokoUtil.
//
// -*- Added soko_read_line_terminator() to SokoUtil.  Unlike
//     soko_read_line() which always null-terminates the returned string even
//     if the input line would have overflowed the buffer, this function
//     allows control over the termination.
//
// Revision 1.3  2001/12/23 00:21:47  sunshine
// v15
// -*- Extracted core game logic out of GUI code and generalized it so that
//     the same core code can be used by any platform.  Logic from
//     SokoBoard.m now resides in SokoPuzzle.c, etc.
//
// -*- Augmented all input/output logic so that it now deals gracefully with
//     line terminators from all common platforms; Unix (LF), Macintosh (CR),
//     and Windows/DOS (CRLF).
//
// -*- Added soko_collapse() to SokoUtil.  This function trims leading and
//     trailing whitespace from a string and collapses all internal
//     whitespace to a single space.
//
// -*- Added soko_trim(), soko_ltrim() and soko_rtrim() to SokoUtil.
//
// -*- Added portable soko_stricmp() and soko_strnicmp() to replace
//     non-portable use of strcasecmp() and/or stricmp().
//
// -*- Renamed soko_starts_with() and soko_ends_with() to soko_has_prefix()
//     and soko_has_suffix(), respectively.  Also added an input flag which
//     controls case-sensitivity of these functions.
//
// -*- Added soko_normalize_path() and soko_denormalize_path() to SokoUtil.
//     All path manipulation functions now utilize these functions in order
//     to ensure correctness of path parsing and composition on all
//     platforms.
//
// -*- Moved soko_filename_part(), soko_directory_part(), soko_basename(),
//     and soko_mkdir() from SokoFile to SokoUtil.  Augmented these functions
//     so that they work correctly with pathnames from Unix, Macintosh, and
//     Windows (including Microsoft's UNC-style pathnames).
//
// -*- Added soko_add_path_component(), soko_add_path_extension(),
//     soko_replace_path_extension(), soko_remove_path_extension(), and
//     soko_path_exists() to SokoUtil.  These functions deal correctly with
//     pathnames from Unix, Macintosh, and Windows (including Microsoft's
//     UNC-style pathnames).
//
// -*- Added soko_readline() to SokoUtil.  This function is a replacement for
//     fgets().  It correctly handles Unix (LF), Macintosh (CR), and
//     Windows/DOS (CRLF) line terminators.
//-----------------------------------------------------------------------------
#include "SokoPool.h"
#include "SokoUtil.h"
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

#if defined(SOKO_PLATFORM_WIN32)
#  if defined(SOKO_BUILD_ENV_BORLAND)
#    include <dir.h>
#  else
#    include <direct.h>
#  endif
#  include <fcntl.h>
#  include <io.h>
#  if !defined(O_BINARY) && defined(_O_BINARY)
#    define O_BINARY _O_BINARY
#  endif
#  define MKDIR(PATH,MODE) mkdir(PATH)
#  define SET_BINARY_MODE(F) setmode(fileno(F), O_BINARY)
#else
#  if defined(SOKO_PLATFORM_NEXT)
#    include <libc.h>
#  endif
#  define MKDIR(PATH,MODE) mkdir(PATH,MODE)
#  define SET_BINARY_MODE(F) do {} while (0)
#endif

// Assume that STDIO functions on Macintosh will understand Unix-style paths
// even on an HFS volume, thus this is not defined.
#undef SOKO_USE_HFS_SYTLE_PATH

#if defined(SOKO_PLATFORM_WIN32)
#  define SOKO_PATH_SEP '\\'
#elif defined(SOKO_PLATFORM_MACINTOSH) && defined(SOKO_USE_HFS_SYTLE_PATH)
#  define SOKO_PATH_SEP ':'
#else
#  define SOKO_PATH_SEP '/'
#endif

//-----------------------------------------------------------------------------
// soko_strdup
//-----------------------------------------------------------------------------
char* soko_strdup( char const* s )
    {
    char *p;
    if (s == 0) s = "";
    p = (char*) malloc( strlen(s) + 1 );
    strcpy( p, s );
    return p;
    }


//-----------------------------------------------------------------------------
// soko_collapse
//	Trim all leading and trailing whitespace, and collapse internal
//	whitespace to a single space character.
//-----------------------------------------------------------------------------
char const* soko_collapse( cSokoPool pool, char const* s )
    {
    char* buff = SokoPool_allocate( pool, strlen(s) + 1 );
    char* dest = buff;
    char* graphic = dest;
    while (*s != '\0' && isspace(*s))
	s++;
    if (*s != '\0')
	{
	soko_bool whitespace = soko_false;
	for ( ; *s != '\0'; s++)
	    {
	    if (!isspace(*s))
		{
		*dest++ = *s;
		graphic = dest;
		whitespace = soko_false;
		}
	    else if (!whitespace)
		{
		*dest++ = ' ';
		whitespace = soko_true;
		}
	    }
	}
    *graphic = '\0';
    return buff;
    }


//-----------------------------------------------------------------------------
// soko_trim
//	Trim all leading and trailing whitespace.
//-----------------------------------------------------------------------------
char const* soko_trim( cSokoPool pool, char const* s )
    {
    char* buff = SokoPool_allocate( pool, strlen(s) + 1 );
    char* dest = buff;
    char* graphic = dest;
    while (*s != '\0' && isspace(*s))
	s++;
    for ( ; *s != '\0'; s++)
	{
	*dest++ = *s;
	if (!isspace(*s))
	    graphic = dest;
	}
    *graphic = '\0';
    return buff;
    }


//-----------------------------------------------------------------------------
// soko_ltrim
//	Trim all leading whitespace.
//-----------------------------------------------------------------------------
char const* soko_ltrim( cSokoPool pool, char const* s )
    {
    while (*s != '\0' && isspace(*s))
	s++;
    return SokoPool_store( pool, s );
    }


//-----------------------------------------------------------------------------
// soko_rtrim
//	Trim all trailing whitespace.
//-----------------------------------------------------------------------------
char const* soko_rtrim( cSokoPool pool, char const* s )
    {
    char* buff = SokoPool_allocate( pool, strlen(s) + 1 );
    char* dest = buff;
    char* graphic = dest;
    for ( ; *s != '\0'; s++)
	{
	*dest++ = *s;
	if (!isspace(*s))
	    graphic = dest;
	}
    *graphic = '\0';
    return buff;
    }


//-----------------------------------------------------------------------------
// soko_stricmp
//-----------------------------------------------------------------------------
int soko_stricmp( char const* s1, char const* s2 )
    {
    char c1, c2;
    int delta;
    do
	{
	c1 = *s1; c2 = *s2;
	if ((c1 & 0x80) == 0 && isupper(c1)) c1 = (char)tolower(c1);
	if ((c2 & 0x80) == 0 && isupper(c2)) c2 = (char)tolower(c2);
	delta = c1 - c2;
	if (delta != 0)
	    return delta;
	s1++; s2++;
	} while (c1 != '\0');
    return 0;
    }


//-----------------------------------------------------------------------------
// soko_strnicmp
//-----------------------------------------------------------------------------
int soko_strnicmp( char const* s1, char const* s2, size_t n )
    {
    if (n > 0)
	{
	char c1, c2;
	int delta;
	do
	    {
	    c1 = *s1; c2 = *s2;
	    if ((c1 & 0x80) == 0 && isupper(c1)) c1 = (char)tolower(c1);
	    if ((c2 & 0x80) == 0 && isupper(c2)) c2 = (char)tolower(c2);
	    delta = c1 - c2;
	    if (delta != 0)
		return delta;
	    s1++; s2++; n--;
	    } while (n > 0 && c1 != '\0');
	}
    return 0;
    }


//-----------------------------------------------------------------------------
// soko_has_suffix
//-----------------------------------------------------------------------------
soko_bool soko_has_suffix(
    char const* str, char const* sfx, soko_bool ignore_case )
    {
    soko_bool ok = soko_false;
    int const str_len = strlen( str );
    int const sfx_len = strlen( sfx );
    if (str_len >= sfx_len)
	{
	int rc;
	if (ignore_case)
	    rc = soko_strnicmp( str + str_len - sfx_len, sfx, sfx_len );
	else
	    rc = strncmp( str + str_len - sfx_len, sfx, sfx_len );
	ok = (rc == 0);
	}
    return ok;
    }


//-----------------------------------------------------------------------------
// soko_has_prefix
//-----------------------------------------------------------------------------
soko_bool soko_has_prefix(
    char const* str, char const* pfx, soko_bool ignore_case )
    {
    soko_bool ok = soko_false;
    int const str_len = strlen( str );
    int const pfx_len = strlen( pfx );
    if (str_len >= pfx_len)
	{
	int rc;
	if (ignore_case)
	    rc = soko_strnicmp( str, pfx, pfx_len );
	else
	    rc = memcmp( str, pfx, pfx_len );
	ok = (rc == 0);
	}
    return ok;
    }


//-----------------------------------------------------------------------------
// soko_basename
//-----------------------------------------------------------------------------
char const* soko_basename( cSokoPool pool, char const* path, char const* ext )
    {
    char* p;
    char* const buff = SokoPool_allocate( pool, FILENAME_MAX );
    strcpy( buff, path );
    p = strrchr( buff, '.' );
    if (*ext == '.') ext++;
    if (p != 0 && soko_stricmp( p + 1, ext ) == 0)
	*p = 0;
    return buff;
    }


//-----------------------------------------------------------------------------
// soko_extension_part
//-----------------------------------------------------------------------------
char const* soko_extension_part( cSokoPool pool, char const* path )
    {
    char const* f = soko_filename_part( pool, path );
    char const* p = strrchr( f, '.' );
    return (p != 0 ? p + 1 : "" );
    }


//-----------------------------------------------------------------------------
// soko_filename_part
//-----------------------------------------------------------------------------
char const* soko_filename_part( cSokoPool pool, char const* path )
    {
    char const* n = soko_normalize_path( pool, path );
    char const* p = strrchr( n, '/' );
    return soko_denormalize_path( pool, p != 0 ? p + 1 : n );
    }


//-----------------------------------------------------------------------------
// soko_directory_part
//	Returned value excludes the trailing path separator.
//-----------------------------------------------------------------------------
char const* soko_directory_part( cSokoPool pool, char const* path )
    {
    char const* n = soko_normalize_path( pool, path );
    char const* p = strrchr( n, '/' );
    if (p == 0)
	path = ".";
    else
	{
	int const len = p - n;
	if (len == 0)
	    path = "/";
	else
	    {
	    char* buff = SokoPool_allocate( pool, FILENAME_MAX );
	    memcpy( buff, n, len );
	    buff[len] = '\0';
	    path = buff;
	    }
	path = soko_denormalize_path( pool, path );
	}
    return path;
    }


//-----------------------------------------------------------------------------
// soko_add_path_component
//-----------------------------------------------------------------------------
char const* soko_add_path_component(
    cSokoPool pool, char const* path, char const* cmp )
    {
    char const* n = soko_normalize_path( pool, path );
    char const* c = soko_normalize_path( pool, cmp  );
    int const z = strlen(n);
    char* buff = SokoPool_allocate( pool, FILENAME_MAX );
    strcpy( buff, n );
    if (z > 0 && buff[z-1] != '/')
	strcat( buff, "/" );
    strcat( buff, c );
    return soko_denormalize_path( pool, buff );
    }


//-----------------------------------------------------------------------------
// soko_add_path_extension
//-----------------------------------------------------------------------------
char const* soko_add_path_extension( 
    cSokoPool pool, char const* path, char const* ext )
    {
    char* buff = SokoPool_allocate( pool, FILENAME_MAX );
    strcpy( buff, path );
    if (*ext != '.')
	strcat( buff, "." );
    strcat( buff, ext );
    return buff;
    }


//-----------------------------------------------------------------------------
// soko_replace_path_extension
//-----------------------------------------------------------------------------
char const* soko_replace_path_extension(
    cSokoPool pool, char const* path, char const* ext )
    {
    return soko_add_path_extension(
	pool, soko_remove_path_extension(pool, path), ext );
    }


//-----------------------------------------------------------------------------
// soko_remove_path_extension
//-----------------------------------------------------------------------------
char const* soko_remove_path_extension( cSokoPool pool, char const* path )
    {
    char const* filename = soko_filename_part( pool, path );
    if (strrchr( filename, '.' ) != 0)
	{
	int const n = strrchr( path, '.' ) - path;
	char* buff = SokoPool_allocate( pool, FILENAME_MAX );
	strncpy( buff, path, n );
	buff[n] = '\0';
	path = buff;
	}
    return path;
    }


//-----------------------------------------------------------------------------
// soko_normalize_path
//	Unix
//	    /dir/foo --> /dir/foo            [absolute]
//	    dir/foo --> dir/foo              [relative]
//	Windows/DOS
//	    C:\foo --> /c/foo                [absolute]
//	    C:/foo --> /c/foo                [absolute]
//	    \\host\foo --> //host/foo        [absolute]
//	    dir\foo --> dir/foo              [relative]
//	Macintosh/HFS
//	    volume:foo --> /volume/foo       [absolute]
//	    :dir:foo --> dir/foo             [relative]
//-----------------------------------------------------------------------------
char const* soko_normalize_path( cSokoPool pool, char const* path )
    {
    soko_bool windows_path = soko_false;
    int const n = strlen( path );
    char* buff = SokoPool_allocate( pool, FILENAME_MAX );
    char* buffp = buff;
    char c;

    if (n >= 2) // Check for Windows/DOS "C:\", "C:/", and "\\host\".
	{
	char c1 = path[0], c2 = path[1];
	if (c1 == '\\' && c2 == '\\')
	    windows_path = soko_true;
	else if (n >= 3)
	    {
	    char c3 = path[2];
	    if (c2 == ':' && (c3 == '\\' || c3 == '/') &&
		(c1 & 0x80) == 0 && isalpha(c1))
		{
		*buffp++ = '/'; // "C:\" becomes "/c/".
		*buffp++ = (char)tolower(c1);
		path += 2; // Advance past "C:".
		windows_path = soko_true;
		}
	    }
	}

    if (!windows_path && strchr( path, ':' ) != 0) // Check for Macintosh/HFS.
	{ // strlen(path) >= 1 guaranteed if we get this far.
	if (path[0] == ':') // Relative path.
	    path++; // Advance past ":".
	else // Absolute path.
	    *buffp++ = '/'; // "volume:foo" becomes "/volume/foo".
	}

    for (c = *path++; c != '\0'; c = *path++)
	*buffp++ = (c == '\\' || c == ':' ? (char)'/' : c);
    *buffp = '\0';

    if (buffp - buff >= 2 && buff[0] == '.' && buff[1] == '/')
	buff += 2; // "./foo" becomes "foo".

    return buff;
    }


//-----------------------------------------------------------------------------
// soko_denormalize_path
//	Unix
//	    /dir/foo --> /dir/foo            [absolute]
//	    //dir/foo --> /dir/foo           [absolute]
//	    dir/foo --> dir/foo              [relative]
//	Windows/DOS
//	    /c/foo --> c:\foo                [absolute]
//	    //host/foo --> \\host\foo        [absolute]
//	    dir/foo --> dir\foo              [relative]
//	Macintosh/HFS
//	    /volume/foo --> volume:foo       [absolute]
//	    //volume/foo --> volume:foo      [absolute]
//	    dir/foo --> :dir:foo             [relative]
//-----------------------------------------------------------------------------
char const* soko_denormalize_path( cSokoPool pool, char const* path )
    {
    int n = strlen( path );
    char* buff = SokoPool_allocate( pool, FILENAME_MAX );
    char* buffp = buff;
    char c;

#if !defined(SOKO_PLATFORM_WIN32)
    if (n >= 2 && path[0] == '/' && path[1] == '/')
	{
	path++; // "//" becomes "/" on platforms other than Windows/DOS.
	n--;
	}
#endif

#if defined(SOKO_PLATFORM_WIN32)
    if (n >= 3)
	{
	char c1 = path[0], c2 = path[1], c3 = path[2];
	if (c1 == '/' && c3 == '/' && (c2 & 0x80) == 0 && isalpha(c2))
	    {
	    *buffp++ = c2;  // "/c/foo" becomes "c:\foo".
	    *buffp++ = ':';
	    path += 2;
	    n -= 2;
	    }
	}
#elif defined(SOKO_USE_HFS_SYTLE_PATH)
    if (n >= 1)
	{
	if (path[0] != '/') // Relative path.
	    {
	    if (strchr( path, '/' ) != 0) // Not a bare filename.
		*buffp++ = ':'; // "dir/foo" becomes ":dir:foo".
	    }
	else // Absolute path.
	    {
	    path++; // "/volume/foo" becomes "volume:foo".
	    n--;
	    }
	}
#endif
    (void)n; // Pacify compiler about earlier assignments to 'n'.

    for (c = *path++; c != '\0'; c = *path++)
	*buffp++ = (c == '/' ? (char)SOKO_PATH_SEP : c);
    *buffp = '\0';

    return buff;
    }


//-----------------------------------------------------------------------------
// soko_mkdirs
//	Attempts to create all intermediate directories forming "path".
//	Computes leading portion of path which represents the root of the
//	filesystem (Unix, Windows/DOS, Macintosh/HFS).  If root portion of path
//	is present, it is skipped when creating intermediate directories, since
//	we can never actually "create" a root directory.  The following root
//	path portions are recognized:
//
//	Unix
//	    /
//	Windows/DOS
//	   c:\ 
//	   \\host\ 
//	   \ 
//	Macintosh/HFS
//	   volume:
//-----------------------------------------------------------------------------
soko_bool soko_mkdirs( cSokoPool pool, char const* path )
    {
    soko_bool ok = soko_false;
    if (path != 0 && strlen(path) != 0)
	{
	if (soko_path_exists( pool, path ))
	    ok = soko_true;			// Already exists.
	else
	    { // Advance past "root" portion of pathname.
	    int lim;
	    char const* rest;
	    char const* temp;
	    path = soko_denormalize_path( pool, path );
	    rest = path;
	    lim = strlen( path ); // Length of denormalized path.
	    if (lim >= 2 && rest[0]=='\\' && rest[1]=='\\') // "\\host\foo"
		{
		char c;
		rest += 2; // Advance past "\\host\"
		for (c=*rest; c != '\0' && c != '\\' && c != '/'; c=*++rest) {}
		if (c != '\0') rest++;
		}
	    else if (lim >= 3 && rest[1] == ':' && // "C:\" or "C:/"
		(rest[2] == '\\' || rest[2] == '/') &&
		((rest[0] & 0x80) == 0) && isalpha(rest[0]))
		{
		rest += 3; // Advance past "C:\".
		}
	    else if ((temp = strchr(rest, ':')) != 0) // "volume:foo"
		{
		if (temp != rest)    // Is an absolute path.
		    rest = temp + 1; // Advance past "volume:".
		}
	    else if (lim >= 1 && (rest[0] == '/' || rest[0] == '\\')) // "/foo"
		{
		rest++; // Advance past "/".
		}
    
	    if (rest - path == lim) // Path is just root portion.
		ok = soko_true;
	    else                    // More than root portion is present.
		{
		char buff[ FILENAME_MAX ];
		int len = -1, rc;
		soko_bool exists;
		if (rest != path) // If this is an absolute path.
		    {
		    char const* root;
		    strncpy( buff, path, rest - path );
		    buff[ rest - path ] = '\0';
		    root = soko_normalize_path( pool, buff );
		    len = strlen( root ) - 1; // Skip normalized root.
		    }
    
		strcpy( buff, soko_normalize_path(pool, path) );
		lim = strlen( buff ); // `lim' now length of normalized path.
		do  {
		    if (len >= 0)
			buff[len++] = '/';
		    while (len < lim && buff[len] != '/')
			len++;
		    buff[len] = 0;
		    exists = soko_path_exists( pool, buff );
		    }
		while (exists && len < lim);
	
		while ((rc = MKDIR(soko_denormalize_path(pool,buff),0777)) == 0
		    && len < lim)
		    {
		    buff[len++] = '/';
		    while (len < lim && buff[len] != '/')
			len++;
		    buff[len] = 0;
		    }
		if (rc == 0)
		    ok = soko_true;
		}
	    }
	}
    return ok;
    }


//-----------------------------------------------------------------------------
// soko_path_exists
//-----------------------------------------------------------------------------
soko_bool soko_path_exists( cSokoPool pool, char const* path )
    {
    struct stat st;
    return (stat( soko_denormalize_path(pool, path), &st ) == 0);
    }


//-----------------------------------------------------------------------------
// soko_read_line_terminate
//	Read a line from a file.  Does not include line terminator (LF, CR, or
//	CRLF) in result.  Always null-terminates the buffer if buffer is large
//	enough to hold input.  If input is too large for buffer, then only
//	terminates if explicitly requested.  Takes special care to strip line
//	terminator even in case when input line fits exactly into buffer.
// *NOTE-CR*
//	Unfortunately, Microsoft Windows eats bare CR terminators completely
//	when the file is opened in "text" mode.  This makes it impossible to
//	recognize Macintosh-style line termination.  Therefore, attempt to set
//	the file handle to "binary" mode.
//-----------------------------------------------------------------------------
int soko_read_line_terminate(
    FILE* fp, char* buffer, int buffer_size, soko_bool terminate )
    {
    int n = -1;
    int c;
    SET_BINARY_MODE(fp);	// *NOTE-CR*
    c = fgetc(fp);
    if (c != EOF)
	{
	int const sz = buffer_size - (terminate ? 1 : 0);
	n = 0;
	while (c != EOF)
	    {
	    if (c == '\n' || c == '\r') // Eat LF, CR, or CRLF.
		{
		if (c == '\r')
		    {
		    c = fgetc(fp);
		    if (c != '\n' && c != EOF)
			ungetc( c, fp );
		    }
		break;
		}
	    else
		{
		if (n < sz)
		    {
		    *buffer++ = (char)c;
		    n++;
		    c = fgetc(fp);
		    }
		else
		    {
		    ungetc( c, fp );
		    break;
		    }
		}
	    }
	if (n < buffer_size)
	    *buffer = '\0';
	}
    return n;
    }


//-----------------------------------------------------------------------------
// soko_read_line
//	Read a line from a file.  Does not include line terminator (LF, CR, or
//	CRLF) in result.  Always null-terminates the buffer even if the input
//	line would have overflown the buffer.  Takes special care to strip line
//	terminator even in case when input line fits exactly into buffer.
//-----------------------------------------------------------------------------
int soko_read_line( FILE* fp, char* buffer, int buffer_size )
    {
    return soko_read_line_terminate( fp, buffer, buffer_size, soko_true );
    }
