//-----------------------------------------------------------------------------
// SokoSetting.c
//
//	User-specific persistent settings.
//
// Copyright (c), 2001, Eric Sunshine <sunshine@sunshineco.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoSetting.c,v 1.1 2001/12/21 22:03:55 sunshine Exp $
// $Log: SokoSetting.c,v $
// Revision 1.1  2001/12/21 22:03:55  sunshine
// v15
// -*- Extracted core game logic out of GUI code and generalized it so that
//     the same core code can be used by any platform.  Logic from
//     SokoBoard.m now resides in SokoPuzzle.c, etc.
//
// -*- Added SokoSetting implementation which provides a platform-independent
//     API for accessing user settings and well-known paths, such as
//     $(SokoSave) and $(SokoUser).
//
//-----------------------------------------------------------------------------
#include "SokoPool.h"
#include "SokoSetting.h"
#include "SokoAssert.h"
#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>

#define DELEGATE(p) ((p)->data->delegate)

struct SokoSettingData
    {
    SokoSettingDelegate delegate;
    };

static SokoSettingDelegate DEFAULT_DELEGATE = 0;

//-----------------------------------------------------------------------------
// get
//-----------------------------------------------------------------------------
static char const* get( SokoSetting p, char const* name, char const* fallback,
    SokoPool pool )
    {
    char const* s = DELEGATE(p)->get_callback( p, DELEGATE(p), name, pool );
    if (s == 0)
	s = fallback;
    return s;
    }


//-----------------------------------------------------------------------------
// set
//-----------------------------------------------------------------------------
static void set( SokoSetting p, char const* name, char const* value )
    {
    DELEGATE(p)->set_callback( p, DELEGATE(p), name, value );
    }


//-----------------------------------------------------------------------------
// SokoSetting_set_delegate
//-----------------------------------------------------------------------------
void SokoSetting_set_delegate( SokoSettingDelegate d )
    {
    DEFAULT_DELEGATE = d;
    }


//-----------------------------------------------------------------------------
// SokoSetting_get_delegate
//-----------------------------------------------------------------------------
SokoSettingDelegate SokoSetting_get_delegate( void )
    {
    return DEFAULT_DELEGATE;
    }


//-----------------------------------------------------------------------------
// SokoSetting_new
//-----------------------------------------------------------------------------
SokoSetting SokoSetting_new( void* info )
    {
    SokoSetting p;
    SOKO_ASSERT( DEFAULT_DELEGATE != 0 );
    p = (SokoSetting)malloc( sizeof(struct _SokoSetting) );
    p->data = (struct SokoSettingData*)malloc(sizeof(struct SokoSettingData));
    p->data->delegate = DEFAULT_DELEGATE;
    p->info = info;
    return p;
    }


//-----------------------------------------------------------------------------
// SokoSetting_destroy
//-----------------------------------------------------------------------------
void SokoSetting_destroy( SokoSetting p )
    {
    free(p->data);
    free(p);
    }


//-----------------------------------------------------------------------------
// SokoSetting_get_string
//-----------------------------------------------------------------------------
char const* SokoSetting_get_string(
    SokoSetting p, char const* name, char const* fallback, cSokoPool pool )
    {
    return get( p, name, fallback, pool );
    }


//-----------------------------------------------------------------------------
// SokoSetting_set_string
//-----------------------------------------------------------------------------
void SokoSetting_set_string(SokoSetting p, char const* name, char const* value)
    {
    set( p, name, value );
    }


//-----------------------------------------------------------------------------
// SokoSetting_get_bool
//-----------------------------------------------------------------------------
soko_bool SokoSetting_get_bool(
    SokoSetting p, char const* name, soko_bool fallback )
    {
    SokoPool pool = SokoPool_new(0);
    soko_bool value = fallback;
    char const* s = get( p, name, 0, pool );
    if (s != 0 && *s != '\0')
	{
	char const c = (char)toupper(*s);
	value = (c == 'Y' || c == 'T' || c == '1');
	}
    SokoPool_destroy( pool );
    return value;
    }


//-----------------------------------------------------------------------------
// SokoSetting_set_bool
//-----------------------------------------------------------------------------
void SokoSetting_set_bool( SokoSetting p, char const* name, soko_bool value )
    {
    set( p, name, value ? "Yes" : "No" );
    }


//-----------------------------------------------------------------------------
// SokoSetting_get_int
//-----------------------------------------------------------------------------
int SokoSetting_get_int( SokoSetting p, char const* name, int fallback )
    {
    SokoPool pool = SokoPool_new(0);
    int value = fallback;
    char const* s = get( p, name, 0, pool );
    if (s != 0 && *s != '\0')
	value = atoi(s);
    SokoPool_destroy( pool );
    return value;
    }


//-----------------------------------------------------------------------------
// SokoSetting_set_int
//-----------------------------------------------------------------------------
void SokoSetting_set_int( SokoSetting p, char const* name, int value )
    {
    char buff[ 32 ];
    sprintf( buff, "%d", value );
    set( p, name, buff );
    }


//-----------------------------------------------------------------------------
// SokoSetting_get_float
//-----------------------------------------------------------------------------
float SokoSetting_get_float( SokoSetting p, char const* name, float fallback )
    {
    SokoPool pool = SokoPool_new(0);
    float value = fallback;
    char const* s = get( p, name, 0, pool );
    if (s != 0 && *s != '\0')
	value = atof(s);
    SokoPool_destroy( pool );
    return value;
    }


//-----------------------------------------------------------------------------
// SokoSetting_set_float
//-----------------------------------------------------------------------------
void SokoSetting_set_float( SokoSetting p, char const* name, float value )
    {
    char buff[ 64 ];
    sprintf( buff, "%g", value );
    set( p, name, buff );
    }


//-----------------------------------------------------------------------------
// SokoSetting_exists
//-----------------------------------------------------------------------------
soko_bool SokoSetting_exists( SokoSetting p, char const* name )
    {
    return DELEGATE(p)->exists_callback( p, DELEGATE(p), name );
    }


//-----------------------------------------------------------------------------
// SokoSetting_remove
//-----------------------------------------------------------------------------
void SokoSetting_remove( SokoSetting p, char const* name )
    {
    if (SokoSetting_exists( p, name ))
	DELEGATE(p)->remove_callback( p, DELEGATE(p), name );
    }


//-----------------------------------------------------------------------------
// SokoSetting_factory_path
//-----------------------------------------------------------------------------
char const* SokoSetting_factory_path( SokoSetting p, cSokoPool pool )
    {
    return DELEGATE(p)->factory_path_callback( p, DELEGATE(p), pool );
    }


//-----------------------------------------------------------------------------
// SokoSetting_user_path
//-----------------------------------------------------------------------------
char const* SokoSetting_user_path( SokoSetting p, cSokoPool pool )
    {
    return DELEGATE(p)->user_path_callback( p, DELEGATE(p), pool );
    }
