//-----------------------------------------------------------------------------
// SokoScore.c
//
//	High score management for SokoSave.
//
// Copyright (c), 2001,2002, Eric Sunshine <sunshine@sunshineco.com>
// Copyright (c), 1997, Paul McCarthy <zarnuk@high-speed-software.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoScore.c,v 1.5 2002/01/29 18:37:29 sunshine Exp $
// $Log: SokoScore.c,v $
// Revision 1.5  2002/01/29 18:37:29  sunshine
// v17:
// -*- SokoPuzzle now calculates two new scores in addition to "moves" and
//     "pushes".  The "runs" score is the number of straight lines in which
//     crates have been pushed.  Looked at another way, it is the number of
//     turns crates have made while being pushed.  The "focus" score is the
//     number of times the player's focus has changed from one crate to
//     another.
//
// -*- SokoScore now records "runs" and "focus" scores in SCORES file.
//
// Revision 1.4  2001/12/23 00:28:56  sunshine
// v15
// -*- Extracted core game logic out of GUI code and generalized it so that
//     the same core code can be used by any platform.  Logic from
//     SokoBoard.m now resides in SokoPuzzle.c, etc.
//
// -*- Augmented all input/output logic so that it now deals gracefully with
//     line terminators from all common platforms; Unix (LF), Macintosh (CR),
//     and Windows/DOS (CRLF).
//
// -*- For safety, SCORES is now written out immediately after a new score is
//     recorded, rather than only at program termination time.
//
// -*- Replaced the terminology "maze" with "puzzle" throughout the project,
//     including source code, documentation, and all user-visible UI
//     elements.  The only remaining place where "maze" is still used is in
//     the file extension ".sokomaze".  I haven't decided what, if anything,
//     to do about that, yet.
//
// -*- SokoScore now trims and collapses strings before recording them in the
//     score table.
//-----------------------------------------------------------------------------
#include "SokoPool.h"
#include "SokoPuzzle.h"
#include "SokoScore.h"
#include "SokoAssert.h"
#include "SokoFile.h"
#include "SokoUtil.h"
#include <ctype.h>
#include <errno.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

struct SokoScoreData
    {
    SokoScoreDelegate delegate;
    SokoHighScore* scores;
    int* scores_sorted;
    int scores_count;
    int scores_max;
    SokoScoreSort criteria[ SOKO_SCORE_MAX ];
    int criteria_count;
    SokoPool pool;
    };

static SokoScoreSort const SOKO_CRITERIA_DEFAULT[] =
    {
    SOKO_SCORE_PUZZLE,
    SOKO_SCORE_MOVES,
    SOKO_SCORE_PUSHES,
    SOKO_SCORE_RUNS,
    SOKO_SCORE_FOCUS,
    SOKO_SCORE_DATE,
    SOKO_SCORE_PLAYER,
    SOKO_SCORE_NOTES
    };
#define SOKO_CRITERIA_DEFAULT_COUNT \
    (sizeof(SOKO_CRITERIA_DEFAULT) / sizeof(SOKO_CRITERIA_DEFAULT[0]))

static SokoHighScore const* SORT_SCORES = 0;
static SokoScoreSort const* SORT_CRITERIA = 0;
static int SORT_CRITERIA_COUNT = 0;

//-----------------------------------------------------------------------------
// send_alert
//-----------------------------------------------------------------------------
static void send_alert( SokoScore p, SokoAlert severity, char const* title,
    char const* fmt, ... )
    {
    SokoScoreDelegate d = p->data->delegate;
    va_list args;
    va_start( args, fmt );
    if (d == 0)
	vfprintf( stderr, fmt, args );
    else
	{
	char buff[ 1024 ];
	vsprintf( buff, fmt, args );
	SOKO_ASSERT( strlen(buff) < sizeof(buff) );
	d->alert_callback( p, d, severity, title, buff );
	}
    va_end( args );
    }


//-----------------------------------------------------------------------------
// score_cmp
//-----------------------------------------------------------------------------
static int score_cmp( void const* p1, void const* p2 )
    {
    int i;
    SokoHighScore const *r1, *r2;
    SOKO_ASSERT( SORT_SCORES != 0 );
    r1 = &SORT_SCORES[ *((int*)p1) ];
    r2 = &SORT_SCORES[ *((int*)p2) ];
    for (i = 0; i < SORT_CRITERIA_COUNT; i++)
	{
	int rc = 0;
	switch (SORT_CRITERIA[i])
	    {
	    case SOKO_SCORE_PUZZLE:
		{
		rc = soko_stricmp( r1->puzzle, r2->puzzle ); break;
		}
	    case SOKO_SCORE_MOVES:
		{
		int m1 = r1->moves, m2 = r2->moves;
		rc = (m1 < m2 ? -1 : (m1 > m2 ? 1 : 0));
		break;
		}
	    case SOKO_SCORE_PUSHES:
		{
		int p1 = r1->pushes, p2 = r2->pushes;
		rc = (p1 < p2 ? -1 : (p1 > p2 ? 1 : 0));
		break;
		}
	    case SOKO_SCORE_DATE:
		{
		unsigned long d1 = r1->date, d2 = r2->date;
		rc = (d1 < d2 ? -1 : (d1 > d2 ? 1 : 0));
		break;
		}
	    case SOKO_SCORE_PLAYER:
		{
		rc = soko_stricmp( r1->player, r2->player ); break;
		}
	    case SOKO_SCORE_NOTES:
		{
		rc = soko_stricmp( r1->notes, r2->notes ); break;
		}
	    case SOKO_SCORE_RUNS:
		{
		int n1 = r1->runs, n2 = r2->runs;
		rc = (n1 < n2 ? -1 : (n1 > n2 ? 1 : 0));
		break;
		}
	    case SOKO_SCORE_FOCUS:
		{
		int f1 = r1->focus, f2 = r2->focus;
		rc = (f1 < f2 ? -1 : (f1 > f2 ? 1 : 0));
		break;
		}
	    }
	if (rc != 0)
	    return rc;
	}
    return 0; // Items equal (even correct when SORT_CRITERIA_COUNT == 0).
    }


//-----------------------------------------------------------------------------
// sort_worker
//-----------------------------------------------------------------------------
static void sort_worker( SokoScore p, int* sorted,
    SokoScoreSort const* criteria, int criteria_count )
    {
    int const n = p->data->scores_count;
    if (n == 1)
	sorted[0] = 0;
    else if (n > 1)
	{
	int i;
	SORT_SCORES = p->data->scores;
	SORT_CRITERIA = criteria;
	SORT_CRITERIA_COUNT = criteria_count;
	for (i = n - 1; i >= 0; i--)
	    sorted[i] = i;
	qsort( sorted, n, sizeof(sorted[0]), score_cmp );
	SORT_SCORES = 0;
	SORT_CRITERIA = 0;
	SORT_CRITERIA_COUNT = 0;
	}
    }


//-----------------------------------------------------------------------------
// sort
//-----------------------------------------------------------------------------
static void sort( SokoScore p )
    {
    sort_worker( p, p->data->scores_sorted, p->data->criteria,
	p->data->criteria_count );
    }


//-----------------------------------------------------------------------------
// set_criteria
//-----------------------------------------------------------------------------
static void set_criteria(
    SokoScore p, SokoScoreSort const* criteria, int criteria_count )
    {
    int i;
    SOKO_ASSERT(criteria_count >= 0 && criteria_count <= SOKO_SCORE_MAX);
    SOKO_ASSERT(criteria_count == 0 || criteria != 0);
    p->data->criteria_count = criteria_count;
    for (i = 0; i < criteria_count; i++)
	p->data->criteria[i] = criteria[i];
    }


//-----------------------------------------------------------------------------
// open_for_read
// *NOTE-CR*
//	Unfortunately, Microsoft Windows eats bare CR terminators completely
//	when the file is opened in "text" mode.  This makes it impossible to
//	recognize Macintosh-style line termination.  Therefore, use "binary"
//	mode.
//-----------------------------------------------------------------------------
static FILE* open_for_read( SokoScore p )
    {
    FILE* fp;						// *NOTE-CR*
    SokoPool pool = SokoPool_new(0);
    if ((fp = fopen(soko_expand_path(pool, soko_get_score_file(pool)),
	"rb")) == 0)
	fp = fopen(soko_expand_path(pool, soko_get_factory_score_file(pool)),
	    "rb");
    SokoPool_destroy( pool );
    (void)p; // Silence compiler.
    return fp;
    }


//-----------------------------------------------------------------------------
// open_for_write
//-----------------------------------------------------------------------------
static FILE* open_for_write( SokoScore p )
    {
    FILE* fp;
    SokoPool pool = SokoPool_new(0);
    char const* filename = soko_expand_path(pool, soko_get_score_file(pool));
    if ((fp = fopen( filename, "w" )) == 0)
	{
	char const* const dir = soko_directory_part( pool, filename );
	if (!soko_mkdirs(pool, dir) || (fp = fopen(filename, "w")) == 0)
	    send_alert( p, SOKO_ALERT_ERROR, "Sorry",
		"Cannot create score file: \"%s\" [%s]\n",
		filename, strerror(errno) );
	}
    SokoPool_destroy( pool );
    return fp;
    }


//-----------------------------------------------------------------------------
// write_scores
//-----------------------------------------------------------------------------
static void write_scores( SokoScore p )
    {
    FILE* fp = open_for_write(p);
    if (fp != 0)
	{
	int i;
	int* sorted = (int*)malloc(p->data->scores_count * sizeof(sorted[0]));
	sort_worker(
	    p, sorted, SOKO_CRITERIA_DEFAULT, SOKO_CRITERIA_DEFAULT_COUNT );
	for (i = 0; i < p->data->scores_count; i++)
	    {
	    SokoHighScore const* r = p->data->scores + sorted[i];
	    fprintf( fp, "%s\t%d\t%d\t%lu\t%s\t%s\t%d\t%d\n",
		r->puzzle,
		r->moves,
		r->pushes,
		r->date,
		r->player,
		r->notes,
		r->runs,
		r->focus );
	    }
	free( sorted );
	fclose(fp);
	}
    }


//-----------------------------------------------------------------------------
// add_score
//-----------------------------------------------------------------------------
static void add_score(
    SokoScore p,
    char const* puzzle,
    int moves,
    int pushes,
    int runs,
    int focus,
    unsigned long date,
    char const* player,
    char const* notes )
    {
    SokoHighScore* r;
    char buff[ 128 ];
    time_t ts = (time_t)date;
    if (p->data->scores_count + 1 >= p->data->scores_max)
	{
	int i, j;
	p->data->scores_max += 32;
	i = p->data->scores_max * sizeof(p->data->scores[0]);
	j = p->data->scores_max * sizeof(p->data->scores_sorted[0]);
	if (p->data->scores == 0)
	    {
	    p->data->scores = (SokoHighScore*)malloc(i);
	    p->data->scores_sorted = (int*)malloc(j);
	    }
	else
	    {
	    p->data->scores = (SokoHighScore*)realloc(p->data->scores, i);
	    p->data->scores_sorted = (int*)realloc(p->data->scores_sorted, j);
	    }
	}
    strftime( buff, sizeof(buff), "%Y/%m/%d %H:%M:%S", localtime(&ts) );
    r = p->data->scores + p->data->scores_count;
    r->puzzle = SokoPool_store( p->data->pool, puzzle );
    r->moves = moves;
    r->pushes = pushes;
    r->runs = runs;
    r->focus = focus;
    r->date = date;
    r->date_formatted = SokoPool_store( p->data->pool, buff );
    r->player = SokoPool_store( p->data->pool, player );
    r->notes = SokoPool_store( p->data->pool, notes );
    p->data->scores_count++;
    }


//-----------------------------------------------------------------------------
// good_int
//-----------------------------------------------------------------------------
static soko_bool good_int( char const* t, int* p )
    {
    soko_bool ok = soko_false;
    if (t != 0)
	{
	unsigned char const* s = (unsigned char const*)t;
	while (*s != 0 && isspace(*s))		// Skip leading whitespace
	    s++;
	if (*s == 0)
	    {
	    *p = -1;
	    ok = soko_true;
	    }
	else
	    {
	    soko_bool negate = soko_false;
	    if (*s == '-')
		{
		negate = soko_true;
		s++;
		}
	    if ('0' <= *s && *s <= '9')
		{
		int x = 0;
		do  {
		    x = x * 10 + *s - '0';
		    s++;
		    }
		while ('0' <= *s && *s <= '9');
		if (x >= 0)			   // If no overflow
		    {
		    while (*s != 0 && isspace(*s)) // Skip trailing whitespace
			s++;
		    if (*s == 0)		   // Disallow trailing garbage
			{
			*p = x * (negate ? -1 : 1);
			ok = soko_true;
			}
		    }
		}
	    }
	}
    return ok;
    }


//-----------------------------------------------------------------------------
// read_scores
//-----------------------------------------------------------------------------
static void read_scores( SokoScore p )
    {
    FILE* fp = open_for_read(p);
    if (fp != 0)
	{
	int line = 0;
	char buff[ 2048 ];
	while (soko_read_line( fp, buff, sizeof(buff) ) != -1)
	    {
	    unsigned long date;
	    int i, moves, pushes, runs, focus;
	    char* fld[ SOKO_SCORE_MAX ];
	    char* s;

	    line++;
	    s = buff;			// Break into fields at tabs.
	    for (i = 0; i < SOKO_SCORE_MAX; i++)
		{
		fld[i] = s;
		while (*s != 0 && *s != '\t')
		    s++;
		if (*s != 0)
		    *s++ = 0;
		}

	    if (good_int( fld[SOKO_SCORE_MOVES ], &moves      ) &&
		good_int( fld[SOKO_SCORE_PUSHES], &pushes     ) &&
		good_int( fld[SOKO_SCORE_DATE  ], (int*)&date ) &&
		good_int( fld[SOKO_SCORE_RUNS  ], &runs       ) &&
		good_int( fld[SOKO_SCORE_FOCUS ], &focus      ))
		{
		add_score( p, fld[SOKO_SCORE_PUZZLE], moves, pushes, runs,
		    focus, date, fld[SOKO_SCORE_PLAYER],fld[SOKO_SCORE_NOTES]);
		}
	    else
		{
		SokoPool pool = SokoPool_new(0);
		send_alert( p, SOKO_ALERT_WARNING, "Corrupt Score",
		    "Format error in scores file at line %d: %s",
		    line, soko_expand_path(pool, soko_get_score_file(pool)) );
		SokoPool_destroy( pool );
		}
	    }

	fclose(fp);
	}
    sort(p);
    }


//-----------------------------------------------------------------------------
// SokoScore_score_count
//-----------------------------------------------------------------------------
int SokoScore_score_count( SokoScore p )
    {
    return p->data->scores_count;
    }


//-----------------------------------------------------------------------------
// SokoScore_get_score
//-----------------------------------------------------------------------------
SokoHighScore const* SokoScore_get_score( SokoScore p, int n )
    {
    SOKO_ASSERT( n >= 0 );
    SOKO_ASSERT( n < p->data->scores_count );
    return p->data->scores + p->data->scores_sorted[n];
    }


//-----------------------------------------------------------------------------
// SokoScore_get_all_scores_unsorted
//-----------------------------------------------------------------------------
SokoHighScore const* SokoScore_get_all_scores_unsorted( SokoScore p, int* n )
    {
    *n = p->data->scores_count;
    return (*n == 0 ? 0 : p->data->scores);
    }


//-----------------------------------------------------------------------------
// SokoScore_sort
//-----------------------------------------------------------------------------
void SokoScore_sort(
    SokoScore p, SokoScoreSort const* criteria, int criteria_count )
    {
    set_criteria( p, criteria, criteria_count );
    sort(p);
    }


//-----------------------------------------------------------------------------
// SokoScore_add
//	Returns sorted index of new entry.  Clients which, instead, access the
//	unsorted list can find the new entry at position (score_count - 1).
// *COLLAPSE*
//	Since the scores file is delimited by tabs and newlines, these
//	characters must not appear in user-entered fields such as "player" and
//	"notes".  In addition to collapsing whitespace, soko_collapse() also
//	converts tabs and newlines to spaces.
//-----------------------------------------------------------------------------
int SokoScore_add( SokoScore p, cSokoPuzzle puzzle, int moves, int pushes,
    int runs, int focus, char const* player, char const* notes )
    {
    int i, new_pos;
    unsigned long date = time(0);
    SokoPool pool = SokoPool_new(0);

    // Recorded puzzle name is basename of original path sans file extension.
    char const* puzzle_name = soko_remove_path_extension( pool,
	soko_filename_part(pool,SokoPuzzle_get_puzzle_file_name(puzzle,pool)));

    add_score( p,
	soko_collapse(pool, puzzle_name),
	moves,
	pushes,
	runs,
	focus,
	date,
	soko_collapse(pool, player),	// *COLLAPSE*
	soko_collapse(pool, notes) );

    SokoPool_destroy( pool );

    write_scores(p);
    sort(p);

    new_pos = p->data->scores_count - 1; // Physical location of new record.
    for (i = new_pos; i >= 0; i--)       // Find new record in sorted array.
	if (new_pos == p->data->scores_sorted[i])
	    break;
    SOKO_ASSERT( i >= 0 );
    return i;
    }


//-----------------------------------------------------------------------------
// SokoScore_get_delegate
//-----------------------------------------------------------------------------
SokoScoreDelegate SokoScore_get_delegate( SokoScore p )
    {
    return p->data->delegate;
    }


//-----------------------------------------------------------------------------
// SokoScore_new
//-----------------------------------------------------------------------------
SokoScore SokoScore_new( SokoScoreDelegate delegate, void* info )
    {
    SokoScore p = (SokoScore)malloc( sizeof(struct _SokoScore) );
    p->data = (struct SokoScoreData*)malloc( sizeof(struct SokoScoreData) );
    p->data->delegate = delegate;
    p->data->scores = 0;
    p->data->scores_sorted = 0;
    p->data->scores_count = 0;
    p->data->scores_max = 0;
    p->data->pool = SokoPool_new(0);
    p->info = info;
    set_criteria( p, SOKO_CRITERIA_DEFAULT, SOKO_CRITERIA_DEFAULT_COUNT );
    read_scores(p);
    return p;
    }


//-----------------------------------------------------------------------------
// SokoScore_destroy
//-----------------------------------------------------------------------------
void SokoScore_destroy( SokoScore p )
    {
    if (p->data->scores != 0)
	free( p->data->scores );
    if (p->data->scores_sorted != 0)
	free( p->data->scores_sorted );
    SokoPool_destroy( p->data->pool );
    free(p->data);
    free(p);
    }
