//-----------------------------------------------------------------------------
// SokoPuzzle.c
//
//	Object representing a single puzzle instance.
//
// Copyright (c), 1997,2001,2002, Eric Sunshine <sunshine@sunshineco.com>
// Copyright (c), 1997, Paul McCarthy <zarnuk@high-speed-software.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoPuzzle.c,v 1.9 2002/05/06 05:46:59 sunshine Exp $
// $Log: SokoPuzzle.c,v $
// Revision 1.9  2002/05/06 05:46:59  sunshine
// v19
// -*- Augmented the path finding code for crates so that it is now capable
//     of discovering arbitrarily complex paths which overlap upon themselves
//     any number of times.  Previously, it was limited to finding only
//     non-overlapping paths.
//
// Revision 1.8  2002/02/19 07:34:26  sunshine
// v18
// -*- Added support for new triangular-style Trioban puzzles.
//
// -*- Fixed bug: Wrong scores were recorded after loading a saved game and
//     achieving a new high score for one of the metrics.  Problem was that
//     it was sending the "best" scores for all metrics to the SCORES file
//     entry for this game, rather than the actual scores for each metric for
//     this game.  (v17 bug)
//
// -*- Fixed bug: Least-cost player movement in crate path finding was
//     broken.  There was a case where it would not always choose the path
//     with least player-movement cost.  Problem was that
//     shortest_crate_path() was not tracking sufficient information.  It was
//     only tracking the minimum player-movement cost from cell to cell but
//     did not know from which previous cell that cost arose.  This was a
//     problem during the final path gleaning step when walking backward over
//     the flood-fill if it came to a cell at which player-movement cost was
//     the same from more than one approach.  It was unable to decide which
//     approach was less costly for the previous push over which it had just
//     walked backward.  (v11 bug)
//
// -*- Fixed bug: Auto-detection of even-on-even/odd-on-even for Hexoban
//     puzzles did not work when meta-data appeared in the file before the
//     puzzle data.
//
// -*- SokoPuzzle now maintains the raw meta-data from the puzzle file rather
//     than merely ignoring it.  Clients may access the raw data via the new
//     SokoPuzzle meta_data() method.
//
// -*- SokoPuzzle now parses a puzzle's meta-data and builds a list of
//     key/value tuples from it.  The tuples are extracted from lines which
//     resemble "Keyword: value", optionally preceded by any number of
//     semi-colons (`;') since many existing puzzle sets use this format.
//     Clients may look up keywords (case insensitive) or retrieve the entire
//     list of key/value tuples.
//
// -*- SokoPuzzle now recognizes the following (case insensitive) "pragmas"
//     in puzzle files and save files.  These pragmas may be useful for
//     puzzle authors at design time, since they allow the author to disable
//     behaviors which are typically undesirable and/or annoying during the
//     iterative edit-test-edit-test design cycle.
//
//     Pragma: no-high-score
//         Inhibits recording high scores and prevents launch of "New Score"
//         panel.
//     Pragma: no-auto-save
//         Inhibits automatic saving of game when solved.
//     Pragma: no-auto-advance
//         Inhibits advancing the user's "highest" level.
//
// -*- SokoPuzzle now exports all meta-data from the original puzzle file to
//     the saved-game file.
//
// -*- Consolidated the user-interface movement constraints into SokoPuzzle
//     rather than having the same code repeated in each port.  This code
//     decides whether or not the direction movement buttons in the
//     user-interface should be enabled depending upon cell type,
//     orientation, etc.  Added the SokoPuzzle methods can_move_constrained()
//     and can_move_diagonal_constrained() to complement the existing
//     can_move() and can_move_diagonal().
//
// -*- Previously, for square puzzles, the diagonal movement buttons in the
//     user-interface would be disabled if any pushes were involved in the
//     diagonal movement.  This is no longer the case.
//
// -*- Fixed bug: SokoPuzzle was leaking a file descriptor each time a puzzle
//     was loaded.
//-----------------------------------------------------------------------------
#include "SokoPool.h"
#include "SokoPuzzle.h"
#include "SokoAssert.h"
#include "SokoEncode.h"
#include "SokoFile.h"
#include "SokoSetting.h"
#include "SokoUtil.h"
#include <ctype.h>
#include <errno.h>
#include <limits.h>
#include <math.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
static void undo_move( SokoPuzzle );
static void redo_move( SokoPuzzle );

typedef soko_bool (*SokoDragCallback)( SokoPuzzle, int row, int col );

struct SokoDragState
    {
    soko_bool drag_active;
    soko_bool drag_abort;
    soko_bool drag_attempted;
    int row;
    int col;
    SokoDragCallback callback;
    };

struct SokoMetaTuple
    {
    char* key;
    char const* value;
    };

struct SokoMetaData
    {
    int tuple_count;
    int tuple_max;
    struct SokoMetaTuple* tuples;
    char* raw;
    int raw_count;
    int raw_max;
    };

struct SokoPuzzleData
    {
    SokoPuzzleDelegate delegate;
    char* save_file_name;
    char* puzzle_file_name;
    SokoPuzzleStyle puzzle_style;
    int row_count;
    int col_count;
    int player_row;
    int player_col;
    char* board;
    int history_max;
    int history_len;
    int history_pos;
    char* history;
    soko_bool history_dirty;
    int move_count;
    int push_count;
    int run_count;
    int focus_count;
    int unsafe_crate_count;
    int recorded_move_count;
    int recorded_push_count;
    int recorded_run_count;
    int recorded_focus_count;
    int selected_row;
    int selected_col;
    int last_push_row;
    int last_push_col;
    int animating;
    soko_bool playback_active;
    int playback_target;
    struct SokoDragState drag_state;
    struct SokoMetaData meta_data;
    soko_bool inhibit_high_score;
    soko_bool inhibit_auto_save;
    soko_bool inhibit_auto_advance;
    };

#define SAVE_FILE_VERSION (2)

// Characters from input file.		// ASCII
#define EMPTY_CHAR		' '	// 32	0010 0000
#define WALL_CHAR		'#'	// 35	0010 0011
#define CRATE_CHAR		'$'	// 36	0010 0100
#define SAFE_CRATE_CHAR		'*'	// 42	0010 1010
#define SAFE_EMPTY_CHAR		'.'	// 46	0010 1110
#define PLAYER_CHAR		'@'	// 64	0100 0000
#define SAFE_PLAYER_CHAR	'+'	// 43	0010 1011
#define SAFE_PLAYER_CHAR_ALT	'^'	// 94	0101 1110 (deprecated)

static char const PUZZLE_CHARS[] = " #$*.@+^";

// Characters from v1 save file.	// ASCII
#define EMPTY_CHAR_V1		'@'	// 64	0100 0000
#define WALL_CHAR_V1		'H'	// 72	0100 1000
#define CRATE_CHAR_V1		'D'	// 68	0100 0100
#define SAFE_CRATE_CHAR_V1	'E'	// 69	0010 1011
#define SAFE_EMPTY_CHAR_V1	'A'	// 65	0100 0001
#define PLAYER_CHAR_V1		'B'	// 66	0100 0010
#define SAFE_PLAYER_CHAR_V1	'C'	// 67	0100 0011
#define NULL_CHAR_V1		'F'	// 70	0010 1100

// Internal representation.
#define SAFE_BIT	(1 << 0)
#define PLAYER_BIT	(1 << 1)
#define CRATE_BIT	(1 << 2)
#define WALL_BIT	(1 << 3)
#define SELECTED_BIT	(1 << 4)
#define NULL_BIT	(1 << 5)
#define ILLEGAL_BIT	(1 << 6)

#define EMPTY_CH	((char)(0))
#define SAFE_EMPTY_CH	((char)(SAFE_BIT))
#define PLAYER_CH	((char)(PLAYER_BIT))
#define SAFE_PLAYER_CH	((char)(PLAYER_BIT|SAFE_BIT))
#define CRATE_CH	((char)(CRATE_BIT))
#define SAFE_CRATE_CH	((char)(CRATE_BIT|SAFE_BIT))
#define WALL_CH		((char)(WALL_BIT))
#define NULL_CH		((char)(NULL_BIT))
#define ILLEGAL_CH	((char)(ILLEGAL_BIT))

#define IS_SAFE(x)	((x) & (SAFE_BIT))
#define IS_EMPTY(x)	(((x) & (PLAYER_BIT|CRATE_BIT|WALL_BIT|NULL_BIT)) == 0)
#define IS_PLAYER(x)	((x) & (PLAYER_BIT))
#define IS_CRATE(x)	((x) & (CRATE_BIT))
#define IS_SELECTED(x)	((x) & (SELECTED_BIT))
#define IS_WALL(x)	((x) & (WALL_BIT))
#define IS_NULL(x)	((x) & (NULL_BIT))
#define IS_CLOSED(x)	(((x) & (WALL_BIT|NULL_BIT)) != 0)
#define IS_ILLEGAL(x)	((x) & (ILLEGAL_BIT))

#define SOKO_CELL_ILLEGAL ((SokoCellType)-1)

static SokoCellType const CELL_TYPE_FOR_CH[] =
    {
    SOKO_CELL_EMPTY,			// 0000 0000
    SOKO_CELL_EMPTY_SAFE,		// 0000 0001
    SOKO_CELL_PLAYER,			// 0000 0010
    SOKO_CELL_PLAYER_SAFE,		// 0000 0011
    SOKO_CELL_CRATE,			// 0000 0100
    SOKO_CELL_CREATE_SAFE,		// 0000 0101
    SOKO_CELL_ILLEGAL,			// 0000 0110
    SOKO_CELL_ILLEGAL,			// 0000 0111
    SOKO_CELL_WALL,			// 0000 1000
    SOKO_CELL_ILLEGAL,			// 0000 1001
    SOKO_CELL_ILLEGAL,			// 0000 1010
    SOKO_CELL_ILLEGAL,			// 0000 1011
    SOKO_CELL_ILLEGAL,			// 0000 1100
    SOKO_CELL_ILLEGAL,			// 0000 1101
    SOKO_CELL_ILLEGAL,			// 0000 1110
    SOKO_CELL_ILLEGAL,			// 0000 1111
    SOKO_CELL_ILLEGAL,			// 0001 0000
    SOKO_CELL_ILLEGAL,			// 0001 0001
    SOKO_CELL_ILLEGAL,			// 0001 0010
    SOKO_CELL_ILLEGAL,			// 0001 0011
    SOKO_CELL_CRATE_SELECTED,		// 0001 0100
    SOKO_CELL_CRATE_SELECTED_SAFE,	// 0001 0101
    SOKO_CELL_ILLEGAL,			// 0001 0110
    SOKO_CELL_ILLEGAL,			// 0001 0111
    SOKO_CELL_ILLEGAL,			// 0001 1000
    SOKO_CELL_ILLEGAL,			// 0001 1001
    SOKO_CELL_ILLEGAL,			// 0001 1010
    SOKO_CELL_ILLEGAL,			// 0001 1011
    SOKO_CELL_ILLEGAL,			// 0001 1100
    SOKO_CELL_ILLEGAL,			// 0001 1101
    SOKO_CELL_ILLEGAL,			// 0001 1110
    SOKO_CELL_ILLEGAL,			// 0001 1111
    SOKO_CELL_NULL			// 0010 0000
    };

#define GOOD_ROW(r)	((0 <= (r)) && ((r) < p->data->row_count))
#define GOOD_COL(c)	((0 <= (c)) && ((c) < p->data->col_count))

#define CELL_REF(m,r,c)	(m)[ ((r) * p->data->col_count) + (c) ]
#define BOARD(r,c)	CELL_REF(p->data->board, r, c)
#define MAP(r,c)	CELL_REF(map, r, c)

#define DELEGATE(p)	((p)->data->delegate)
#define STYLE(p)	((p)->data->puzzle_style)

#define IS_CRATE_SELECTED \
    (p->data->selected_row != -1 && p->data->selected_col != -1)

typedef enum
    {
    RCH_UNKNOWN,
    RCH_UNREACHABLE,
    RCH_REACHABLE
    } ReachableMark;

#define SOKO_DIR_ILLEGAL ((SokoDirection)-1)
#define GOOD_DIR(x)	(((unsigned int)(x)) < ((unsigned int)SOKO_DIR_MAX))

static int const DirMoveCount[ SOKO_STYLE_MAX ] = { 4, 6, 3 };
static SokoDirection const DirMove[ SOKO_STYLE_MAX ][2][2][ SOKO_DIR_MAX ] =
    {{{{
    SOKO_DIR_UP,	// square, even row, even column
    SOKO_DIR_LEFT,
    SOKO_DIR_RIGHT,
    SOKO_DIR_DOWN,
    SOKO_DIR_ILLEGAL,
    SOKO_DIR_ILLEGAL
    },{
    SOKO_DIR_UP,	// square, even row, odd column
    SOKO_DIR_LEFT,
    SOKO_DIR_RIGHT,
    SOKO_DIR_DOWN,
    SOKO_DIR_ILLEGAL,
    SOKO_DIR_ILLEGAL
    }},{{
    SOKO_DIR_UP,	// square odd row, even column
    SOKO_DIR_LEFT,
    SOKO_DIR_RIGHT,
    SOKO_DIR_DOWN,
    SOKO_DIR_ILLEGAL,
    SOKO_DIR_ILLEGAL
    },{
    SOKO_DIR_UP,	// square odd row, odd column
    SOKO_DIR_LEFT,
    SOKO_DIR_RIGHT,
    SOKO_DIR_DOWN,
    SOKO_DIR_ILLEGAL,
    SOKO_DIR_ILLEGAL
    }}},{{{
    SOKO_DIR_UP,	// hexagon, even row, even column
    SOKO_DIR_LEFT,
    SOKO_DIR_RIGHT,
    SOKO_DIR_DOWN,
    SOKO_DIR_NORTH,
    SOKO_DIR_SOUTH
    },{
    SOKO_DIR_UP,	// hexagon, even row, odd column
    SOKO_DIR_LEFT,
    SOKO_DIR_RIGHT,
    SOKO_DIR_DOWN,
    SOKO_DIR_NORTH,
    SOKO_DIR_SOUTH
    }},{{
    SOKO_DIR_UP,	// hexagon, odd row, even column
    SOKO_DIR_LEFT,
    SOKO_DIR_RIGHT,
    SOKO_DIR_DOWN,
    SOKO_DIR_NORTH,
    SOKO_DIR_SOUTH
    },{
    SOKO_DIR_UP,	// hexagon, odd row, odd column
    SOKO_DIR_LEFT,
    SOKO_DIR_RIGHT,
    SOKO_DIR_DOWN,
    SOKO_DIR_NORTH,
    SOKO_DIR_SOUTH
    }}},{{{
    SOKO_DIR_UP,	// triangle, even row, even column
    SOKO_DIR_RIGHT,
    SOKO_DIR_SOUTH,
    SOKO_DIR_ILLEGAL,
    SOKO_DIR_ILLEGAL,
    SOKO_DIR_ILLEGAL
    },{
    SOKO_DIR_DOWN,	// triangle, even row, odd column
    SOKO_DIR_LEFT,
    SOKO_DIR_NORTH,
    SOKO_DIR_ILLEGAL,
    SOKO_DIR_ILLEGAL,
    SOKO_DIR_ILLEGAL
    }},{{
    SOKO_DIR_DOWN,	// triangle, odd row, even column
    SOKO_DIR_LEFT,
    SOKO_DIR_NORTH,
    SOKO_DIR_ILLEGAL,
    SOKO_DIR_ILLEGAL,
    SOKO_DIR_ILLEGAL
    },{
    SOKO_DIR_UP,	// triangle, odd row, odd column
    SOKO_DIR_RIGHT,
    SOKO_DIR_SOUTH,
    SOKO_DIR_ILLEGAL,
    SOKO_DIR_ILLEGAL,
    SOKO_DIR_ILLEGAL
    }}}};
static int const DirPushCount[ SOKO_STYLE_MAX ] = { 4, 6, 6 };
static SokoDirection const DirPush[ SOKO_STYLE_MAX ][ SOKO_DIR_MAX ] =
    {{
    SOKO_DIR_UP,
    SOKO_DIR_LEFT,
    SOKO_DIR_RIGHT,
    SOKO_DIR_DOWN,
    SOKO_DIR_ILLEGAL,
    SOKO_DIR_ILLEGAL
    },{
    SOKO_DIR_UP,
    SOKO_DIR_LEFT,
    SOKO_DIR_RIGHT,
    SOKO_DIR_DOWN,
    SOKO_DIR_NORTH,
    SOKO_DIR_SOUTH
    },{
    SOKO_DIR_UP,
    SOKO_DIR_LEFT,
    SOKO_DIR_RIGHT,
    SOKO_DIR_DOWN,
    SOKO_DIR_NORTH,
    SOKO_DIR_SOUTH
    }};
static int const DirDeltaRow[ SOKO_STYLE_MAX ][2][2][ SOKO_DIR_MAX ] =
    {{{
    { -1,  0,  0,  1, -1,  1 },	// square,   even row, even column
    { -1,  0,  0,  1, -1,  1 }	//                      odd column
    },{
    { -1,  0,  0,  1, -1,  1 },	//            odd row, even column
    { -1,  0,  0,  1, -1,  1 }	//                      odd column
    }},{{
    { -1,  0,  0,  1, -1,  1 },	// hexagon,  even row, even column
    { -1,  0,  0,  1, -1,  1 }	//                      odd column
    },{
    { -1,  0,  0,  1, -1,  1 },	//            odd row, even column
    { -1,  0,  0,  1, -1,  1 }	//                      odd column
    }},{{
    { -1,  0,  0,  0, -1,  0 },	// triangle, even row, even column
    {  0,  0,  0,  1,  0,  1 }	//                      odd column
    },{
    {  0,  0,  0,  1,  0,  1 },	//            odd row, even column
    { -1,  0,  0,  0, -1,  0 }	//                      odd column
    }}};
static int const DirDeltaCol[ SOKO_STYLE_MAX ][2][2][ SOKO_DIR_MAX ] =
    {{{
    {  0, -1,  1,  0,  0,  0 },	// sqaure,   even row, even column
    {  0, -1,  1,  0,  0,  0 }	//                      odd column
    },{
    {  0, -1,  1,  0,  0,  0 },	//            odd row, even column
    {  0, -1,  1,  0,  0,  0 }	//                      odd column
    }},{{
    { -1, -1,  1,  0,  0, -1 },	// hexagon,  even row, even column
    { -1, -1,  1,  0,  0, -1 }	//                      odd column
    },{
    {  0, -1,  1,  1,  1,  0 },	//            odd row, even column
    {  0, -1,  1,  1,  1,  0 }	//                      odd column
    }},{{
    {  0, -1,  1,  1,  0, -1 },	// triangle, even row, even column
    { -1, -1,  1,  0,  1,  0 }	//                      odd column
    },{
    { -1, -1,  1,  0,  1,  0 },	//            odd row, even column
    {  0, -1,  1,  1,  0, -1 }	//                      odd column
    }}};
static SokoDirection const DirOpposite[ SOKO_DIR_MAX ] =
    {
    SOKO_DIR_DOWN,	// SOKO_DIR_UP
    SOKO_DIR_RIGHT,	// SOKO_DIR_LEFT
    SOKO_DIR_LEFT,	// SOKO_DIR_RIGHT
    SOKO_DIR_UP,	// SOKO_DIR_DOWN
    SOKO_DIR_SOUTH,	// SOKO_DIR_NORTH
    SOKO_DIR_NORTH	// SOKO_DIR_SOUTH
    };
static SokoDirection const DirDiagonalSquare[ SOKO_DIAG_MAX ][2] =
    {
    { SOKO_DIR_UP,    SOKO_DIR_LEFT  },	// SOKO_DIAG_UP_LEFT
    { SOKO_DIR_UP,    SOKO_DIR_RIGHT },	// SOKO_DIAG_UP_RIGHT
    { SOKO_DIR_LEFT,  SOKO_DIR_DOWN  },	// SOKO_DIAG_DOWN_LEFT
    { SOKO_DIR_RIGHT, SOKO_DIR_DOWN  }	// SOKO_DIAG_DOWN_RIGHT
    };
static SokoDirection const DirDiagonalHexTri[ SOKO_DIAG_MAX ] =
    {
    SOKO_DIR_UP,	// SOKO_DIAG_UP_LEFT
    SOKO_DIR_NORTH,	// SOKO_DIAG_UP_RIGHT
    SOKO_DIR_SOUTH,	// SOKO_DIAG_DOWN_LEFT
    SOKO_DIR_DOWN	// SOKO_DIAG_DOWN_RIGHT
    };

static char const* const StyleName[ SOKO_STYLE_MAX ] =
    {
    "square",	// SOKO_STYLE_SQUARE
    "hexagon",	// SOKO_STYLE_HEXAGON
    "triangle",	// SOKO_STYLE_TRIANGLE
    };

#define SAVE_EXTENSION "sokosave"
static SokoPuzzleExtension const PuzzleExtensions[] =
    {
    { "sokomaze", SOKO_STYLE_SQUARE   },
    { "xsb",      SOKO_STYLE_SQUARE   },
    { "sokohex",  SOKO_STYLE_HEXAGON  },
    { "hsb",      SOKO_STYLE_HEXAGON  },
    { "sokotri",  SOKO_STYLE_TRIANGLE },
    { "tsb",      SOKO_STYLE_TRIANGLE },
    { 0,          0                   }
    };
#define PUZZLE_EXTENSIONS_COUNT \
    (sizeof(PuzzleExtensions) / sizeof(PuzzleExtensions[0]) - 1)

#define MOVE_UP_CH	'u'
#define MOVE_LEFT_CH	'l'
#define MOVE_RIGHT_CH	'r'
#define MOVE_DOWN_CH	'd'
#define MOVE_NORTH_CH	'n'
#define MOVE_SOUTH_CH	's'
#define PUSH_UP_CH	'U'
#define PUSH_LEFT_CH	'L'
#define PUSH_RIGHT_CH	'R'
#define PUSH_DOWN_CH	'D'
#define PUSH_NORTH_CH	'N'
#define PUSH_SOUTH_CH	'S'
#define IS_PUSH(x)	(isupper(x))
#define IS_MOVE(x)	(islower(x))

static char const MOVE_CHARS[] = "ulrdns";
static char const PUSH_CHARS[] = "ULRDNS";

#define KEYWORD_PRAGMA "pragma"
#define PRAGMA_NO_HIGH_SCORE "no-high-score"
#define PRAGMA_NO_AUTO_SAVE "no-auto-save"
#define PRAGMA_NO_AUTO_ADVANCE "no-auto-advance"

static SokoPuzzle* OPEN_PUZZLES = 0;
static int OPEN_PUZZLES_COUNT = 0;
static int OPEN_PUZZLES_MAX = 0;

typedef struct _Coord
    {
    int row;
    int col;
    } Coord;

typedef struct _CostDir
    {
    int n;		// Minimum movement cost.
    SokoDirection d;	// Direction from which minimum base cost was taken.
    int l;		// Distance from origin of this cost computation.
    } CostDir;

typedef struct _CrateMapRec
    {
    CostDir cost[ SOKO_DIR_MAX ];
    } CrateMapRec;

// Pre-allocated stacks, maps, and paths for shortest-path calculations.
static SokoDirection* PLAYER_PATH    = 0;
static int*           PLAYER_MAP     = 0;
static Coord*         PLAYER_STACK_1 = 0;
static Coord*         PLAYER_STACK_2 = 0;
static SokoDirection* CRATE_PATH     = 0;
static CrateMapRec*   CRATE_MAP      = 0;
static Coord*         CRATE_STACK_1  = 0;
static Coord*         CRATE_STACK_2  = 0;

// Pre-allocated structures for dirty-cell management and reporting.
static char*          DIRTY_MAP      = 0;
static SokoCell*      DIRTY_LIST     = 0;
static int            DIRTY_COUNT    = 0;
static int            DIRTY_DEPTH    = 0;
static SokoPuzzle     DIRTY_PUZZLE   = 0; // For internal error checking only.


//-----------------------------------------------------------------------------
// ch_for_move
//-----------------------------------------------------------------------------
static char ch_for_move( SokoPuzzle p, SokoDirection dir, soko_bool is_push )
    {
    SOKO_ASSERT( GOOD_DIR(dir) );
    return (is_push ? PUSH_CHARS[dir] : MOVE_CHARS[dir]);
    }


//-----------------------------------------------------------------------------
// dir_from_move
//-----------------------------------------------------------------------------
static SokoDirection dir_from_move( char move, soko_bool error_fatal )
    {
    SokoDirection dir = SOKO_DIR_UP;
    switch (move)
	{
	case MOVE_UP_CH:	dir = SOKO_DIR_UP;	break;
	case MOVE_LEFT_CH:	dir = SOKO_DIR_LEFT;	break;
	case MOVE_RIGHT_CH:	dir = SOKO_DIR_RIGHT;	break;
	case MOVE_DOWN_CH:	dir = SOKO_DIR_DOWN;	break;
	case MOVE_NORTH_CH:	dir = SOKO_DIR_NORTH;	break;
	case MOVE_SOUTH_CH:	dir = SOKO_DIR_SOUTH;	break;
	case PUSH_UP_CH:	dir = SOKO_DIR_UP;	break;
	case PUSH_LEFT_CH:	dir = SOKO_DIR_LEFT;	break;
	case PUSH_RIGHT_CH:	dir = SOKO_DIR_RIGHT;	break;
	case PUSH_DOWN_CH:	dir = SOKO_DIR_DOWN;	break;
	case PUSH_NORTH_CH:	dir = SOKO_DIR_NORTH;	break;
	case PUSH_SOUTH_CH:	dir = SOKO_DIR_SOUTH;	break;
	default:
	    if (!error_fatal)
		dir = SOKO_DIR_ILLEGAL;
	    else
		{
		fprintf( stderr,"SokoSave: Internal error: "
		    "dir_from_move('%c') bad move.\n", move );
		exit(3);
		}
	}
    return dir;
    }


//-----------------------------------------------------------------------------
// send_alert
//-----------------------------------------------------------------------------
static void send_alert( SokoPuzzle p, SokoAlert severity, char const* title,
    char const* fmt, ... )
    {
    char buff[ 1024 ];
    va_list args;
    va_start( args, fmt );
    vsprintf( buff, fmt, args );
    va_end( args );
    SOKO_ASSERT( strlen(buff) < sizeof(buff) );
    DELEGATE(p)->alert_callback( p, DELEGATE(p), severity, title, buff );
    }


//=============================================================================
// Simple Accessors
//=============================================================================
int SokoPuzzle_row_count( SokoPuzzle p )    { return p->data->row_count; }
int SokoPuzzle_col_count( SokoPuzzle p )    { return p->data->col_count; }

int SokoPuzzle_move_count ( SokoPuzzle p )  { return p->data->move_count;  }
int SokoPuzzle_push_count ( SokoPuzzle p )  { return p->data->push_count;  }
int SokoPuzzle_run_count  ( SokoPuzzle p )  { return p->data->run_count;   }
int SokoPuzzle_focus_count( SokoPuzzle p )  { return p->data->focus_count; }

int SokoPuzzle_player_row( SokoPuzzle p )   { return p->data->player_row;  }
int SokoPuzzle_player_col( SokoPuzzle p )   { return p->data->player_col;  }

int SokoPuzzle_selected_row( SokoPuzzle p ) { return p->data->selected_row; }
int SokoPuzzle_selected_col( SokoPuzzle p ) { return p->data->selected_col; }

SokoPuzzleStyle SokoPuzzle_puzzle_style( SokoPuzzle p )
    { return p->data->puzzle_style; }
soko_bool SokoPuzzle_puzzle_solved( SokoPuzzle p )
    { return (p->data->unsafe_crate_count == 0); }
soko_bool SokoPuzzle_puzzle_dirty ( SokoPuzzle p )
    { return p->data->history_dirty; }

int SokoPuzzle_history_position( SokoPuzzle p )
    { return p->data->history_pos; }
int SokoPuzzle_history_length( SokoPuzzle p )
    { return p->data->history_len; }

SokoPuzzleDelegate SokoPuzzle_get_delegate( SokoPuzzle p )
    { return DELEGATE(p); }

char const* SokoPuzzle_get_style_name( SokoPuzzleStyle style )
    {
    SOKO_ASSERT( style < SOKO_STYLE_MAX );
    return StyleName[ style ];
    }


//=============================================================================
// Pathname Facilities
//=============================================================================
char const* SokoPuzzle_get_puzzle_file_name( SokoPuzzle p, cSokoPool pool )
    { return soko_denormalize_path( pool, p->data->puzzle_file_name ); }

char const* SokoPuzzle_get_save_file_name( SokoPuzzle p, cSokoPool pool )
    { return soko_denormalize_path( pool, p->data->save_file_name ); }

SokoPuzzleExtension const* SokoPuzzle_get_puzzle_extensions()
    { return PuzzleExtensions; }
int SokoPuzzle_get_puzzle_extensions_count()
    { return PUZZLE_EXTENSIONS_COUNT; }
char const* SokoPuzzle_get_save_extension()
    { return SAVE_EXTENSION; }


//-----------------------------------------------------------------------------
// match_puzzle_extension
//-----------------------------------------------------------------------------
static SokoPuzzleExtension const* match_puzzle_extension( char const* path )
    {
    SokoPool pool = SokoPool_new(0);
    char const* e = soko_extension_part( pool, path );
    SokoPuzzleExtension const* r = PuzzleExtensions;
    for ( ; r->extension != 0; r++)
	if (soko_stricmp( e, r->extension ) == 0)
	    break;
    SokoPool_destroy( pool );
    return (r->extension != 0 ? r : 0);
    }


//-----------------------------------------------------------------------------
// SokoPuzzle_file_is_puzzle
//-----------------------------------------------------------------------------
soko_bool SokoPuzzle_file_is_puzzle( char const* path )
    {
    return (match_puzzle_extension( path ) != 0);
    }


//-----------------------------------------------------------------------------
// SokoPuzzle_file_is_save_game
//-----------------------------------------------------------------------------
soko_bool SokoPuzzle_file_is_save_game( char const* path )
    {
    SokoPool pool = SokoPool_new(0);
    char const* e = soko_extension_part( pool, path );
    soko_bool const b = (soko_stricmp( e, SAVE_EXTENSION ) == 0);
    SokoPool_destroy( pool );
    return b;
    }


//-----------------------------------------------------------------------------
// SokoPuzzle_get_file_style
//-----------------------------------------------------------------------------
SokoPuzzleStyle SokoPuzzle_get_file_style( char const* path )
    {
    SokoPuzzleExtension const* e = match_puzzle_extension( path );
    return (e != 0 ? e->style : SOKO_STYLE_MAX);
    }


//-----------------------------------------------------------------------------
// SokoPuzzle_save_name_for_puzzle_name
//-----------------------------------------------------------------------------
char const* SokoPuzzle_save_name_for_puzzle_name(
    char const* path, cSokoPool pool)
    {
    return soko_add_path_component(
	pool, soko_expand_path(pool, soko_get_save_directory(pool)),
	soko_replace_path_extension(pool, soko_filename_part(pool, path),
	    SAVE_EXTENSION) );
    }


//-----------------------------------------------------------------------------
// SokoPuzzle_puzzle_name_for_level
//-----------------------------------------------------------------------------
char const* SokoPuzzle_puzzle_name_for_level( int level, cSokoPool pool )
    {
    char buff[32];
    sprintf( buff, "%02d", level );
    return soko_add_path_component(
	pool, soko_expand_path(pool, soko_get_puzzle_directory(pool)),
	soko_add_path_extension(pool, buff, PuzzleExtensions[0].extension) );
    }


//-----------------------------------------------------------------------------
// SokoPuzzle_level_for_puzzle_name
//-----------------------------------------------------------------------------
int SokoPuzzle_level_for_puzzle_name( char const* path )
    {
    int level;
    SokoPool pool = SokoPool_new(0);
    if (sscanf(soko_filename_part(pool, path), "%d", &level) != 1)
	level = -1;
    SokoPool_destroy( pool );
    return level;
    }


//=============================================================================
// Meta-data Facilities
//=============================================================================
char const* SokoPuzzle_meta_data( SokoPuzzle p )
    { return p->data->meta_data.raw; }
int SokoPuzzle_meta_data_key_count( SokoPuzzle p )
    { return p->data->meta_data.tuple_count; }
char const* SokoPuzzle_meta_data_get_key( SokoPuzzle p, int n )
    { return p->data->meta_data.tuples[n].key; }
char const* SokoPuzzle_meta_data_get_value( SokoPuzzle p, int n )
    { return p->data->meta_data.tuples[n].value; }


//-----------------------------------------------------------------------------
// SokoPuzzle_meta_data_lookup
//-----------------------------------------------------------------------------
char const* SokoPuzzle_meta_data_lookup( SokoPuzzle p, char const* key )
    {
    int i;
    for (i = 0; i < p->data->meta_data.tuple_count; i++)
	if (soko_stricmp( key, p->data->meta_data.tuples[i].key ) == 0)
	    return p->data->meta_data.tuples[i].value;
    return 0;
    }


//-----------------------------------------------------------------------------
// meta_data_store
//-----------------------------------------------------------------------------
static void meta_data_store( SokoPuzzle p,
    char const* key, int key_len, char const* value, int value_len )
    {
    char *pkey, *pvalue;
    struct SokoMetaTuple* t;
    if (p->data->meta_data.tuple_count >= p->data->meta_data.tuple_max)
	{
	int n;
	p->data->meta_data.tuple_max += 16;
	n = p->data->meta_data.tuple_max*sizeof(p->data->meta_data.tuples[0]);
	p->data->meta_data.tuples =
	    (struct SokoMetaTuple*)(p->data->meta_data.tuples == 0 ?
	    malloc(n) : realloc( p->data->meta_data.tuples, n ));
	}

    pkey = (char*)malloc( (key_len + value_len + 2) * sizeof(key[0]) );
    pvalue = pkey + key_len + 1;
    strncpy( pkey, key, key_len );
    strncpy( pvalue, value, value_len );
    pkey[ key_len ] = '\0';
    pvalue[ value_len ] = '\0';

    t = p->data->meta_data.tuples + p->data->meta_data.tuple_count;
    t->key = pkey;
    t->value = pvalue;
    p->data->meta_data.tuple_count++;
    }


//-----------------------------------------------------------------------------
// meta_data_parse
//	Examine a line of meta-data to see if it forms a key/value tuple.
//	Tuples are formatted as "key: value" and are often preceded by one or
//	more semi-colons (;).
//-----------------------------------------------------------------------------
static void meta_data_parse( SokoPuzzle p, char const* s )
    {
    char const* key;
    char const* key_end;

    while (*s == ';' || isspace(*s))
	s++;					// Skip whitespace and ';'.
    key = s;
    while (*s == '_' || *s == '-' || isalnum(*s))
	s++;					// Collect key.
    key_end = s;
    while (isspace(*s))
    	s++;					// Skip whitespace.

    if (*s == ':')				// Match ':'.
	{
	char const* value;
	char const* value_end;

	s++;					// Skip ':'.
	while (isspace(*s))
	    s++;				// Skip whitespace.
	value = s;
	while (*s != '\n' && *s != '\r' && *s != '\0')
	    s++;				// Collect value.
	while (s > value && isspace(*(s-1)))
	    s--;				// Strip trailing whitespace.
	value_end = s;

	meta_data_store( p, key, key_end - key, value, value_end - value );
	}
    }


//-----------------------------------------------------------------------------
// meta_data_append
//-----------------------------------------------------------------------------
static void meta_data_append( SokoPuzzle p, char c )
    {
    int const l = p->data->meta_data.raw_count;
    if (l >= p->data->meta_data.raw_max)
	{
	int n;
	p->data->meta_data.raw_max += 64;
	n = p->data->meta_data.raw_max * sizeof(p->data->meta_data.raw[0]);
	p->data->meta_data.raw = (char*)(p->data->meta_data.raw == 0 ?
	    malloc(n) : realloc( p->data->meta_data.raw, n ));
	}

    p->data->meta_data.raw[l] = c;
    p->data->meta_data.raw_count++;

    if (l > 0 && (c == '\n' || c == '\r'))
	{
	char const* s = p->data->meta_data.raw + l - 1;
	while (s >= p->data->meta_data.raw && *s != '\n' && *s != '\r')
	    s--;
	meta_data_parse( p, s + 1 );
	}
    }


//-----------------------------------------------------------------------------
// meta_data_close
//-----------------------------------------------------------------------------
static void meta_data_close( SokoPuzzle p )
    {
    int const n = p->data->meta_data.raw_count;
    if (n > 0)
	{
	char const c = p->data->meta_data.raw[ n - 1 ];
	if (c != '\n' && c != '\r')
	    meta_data_append( p, '\n' );
	meta_data_append( p, '\0' ); // Ensure termination for client use.
	}
    }


//-----------------------------------------------------------------------------
// examine_meta_data
//	Examine the key/value tuples in the meta-data.  The following values
//	are recognized for the "Pragma" key:
//
//	    no-auto-save:    Inhibits automatic saving of game when solved.
//	    no-auto-advance: Inhibits advancing the user's "highest" level.
//	    no-high-score:   Inhibits recording high scores and prevents launch
//	                     launch of "New Score" panel.
//
//	The no-high-score, no-auto-save, and no-auto-advance pragmas may be
//	useful for puzzle authors at design time, since they allow the author
//	to disable behaviors which are typically undesirable and annoying
//	during the iterative edit-test-edit-test cycle.
//-----------------------------------------------------------------------------
static void examine_meta_data( SokoPuzzle p )
    {
    int i;
    for (i = 0; i < p->data->meta_data.tuple_count; i++)
	{
	struct SokoMetaTuple const* t = p->data->meta_data.tuples + i;
	if (soko_stricmp( KEYWORD_PRAGMA, t->key ) == 0)
	    {
	    if (soko_stricmp( PRAGMA_NO_HIGH_SCORE, t->value ) == 0)
		p->data->inhibit_high_score = soko_true;
	    else if (soko_stricmp( PRAGMA_NO_AUTO_SAVE, t->value ) == 0)
		p->data->inhibit_auto_save = soko_true;
	    else if (soko_stricmp( PRAGMA_NO_AUTO_ADVANCE, t->value ) == 0)
		p->data->inhibit_auto_advance = soko_true;
	    }
	}
    }


//=============================================================================
// Utility Function
//=============================================================================
//-----------------------------------------------------------------------------
// SokoPuzzle_cell_type
//-----------------------------------------------------------------------------
SokoCellType SokoPuzzle_cell_type( SokoPuzzle p, int row, int col )
    {
    SokoCellType c;
    SOKO_ASSERT( row >= 0 && row < p->data->row_count );
    SOKO_ASSERT( col >= 0 && col < p->data->col_count );
    c = CELL_TYPE_FOR_CH[ (int)BOARD(row, col) ];
    SOKO_ASSERT( c != SOKO_CELL_ILLEGAL );
    return c;
    }


//-----------------------------------------------------------------------------
// remember_puzzle
//-----------------------------------------------------------------------------
static void remember_puzzle( SokoPuzzle p )
    {
    if (OPEN_PUZZLES_COUNT >= OPEN_PUZZLES_MAX)
	{
	size_t nbytes;
	OPEN_PUZZLES_MAX += 16;
	nbytes = OPEN_PUZZLES_MAX * sizeof(OPEN_PUZZLES[0]);
	if (OPEN_PUZZLES == 0)
	    OPEN_PUZZLES = (SokoPuzzle*)malloc( nbytes );
	else
	    OPEN_PUZZLES = (SokoPuzzle*)realloc( OPEN_PUZZLES, nbytes );
	}
    OPEN_PUZZLES[ OPEN_PUZZLES_COUNT++ ] = p;
    }


//-----------------------------------------------------------------------------
// forget_puzzle
//-----------------------------------------------------------------------------
static void forget_puzzle( SokoPuzzle p )
    {
    int i;
    for (i = OPEN_PUZZLES_COUNT - 1; i >= 0; i--)
	{
	if (p == OPEN_PUZZLES[i])
	    {
	    OPEN_PUZZLES_COUNT--;
	    if (i < OPEN_PUZZLES_COUNT)
		memmove( OPEN_PUZZLES + i, OPEN_PUZZLES + i + 1,
		    (OPEN_PUZZLES_COUNT - i) * sizeof(OPEN_PUZZLES[0]) );
	    break;
	    }
	}
    }


//-----------------------------------------------------------------------------
// SokoPuzzle_find_open_puzzle
//-----------------------------------------------------------------------------
SokoPuzzle SokoPuzzle_find_open_puzzle( char const* path )
    {
    int i;
    SokoPool pool = SokoPool_new(0);

    if (!soko_has_suffix( path, SAVE_EXTENSION, soko_true ))
	path = SokoPuzzle_save_name_for_puzzle_name( path, pool );

    path = soko_normalize_path( pool, soko_expand_path(pool, path) );
    for (i = 0; i < OPEN_PUZZLES_COUNT; i++)
	if (soko_stricmp( OPEN_PUZZLES[i]->data->save_file_name, path ) == 0)
	    return OPEN_PUZZLES[i];

    SokoPool_destroy( pool );
    return 0;
    }


//-----------------------------------------------------------------------------
// SokoPuzzle_get_open_puzzle
//-----------------------------------------------------------------------------
SokoPuzzle SokoPuzzle_get_open_puzzle( int n )
    {
    SOKO_ASSERT( n >= 0 && n < OPEN_PUZZLES_COUNT );
    return OPEN_PUZZLES[n];
    }


//-----------------------------------------------------------------------------
// SokoPuzzle_open_puzzle_count
//-----------------------------------------------------------------------------
int SokoPuzzle_open_puzzle_count( void )
    {
    return OPEN_PUZZLES_COUNT;
    }


//-----------------------------------------------------------------------------
// record_new_level
//-----------------------------------------------------------------------------
static void record_new_level( SokoPuzzle p )
    {
    int level = SokoPuzzle_level_for_puzzle_name( p->data->puzzle_file_name );
    if (level >= 0)
	DELEGATE(p)->record_level_callback( p, DELEGATE(p), level );
    }


//-----------------------------------------------------------------------------
// record_new_score
//-----------------------------------------------------------------------------
static void record_new_score( SokoPuzzle p )
    {
    p->data->history_dirty = soko_true;
    DELEGATE(p)->set_dirty_callback( p, DELEGATE(p), soko_true );

    if (!p->data->inhibit_auto_save)
	{
	SokoSetting setting = SokoSetting_new(0);
	if (SokoSetting_get_bool( setting, SOKO_SETTING_AUTO_SAVE, soko_true ))
	    SokoPuzzle_save(p);
	SokoSetting_destroy( setting );
	}

    if (!p->data->inhibit_high_score)
	DELEGATE(p)->record_score_callback( p, DELEGATE(p),
	    p->data->move_count, p->data->push_count,
	    p->data->run_count,  p->data->focus_count );
    }


//-----------------------------------------------------------------------------
// send_solved_state
//-----------------------------------------------------------------------------
static void send_solved_state( SokoPuzzle p )
    {
    DELEGATE(p)->set_solved_callback(
	p, DELEGATE(p), SokoPuzzle_puzzle_solved(p) );
    }


//-----------------------------------------------------------------------------
// better_score
//-----------------------------------------------------------------------------
static soko_bool better_score( int score, int* recorded )
    {
    soko_bool better = soko_false;
    if (*recorded == 0 || score < *recorded)
	{
	*recorded = score;
	better = soko_true;
	}
    return better;
    }


//-----------------------------------------------------------------------------
// solved_state_changed
//-----------------------------------------------------------------------------
static void solved_state_changed( SokoPuzzle p )
    {
    send_solved_state(p);
    if (SokoPuzzle_puzzle_solved(p))
	{
	struct SokoPuzzleData* const d = p->data;
	soko_bool record_score = soko_false;
	if (d->recorded_move_count == 0 && !p->data->inhibit_auto_advance)
	    record_new_level(p);
	record_score |= better_score(d->move_count,  &d->recorded_move_count );
	record_score |= better_score(d->push_count,  &d->recorded_push_count );
	record_score |= better_score(d->run_count,   &d->recorded_run_count  );
	record_score |= better_score(d->focus_count, &d->recorded_focus_count);
	if (record_score)
	    record_new_score(p);
	}
    }


//-----------------------------------------------------------------------------
// solved_check
//	`src' is the cell from which the crate is being moved.
//	`dst' is the cell to which the crate will be moved.
//-----------------------------------------------------------------------------
static void solved_check( SokoPuzzle p, char src, char dst )
    {
    soko_bool src_safe = IS_SAFE(src);
    soko_bool dst_safe = IS_SAFE(dst);
    if (src_safe != dst_safe)
	{
	soko_bool was_solved = SokoPuzzle_puzzle_solved(p);

	if (src_safe)
	    p->data->unsafe_crate_count++;
	else // (dst_safe)
	    p->data->unsafe_crate_count--;
	SOKO_ASSERT( p->data->unsafe_crate_count >= 0 );

	if (was_solved != SokoPuzzle_puzzle_solved(p))
	    solved_state_changed(p);
	}
    }


//-----------------------------------------------------------------------------
// refresh_controls
//-----------------------------------------------------------------------------
static void refresh_controls( SokoPuzzle p )
    {
    DELEGATE(p)->refresh_controls_callback( p, DELEGATE(p) );
    }


//-----------------------------------------------------------------------------
// append_move_to_history
//-----------------------------------------------------------------------------
static void append_move_to_history(
    SokoPuzzle p, SokoDirection dir, soko_bool is_push )
    {
    char move_ch;
    soko_bool changed = soko_true;
    move_ch = ch_for_move( p, dir, is_push );

    if (p->data->history_pos < p->data->history_len)
	{
	if (p->data->history[ p->data->history_pos ] == move_ch)
	    {
	    p->data->history_pos++;
	    changed = soko_false;
	    }
	else
	    {
	    p->data->history[ p->data->history_pos++ ] = move_ch;
	    p->data->history_len = p->data->history_pos;
	    }
	}
    else
	{
	if (p->data->history_len >= p->data->history_max)
	    {
	    if (p->data->history_max != 0)
		{
		p->data->history_max <<= 1;
		p->data->history = (char*)realloc(p->data->history,
		    p->data->history_max * sizeof(p->data->history[0]));
		}
	    else
		{
		p->data->history_max = 1024;
		p->data->history = (char*)
		    malloc(p->data->history_max * sizeof(p->data->history[0]));
		}
	    if (p->data->history == 0)
		{
		fprintf( stderr, "Memory allocation failed: %s\n",
		    strerror(errno) );
		exit(3);
		}
	    }
	p->data->history[ p->data->history_pos++ ] = move_ch;
	p->data->history_len = p->data->history_pos;
	}

    p->data->move_count++;

    if (is_push)
	{
	int oddrow, oddcol;
	int const r = p->data->player_row;
	int const c = p->data->player_col;
	SokoPuzzleStyle const style = STYLE(p);

	p->data->push_count++;

	SOKO_ASSERT( p->data->history_pos != 0 );
	if (p->data->history_pos == 1 ||
	    move_ch != p->data->history[ p->data->history_pos - 2 ])
	    {
	    p->data->run_count++;
	    if (r != p->data->last_push_row || c != p->data->last_push_col)
		p->data->focus_count++;
	    }

	oddrow = (r & 1);
	oddcol = (c & 1);
	p->data->last_push_row = r + DirDeltaRow[style][oddrow][oddcol][dir];
	p->data->last_push_col = c + DirDeltaCol[style][oddrow][oddcol][dir];
	}

    if (p->data->animating == 0)
	refresh_controls(p);

    if (changed && !p->data->history_dirty)
	{
	p->data->history_dirty = soko_true;
	DELEGATE(p)->set_dirty_callback( p, DELEGATE(p), soko_true );
	}
    }


//=============================================================================
// Puzzle Initialization and Input & Output
//=============================================================================
//-----------------------------------------------------------------------------
// shrink_puzzle
//	If save_file_version is -1, then we are translating a puzzle character,
//	rather than a save file character.
// *TRIANGLE*
//	The orientation of a triangular tile is determined by the tile's
//	position in the grid.  Even rows (counting from zero) begin with a
//	south-pointing tile, and odd rows begin with a north-pointing tile.
//	Since tile orientation determines navigational attributes, it is
//	necessary to ensure that we do not trim away orientation-determining
//	whitespace.  Stated another way, unlike square and hexagonal puzzles in
//	which each individual token can be interpreted without any additional
//	contextual information, triangular puzzle tokens can only be
//	interpreted correctly when their position in the grid is known.
//-----------------------------------------------------------------------------
static void shrink_puzzle( SokoPuzzle p, int save_file_version )
    {
    char ch, empty_ch;
    int row, first_row, last_row, col, first_col, last_col, non_blank_row;
    int nrows, ncols;

    first_row = p->data->row_count;
    first_col = p->data->col_count;
    last_row = -1;
    last_col = -1;
    empty_ch = (char)(save_file_version == 1 ? NULL_CHAR_V1 : EMPTY_CHAR);

    for (row = 0;  row < p->data->row_count;  row++)
	{
	non_blank_row = 0;
	for (col = 0;  col < p->data->col_count;  col++)
	    {
	    ch = BOARD(row, col);
	    if (ch != empty_ch)
		{
		if (col < first_col)
		    first_col = col;
		if (col > last_col)
		    last_col = col;
		non_blank_row = 1;
		}
	    }
	if (non_blank_row)
	    {
	    if (row < first_row)
		first_row = row;
	    if (row > last_row)
		last_row = row;
	    }
	}

    if (STYLE(p) == SOKO_STYLE_TRIANGLE && (first_col & 1) != 0)
	first_col--;						// *TRIANGLE*

    nrows = (last_row - first_row) + 1;
    ncols = (last_col - first_col) + 1;

    if (nrows != p->data->row_count || ncols != p->data->col_count)
	{
	char* new_board = (char*)malloc(nrows * ncols * sizeof(new_board[0]));
	char* p_new = new_board;
	char const* p_old =
	    p->data->board + first_row * p->data->col_count + first_col;
	for (row=0; row < nrows;
	    row++, p_old += p->data->col_count, p_new += ncols)
	    memcpy( p_new, p_old, ncols );
	free( p->data->board );
	p->data->board = new_board;
	}

    p->data->row_count = nrows;
    p->data->col_count = ncols;
    }


//-----------------------------------------------------------------------------
// translate_in
//	Translate an incoming puzzle file character into an internal
//	identifier.  If save_file_version is -1, then we are translating a
//	puzzle character, rather than a save file character.  For save files,
//	version 1 stored puzzles using a custom character set (which was based
//	upon the internal identifiers used by SokoPuzzle).  For save file
//	version 2 and later, the normal external puzzle characters are used by
//	save files.
//-----------------------------------------------------------------------------
static char translate_in( char input_ch, int save_file_version )
    {
    char ch;
    if (save_file_version == -1 || save_file_version > 1)
	{
	switch (input_ch)
	    {
	    case EMPTY_CHAR:           ch = EMPTY_CH;       break;
	    case WALL_CHAR:            ch = WALL_CH;        break;
	    case CRATE_CHAR:           ch = CRATE_CH;       break;
	    case SAFE_CRATE_CHAR:      ch = SAFE_CRATE_CH;  break;
	    case SAFE_EMPTY_CHAR:      ch = SAFE_EMPTY_CH;  break;
	    case PLAYER_CHAR:          ch = PLAYER_CH;      break;
	    case SAFE_PLAYER_CHAR:     ch = SAFE_PLAYER_CH; break;
	    case SAFE_PLAYER_CHAR_ALT: ch = SAFE_PLAYER_CH; break;
	    default:                   ch = ILLEGAL_CH;     break;
	    }
	}
    else // (save_file_version == 1)
	{
	switch (input_ch)
	    {
	    case EMPTY_CHAR_V1:       ch = EMPTY_CH;       break;
	    case WALL_CHAR_V1:        ch = WALL_CH;        break;
	    case CRATE_CHAR_V1:       ch = CRATE_CH;       break;
	    case SAFE_CRATE_CHAR_V1:  ch = SAFE_CRATE_CH;  break;
	    case SAFE_EMPTY_CHAR_V1:  ch = SAFE_EMPTY_CH;  break;
	    case PLAYER_CHAR_V1:      ch = PLAYER_CH;      break;
	    case SAFE_PLAYER_CHAR_V1: ch = SAFE_PLAYER_CH; break;
	    case NULL_CHAR_V1:        ch = EMPTY_CH;       break;
	    default:                  ch = ILLEGAL_CH;     break;
	    }
	}
    return ch;
    }


//-----------------------------------------------------------------------------
// validate_puzzle
//	Ensure that the puzzle is valid and perform bookkeeping tasks.  Note
//	that the incoming `save_file_version' will be -1 when validating a new
//	puzzle, rather than a puzzle from a saved game file.
//
// CONSTRAINTS
//	Make sure all cells are valid.
//	Make sure there is exactly one player.
//	Make sure all border cells are closed (not accessible by the player).
//
// BOOKKEEPING
//	Find the player's starting position.
//	Mark all unreachable cells as null cells.
//	Count the number of unsafe crates (used later for solved-check).
//-----------------------------------------------------------------------------
soko_bool validate_puzzle( SokoPuzzle p, int save_file_version )
    {
    soko_bool ok = soko_false;
    char const* errmsg = 0;
    soko_bool found = soko_false;
    SokoPuzzleStyle const style = STYLE(p);
    int const ndirs = DirMoveCount[ style ];
    int row, col, oddrow, oddcol, dirs;
    char ch;
    char* map;
    Coord coord;
    Coord* stack;
    int stack_top;

    map = (char*)
	malloc( p->data->row_count * p->data->col_count * sizeof(map[0]) );
    stack = (Coord*)
	malloc( p->data->row_count * p->data->col_count * sizeof(stack[0]) );

    p->data->unsafe_crate_count = 0;

    for (row = 0;  row < p->data->row_count;  row++)
	{
	for (col = 0;  col < p->data->col_count;  col++)
	    {
	    ch = translate_in( BOARD(row, col), save_file_version );
	    if (IS_ILLEGAL(ch))
		{
		errmsg = "Unrecognized characters in puzzle.";
		goto ERR_EXIT;
		}
	    else
		{
		BOARD(row, col) = ch;
		MAP(row, col) = (IS_CLOSED(ch) ? RCH_UNREACHABLE:RCH_UNKNOWN);
		if (IS_CRATE(ch) && !IS_SAFE(ch))
		    p->data->unsafe_crate_count++;
		else if (IS_PLAYER(ch))
		    {
		    if (!found)
			{
			found = soko_true;
			p->data->player_row = row;
			p->data->player_col = col;
			}
		    else
			{
			errmsg = "More than one player in puzzle.";
			goto ERR_EXIT;
			}
		    }
		}
	    }
	}

    if (!found)
	{
	errmsg = "Cannot find player in puzzle.";
	goto ERR_EXIT;
	}

    stack_top = 0;					// "mark"
    MAP(p->data->player_row, p->data->player_col) = RCH_REACHABLE;
    coord.row = p->data->player_row;
    coord.col = p->data->player_col;
    stack[ stack_top++ ] = coord;			// "push"
    while (stack_top > 0)
	{
	coord = stack[ --stack_top ];			// "pop"
	oddrow = (coord.row & 1);
	oddcol = (coord.col & 1);
	for (dirs = 0; dirs < ndirs; dirs++)
	    {
	    SokoDirection const dir = DirMove[style][oddrow][oddcol][dirs];
	    row = coord.row + DirDeltaRow[style][oddrow][oddcol][dir];
	    col = coord.col + DirDeltaCol[style][oddrow][oddcol][dir];
	    if (GOOD_ROW(row) && GOOD_COL(col) && MAP(row, col) == RCH_UNKNOWN)
		{
		MAP(row, col) = RCH_REACHABLE;		// "mark"
		stack[ stack_top ].row = row;
		stack[ stack_top ].col = col;
		stack_top++;				// "push"
		}
	    }
	}

    for (row = 0;  row < p->data->row_count;  row++)
	for (col = 0;  col < p->data->col_count;  col++)
	    if (MAP(row, col) != RCH_REACHABLE)
		{
		ch = BOARD(row, col);
		if (IS_EMPTY(ch))
		    {
		    BOARD(row, col) = NULL_CH;
		    }
		else if (!IS_CLOSED(ch) && !(IS_CRATE(ch) && IS_SAFE(ch)))
		    { // Safe-crate okay even if unreachable, but others not.
		    if (IS_CRATE(ch))
			errmsg = "Puzzle contains unreachable crates.";
		    else if (IS_SAFE(ch))
			errmsg = "Puzzle contains unreachable goal squares.";
		    else
			errmsg = "Internal: reachable error.";
		    goto ERR_EXIT;
		    }
		}

    for (row = 0;  row < p->data->row_count;  row++)
	if (!IS_CLOSED(BOARD(row,0)) ||
	    !IS_CLOSED(BOARD(row,p->data->col_count-1)))
	    {
	    errmsg = "Puzzle is not closed";
	    goto ERR_EXIT;
	    }

    for (col = 0;  col < p->data->col_count;  col++)
	if (!IS_CLOSED(BOARD(0,col)) ||
	    !IS_CLOSED(BOARD(p->data->row_count-1,0)))
	    {
	    errmsg = "Puzzle is not closed";
	    goto ERR_EXIT;
	    }

    ok = soko_true;

ERR_EXIT:
    free( stack );
    free( map );

    if (!ok)
	send_alert( p, SOKO_ALERT_ERROR, "Corrupt Puzzle", errmsg );
    return ok;
    }


//-----------------------------------------------------------------------------
// preallocate_buffers
//	Pre-allocate stacks, maps, and paths for shortest-path calculations
//	and dirty-cell support.
//-----------------------------------------------------------------------------
static void preallocate_buffers( int rows, int cols )
    {
    static int preallocated_area = 0;
    int const area = rows * cols;
    int const stack_max = area * SOKO_DIR_MAX;
    SOKO_ASSERT( area > 0 );
#define SOKO_A_MALLOC(V,T)  V = (T*)malloc( area * sizeof(T) );
#define SOKO_A_REALLOC(V,T) V = (T*)realloc( V, area * sizeof(T) );
#define SOKO_S_MALLOC(V,T)  V = (T*)malloc( stack_max * sizeof(T) );
#define SOKO_S_REALLOC(V,T) V = (T*)realloc( V, stack_max * sizeof(T) );
    if (area > preallocated_area)
	{
	if (preallocated_area == 0)
	    {
	    SOKO_A_MALLOC( PLAYER_PATH,    SokoDirection );
	    SOKO_A_MALLOC( PLAYER_MAP,     int           );
	    SOKO_A_MALLOC( PLAYER_STACK_1, Coord         );
	    SOKO_A_MALLOC( PLAYER_STACK_2, Coord         );
	    SOKO_S_MALLOC( CRATE_PATH,     SokoDirection );
	    SOKO_A_MALLOC( CRATE_MAP,      CrateMapRec   );
	    SOKO_S_MALLOC( CRATE_STACK_1,  Coord         );
	    SOKO_S_MALLOC( CRATE_STACK_2,  Coord         );
	    SOKO_A_MALLOC( DIRTY_MAP,      char          );
	    SOKO_A_MALLOC( DIRTY_LIST,     SokoCell      );
	    }
	else
	    {
	    SOKO_A_REALLOC( PLAYER_PATH,    SokoDirection );
	    SOKO_A_REALLOC( PLAYER_MAP,     int           );
	    SOKO_A_REALLOC( PLAYER_STACK_1, Coord         );
	    SOKO_A_REALLOC( PLAYER_STACK_2, Coord         );
	    SOKO_S_REALLOC( CRATE_PATH,     SokoDirection );
	    SOKO_A_REALLOC( CRATE_MAP,      CrateMapRec   );
	    SOKO_S_REALLOC( CRATE_STACK_1,  Coord         );
	    SOKO_S_REALLOC( CRATE_STACK_2,  Coord         );
	    SOKO_A_REALLOC( DIRTY_MAP,      char          );
	    SOKO_A_REALLOC( DIRTY_LIST,     SokoCell      );
	    }
	preallocated_area = area;
	}
#undef SOKO_S_REALLOC
#undef SOKO_S_ALLOC
#undef SOKO_A_REALLOC
#undef SOKO_A_ALLOC
    }


//-----------------------------------------------------------------------------
// common_init
//-----------------------------------------------------------------------------
static void common_init( SokoPuzzle p, SokoPuzzleDelegate delegate )
    {
    SOKO_ASSERT( delegate != 0 );
    p->data->delegate = delegate;
    p->data->save_file_name = 0;
    p->data->puzzle_file_name = 0;
    p->data->puzzle_style = (SokoPuzzleStyle)SOKO_STYLE_MAX;
    p->data->row_count = -1;
    p->data->col_count = -1;
    p->data->player_row = -1;
    p->data->player_col = -1;
    p->data->board = 0;
    p->data->history_max = 0;
    p->data->history_len = 0;
    p->data->history_pos = 0;
    p->data->history = 0;
    p->data->history_dirty = soko_false;
    p->data->move_count = 0;
    p->data->push_count = 0;
    p->data->run_count = 0;
    p->data->focus_count = 0;
    p->data->unsafe_crate_count = 0;
    p->data->recorded_move_count = 0;
    p->data->recorded_push_count = 0;
    p->data->recorded_run_count = 0;
    p->data->recorded_focus_count = 0;
    p->data->selected_row = -1;
    p->data->selected_col = -1;
    p->data->last_push_row = -1;
    p->data->last_push_col = -1;
    p->data->animating = 0;
    p->data->playback_active = soko_false;
    p->data->playback_target = -1;
    p->data->drag_state.drag_active = soko_false;
    p->data->drag_state.drag_abort = soko_false;
    p->data->drag_state.drag_attempted = soko_false;
    p->data->drag_state.row = -1;
    p->data->drag_state.col = -1;
    p->data->drag_state.callback = 0;
    p->data->meta_data.tuple_count = 0;
    p->data->meta_data.tuple_max = 0;
    p->data->meta_data.tuples = 0;
    p->data->meta_data.raw = 0;
    p->data->meta_data.raw_count = 0;
    p->data->meta_data.raw_max = 0;
    p->data->inhibit_high_score = soko_false;
    p->data->inhibit_auto_save = soko_false;
    p->data->inhibit_auto_advance = soko_false;
    }


//-----------------------------------------------------------------------------
// read_puzzle
//	Read and parse a square, hexagonal, or triangular puzzle.
// *COMMENT*
//	Puzzles in many external puzzle sets contain author comments and
//	meta-data which we must ignore when parsing the puzzle.  Since there is
//	no common convention regarding how this information should be
//	represented in puzzle files, we define comments and meta-data as any
//	non-puzzle character appearing at the beginning of a line (optionally
//	preceded by whitespace), and extending to the end of line.
// *IGNORE*
//	For hexagonal puzzles, by convention, tokens on even rows are placed in
//	even columns, and tokens on odd rows are placed in odd columns
//	(counting from 1).  Other tokens are ignored.  However, some puzzles
//	may not follow convention, and may instead place tokens in odd columns
//	on even rows, and in even columns on odd rows.  Therefore, for
//	robustness, we dynamically determine which scheme is in use by the
//	puzzle.  Also, since path-finding and movement logic is predicated upon
//	the even-on-even/odd-on-odd convention, we forcibly prepend an extra
//	column to odd lines if we find that the puzzle is not following
//	convention.  This ensures that path finding and movement will work
//	correctly.
//-----------------------------------------------------------------------------
static soko_bool read_puzzle(
    SokoPuzzle p, SokoPuzzleStyle style, FILE* fp, char const* path )
    {
    soko_bool ok = soko_false;
    struct stat st;
    errno = 0;
    if (stat( path, &st ) == -1)
	send_alert( p, SOKO_ALERT_ERROR, "Sorry",
	    "Cannot determine size of puzzle file: \"%s\" [%s]", path,
	    strerror(errno) );
    else if (st.st_size == 0)
	send_alert( p, SOKO_ALERT_ERROR, "Sorry",
	    "Puzzle file is empty: \"%s\".", path );
    else
	{
	char* buff = (char*)malloc( st.st_size * sizeof(buff[0]) );
	int const sz = fread( buff, 1, st.st_size, fp );
	if (sz <= 0)
	    send_alert(p, SOKO_ALERT_ERROR, "Sorry",
		"Cannot read puzzle file: \"%s\" [%s].", path,strerror(errno));
	else
	    {
	    int x, y;
	    int ignore_base, first_row;				// *IGNORE*
	    soko_bool comment;					// *COMMENT*
	    char const* b = buff;
	    char const* const blim = buff + sz;

	    ignore_base = -1;
	    first_row = -1;
	    comment = soko_false;
	    p->data->row_count = 0;
	    p->data->col_count = 0;
	    x = 0;
	    for ( ; b < blim; b++)
		{
		char const c = *b;
		if (c != '\n' && c != '\r')
		    {
		    if (comment || strchr( PUZZLE_CHARS, c ) == 0)
			{
			meta_data_append( p, c );
			comment = soko_true;
			}
		    else
			{
			if (ignore_base == -1 &&
			    style == SOKO_STYLE_HEXAGON && c != EMPTY_CHAR)
			    {
			    ignore_base = (x & 1);
			    first_row = p->data->row_count;
			    }
			x++;
			}
		    }
		else
		    {
		    if (comment)
			meta_data_append( p, c );
		    if (c == '\r' && b + 1 < blim && *(b + 1) == '\n')
			b++; // Eat LF, CR, or CRLF.
		    if (x > p->data->col_count)
			p->data->col_count = x;
		    x = 0;
		    p->data->row_count++;
		    comment = soko_false;
		    }
		}
	    if (x != 0) // Missing line terminator.
		p->data->row_count++;
	    if (ignore_base == -1)
		{
		ignore_base = 0;
		first_row = 0;
		}
	    meta_data_close(p);

	    SOKO_ASSERT( p->data->row_count != 0 );
	    if (p->data->col_count == 0)
		send_alert( p, SOKO_ALERT_ERROR, "Sorry",
		    "Puzzle file is empty: \"%s\".", path );
	    else
		{
		int ignore;					// *IGNORE*
		soko_bool whitespace;				// *COMMENT*
		int const board_sz = p->data->row_count *
		    p->data->col_count * sizeof(p->data->board[0]);
		p->data->board = (char*)malloc( board_sz );
		memset( p->data->board, EMPTY_CHAR, board_sz );

		y = 0;
		x = ignore_base;
		ignore = ignore_base;
		comment = soko_false;
		whitespace = soko_true;
		b = buff;
		for ( ; b < blim; b++)
		    {
		    char const c = *b;
		    if (c == '\n' || c == '\r')
			{
			if (c == '\r' && b + 1 < blim && *(b + 1) == '\n')
			    b++; // Eat LF, CR, or CRLF.
			y++;
			x = (((y - first_row) & 1) == 0 ? ignore_base : 0);
			if (style == SOKO_STYLE_HEXAGON)
			    ignore = ignore_base ^ ((y - first_row) & 1);
			comment = soko_false;
			whitespace = soko_true;
			}
		    else if (whitespace && !comment &&
			strchr( PUZZLE_CHARS, c ) == 0)
			{
			comment = soko_true;
			}
		    else if (!comment)
			{
			if (!ignore)
			    {
			    BOARD(y, x) = c;
			    x++;
			    }
			if (style == SOKO_STYLE_HEXAGON)
			    ignore ^= 1;
			if (whitespace && !isspace(c))
			    whitespace = soko_false;
			}
		    }

		shrink_puzzle( p, -1 );
		examine_meta_data(p);
		ok = soko_true;
		}
	    }
	free( buff );
	}
    return ok;
    }


//-----------------------------------------------------------------------------
// load_puzzle
//	Load a square, hexagonal, or triangular puzzle.
// *NOTE-CR*
//	Unfortunately, Microsoft Windows eats bare CR terminators completely
//	when the file is opened in "text" mode.  This makes it impossible to
//	recognize Macintosh-style line termination.  Therefore, use "binary"
//	mode.
//-----------------------------------------------------------------------------
static soko_bool load_puzzle(
    SokoPuzzle p, SokoPuzzleStyle style, SokoPool pool )
    {
    soko_bool ok = soko_false;
    FILE* fp;
    char const* filename =
	soko_denormalize_path( pool, p->data->puzzle_file_name );
    errno = 0;
    if ((fp = fopen( filename, "rb" )) == 0)		// *NOTE-CR*
	send_alert(p, SOKO_ALERT_ERROR, "Sorry",
	    "Cannot open puzzle file: \"%s\" [%s].", filename,strerror(errno));
    else
	{
	if (read_puzzle( p, style, fp, filename ))
	    ok = validate_puzzle( p, -1 );
	fclose(fp);
	}
    return ok;
    }


//-----------------------------------------------------------------------------
// init_new_puzzle
//-----------------------------------------------------------------------------
static soko_bool init_new_puzzle(
    SokoPuzzle p,
    SokoPuzzleDelegate delegate,
    SokoPuzzleStyle style,
    char const* puzzle_name,
    char const* save_name,
    SokoPool pool )
    {
    soko_bool okay = soko_false;
    common_init( p, delegate );
    p->data->puzzle_file_name = soko_strdup(
	soko_normalize_path(pool, soko_expand_path(pool, puzzle_name)) );
    p->data->save_file_name = soko_strdup(
	soko_normalize_path(pool, soko_expand_path(pool, save_name)) );
    p->data->puzzle_style = style;

    if (load_puzzle( p, style, pool ))
	{
	preallocate_buffers( p->data->row_count, p->data->col_count );
	send_solved_state(p);
	okay = soko_true;
	}
    return okay;
    }


//-----------------------------------------------------------------------------
// read_version
//-----------------------------------------------------------------------------
static int read_version( FILE* f )
    {
    int v;
    char s[ FILENAME_MAX ];
    if (soko_read_line(f, s, sizeof(s)) != -1 && sscanf(s, "%d", &v) == 1)
	return v;
    return -1;
    }


//-----------------------------------------------------------------------------
// read_puzzle_name
//-----------------------------------------------------------------------------
static char* read_puzzle_name( FILE* f, int version )
    {
    char* name = 0;
    char s[ FILENAME_MAX ];
    if (soko_read_line( f, s, sizeof(s) ) != -1)
	{
	SokoPool pool = SokoPool_new(0);
	name = soko_strdup(soko_normalize_path(pool,soko_expand_path(pool,s)));
	SokoPool_destroy( pool );
	}
    return name;
    }


//-----------------------------------------------------------------------------
// read_puzzle_style
//	Returns -1 if unable to parse name; else returns style constant.
//-----------------------------------------------------------------------------
static int read_puzzle_style( FILE* f, int version )
    {
    int style = -1;
    if (version < 2)
	style = SOKO_STYLE_SQUARE;
    else
	{
	char s[ FILENAME_MAX ];
	if (soko_read_line( f, s, sizeof(s) ) != -1)
	    for (style = SOKO_STYLE_MAX - 1; style >= 0; style--)
		if (soko_stricmp( s, StyleName[style] ) == 0)
		    break;
	}
    return style;
    }


//-----------------------------------------------------------------------------
// read_history
//-----------------------------------------------------------------------------
static soko_bool read_history( FILE* f, int version, SokoPuzzle p )
    {
    soko_bool ok = soko_false;
    char s[ FILENAME_MAX ];
    if (soko_read_line( f, s, sizeof(s) ) != -1)
	{
	int dummy;
	soko_bool scan_ok;
	if (version < 2)
	    scan_ok = (sscanf( s, "%d %d %d", &dummy,
		&p->data->history_len, &p->data->history_pos ) == 3);
	else
	    scan_ok = (sscanf( s, "%d %d",
		&p->data->history_len, &p->data->history_pos ) == 2);

	if (scan_ok && 0 <= p->data->history_pos &&
	    p->data->history_pos <= p->data->history_len)
	    {
	    p->data->history_max = (p->data->history_len + 1023) & ~1023;
	    p->data->history = (char*)
		malloc( p->data->history_max * sizeof(p->data->history[0]) );
	    if (version < 2)
		ok = soko_run_length_decode_string(
		    f, p->data->history, p->data->history_len );
	    else
		ok = (soko_read_line_terminate( f, p->data->history,
		    p->data->history_max, soko_false ) != -1);
	    }
	}
    return ok;
    }


//-----------------------------------------------------------------------------
// read_scores
//-----------------------------------------------------------------------------
static soko_bool read_scores( FILE* f, int version, SokoPuzzle p )
    {
    soko_bool ok = soko_false;
    char s[ FILENAME_MAX ];
    if (soko_read_line( f, s, sizeof(s) ) != -1)
	{
	if (version < 2)
	    ok = (sscanf( s, "%d %d",
		&p->data->recorded_move_count,
		&p->data->recorded_push_count ) == 2);
	else
	    ok = (sscanf( s, "%d %d %d %d",
		&p->data->recorded_move_count,
		&p->data->recorded_push_count,
		&p->data->recorded_run_count,
		&p->data->recorded_focus_count ) == 4);

	if (ok && version < 2) // Skip old move- and push-count.
	    ok = (soko_read_line(f, s, sizeof(s) ) != -1);
	}
    return ok;
    }


//-----------------------------------------------------------------------------
// read_saved_puzzle_v1
//	Read puzzle data from a version 1 save file.
//
//	Puzzle data in version 1 save files was compressed using an RLE
//	compressor and was encoded using an implementation-specific character
//	set, rather than the normal puzzle file character set.
//
//	Furthermore, empty area surrounding the puzzle was filled with wall
//	tiles in version 1 save files.  In order to make version 1 save files
//	appear similar to present games (in which empty areas are left empty),
//	we manually strip away the excess wall tiles.  This is done by checking
//	all neighbors of each wall (including diagonal neighbors).  If all
//	neighbors are also walls, then the tile is converted to an empty cell.
//	Since this process may introduce new empty space around the old saved
//	puzzle, it may be necessary to shrink the board, thus the call to
//	shrink_puzzle().  Although, there is no way to exactly recover the
//	original puzzle layout (since information about actual empty areas has
//	been lost), the approximation achieved using this technique gives quite
//	creditable results.  In the vast majority of cases the "restored"
//	puzzle is identical to the original puzzle.
//-----------------------------------------------------------------------------
static soko_bool read_saved_puzzle_v1( FILE* f, SokoPuzzle p )
    {
    soko_bool ok = soko_false;
    char s[ FILENAME_MAX ];
    int nr, nc;
    if (soko_read_line( f, s, sizeof(s) ) != -1 &&
	sscanf(s, "%d %d", &nr, &nc) == 2)
	{
	int r;
	int const sz = nr * nc * sizeof(p->data->board[0]);
	char* b;

	p->data->row_count = nr;
	p->data->col_count = nc;
	p->data->board = (char*)malloc(sz);
	memset( p->data->board, EMPTY_CHAR, sz );
	b = p->data->board;

	ok = soko_true;
	for (r = 0; r < nr && ok; r++, b += nc)
	    ok = soko_run_length_decode_string( f, b, nc );

	if (ok)
	    {
	    SokoPuzzleStyle const style = STYLE(p);
	    int const ndirs = DirMoveCount[style];
	    int c, x, y, oddrow, oddcol, dirs, diag;
	    SokoDirection dir;
	    soko_bool null;
	    SOKO_ASSERT( style == SOKO_STYLE_SQUARE );
	    for (r = 0; r < nr; r++)
		{
		for (c = 0; c < nc; c++)
		    {
		    if (BOARD(r,c) == WALL_CHAR_V1)
			{
			null = soko_true;
			for (dirs = 0; null && dirs < ndirs; dirs++)
			    {
			    oddrow = (r & 1);
			    oddcol = (c & 1);
			    dir = DirMove[style][oddrow][oddcol][dirs];
			    y = r + DirDeltaRow[style][oddrow][oddcol][dir];
			    x = c + DirDeltaCol[style][oddrow][oddcol][dir];
			    if (GOOD_ROW(y) && GOOD_COL(x) &&
				BOARD(y,x) != WALL_CHAR_V1 && 
				BOARD(y,x) != NULL_CHAR_V1)
				null = soko_false;
			    }
			for (diag = 0; null && diag < SOKO_DIAG_MAX; diag++)
			    {
			    oddrow = (r & 1);
			    oddcol = (c & 1);
			    dir = DirDiagonalSquare[diag][0];
			    y = r + DirDeltaRow[style][oddrow][oddcol][dir];
			    x = c + DirDeltaCol[style][oddrow][oddcol][dir];
			    oddrow = (y & 1);
			    oddcol = (x & 1);
			    dir = DirDiagonalSquare[diag][1];
			    y += DirDeltaRow[style][oddrow][oddcol][dir];
			    x += DirDeltaCol[style][oddrow][oddcol][dir];
			    if (GOOD_ROW(y) && GOOD_COL(x) &&
				BOARD(y,x) != WALL_CHAR_V1 && 
				BOARD(y,x) != NULL_CHAR_V1)
				null = soko_false;
			    }
			if (null) // Not accessible.
			    BOARD(r,c) = NULL_CHAR_V1;
			}
		    }
		}
	    shrink_puzzle( p, 1 );
	    }
	}
    return ok;
    }


//-----------------------------------------------------------------------------
// read_saved_puzzle
//-----------------------------------------------------------------------------
static soko_bool read_saved_puzzle(
    FILE* f, int version, SokoPuzzle p, char const* path )
    {
    if (version < 2)
	return read_saved_puzzle_v1( f, p );
    else
	return read_puzzle( p, STYLE(p), f, path );
    }


//-----------------------------------------------------------------------------
// compute_scores
//	Number of moves.
//	Number of pushes.
//	Number of directional changes when pushing crates (straight runs).
//	Number of times player's focus changes to a different crate.
// *TRIANGLE*
//	The focus score can be computed directly from the LuRd solution for
//	square and hexagonal puzzles with no additional information.
//	Unfortunately, this is not true of triangular puzzles.  For triangular
//	puzzles, one must also know if the player started on a north- or
//	south-oriented triangle.  (You can work out for yourself on paper why
//	this is so by tracing a small set of moves and pushes starting first
//	from a north-pointing triangle, and then a south-pointing triangle.)
//	Fortunately, it is possible to intuit the player's starting orientation
//	from the saved puzzle data.  If the number of history entries is even,
//	then the orientation of the player's starting location is the same as
//	the orientation of the cell currently occupied by the player;
//	otherwise, the orientation is opposite.
// *MOVES*
//	In the current implementation, number of moves is always equal to the
//	current position in the history.  Therefore, it is possible to avoid
//	computing the number of moves by simply relying upon the history
//	position.  Nevertheless, for robustness, we independently compute the
//	number of moves in case the implementation of the history changes in
//	the future.
//-----------------------------------------------------------------------------
static soko_bool compute_scores( SokoPuzzle p )
    {
    soko_bool ok = soko_true;
    int r = 0, c = 0, crate_r = INT_MAX, crate_c = INT_MAX;
    int moves = 0, pushes = 0, runs = 0, focus = 0;
    SokoPuzzleStyle const style = STYLE(p);
    SokoDirection last_dir = SOKO_DIR_ILLEGAL;
    char const* h = p->data->history;
    char const* const hlim = h + p->data->history_pos;

    if (style == SOKO_STYLE_TRIANGLE &&			// *TRIANGLE*
       ((p->data->history_pos & 1) ^
	(p->data->player_row  & 1) ^
	(p->data->player_col  & 1)) != 0)
	r++;

    for ( ; h < hlim && ok; h++)
	{
	SokoDirection dir = dir_from_move( *h, soko_false );
	if (dir == SOKO_DIR_ILLEGAL)
	    ok = soko_false;
	else
	    {
	    int oddrow = (r & 1);
	    int oddcol = (c & 1);
	    r += DirDeltaRow[style][oddrow][oddcol][dir];
	    c += DirDeltaCol[style][oddrow][oddcol][dir];
	    moves++;
	    if (IS_MOVE(*h))
		last_dir = SOKO_DIR_ILLEGAL;
	    else // IS_PUSH(*h)
		{
		pushes++;
		if (dir != last_dir)
		    {
		    runs++;
		    last_dir = dir;
		    }
		if (r != crate_r || c != crate_c)
		    focus++;
		oddrow = (r & 1);
		oddcol = (c & 1);
		crate_r = r + DirDeltaRow[style][oddrow][oddcol][dir];
		crate_c = c + DirDeltaCol[style][oddrow][oddcol][dir];
		}
	    }
	}

    if (ok)
	{
	p->data->move_count  = moves;		// *MOVES*
	p->data->push_count  = pushes;
	p->data->run_count   = runs;
	p->data->focus_count = focus;
	}
    else
	send_alert( p, SOKO_ALERT_ERROR, "Corrupt",
		"Illegal character in save file: 0x%02X `%c'.", *h, *h );
    return ok;
    }


//-----------------------------------------------------------------------------
// load_save_file
//	Parse a saved game file.
// *NOTE-CR*
//	Unfortunately, Microsoft Windows eats bare CR terminators completely
//	when the file is opened in "text" mode.  This makes it impossible to
//	recognize Macintosh-style line termination.  Therefore, use "binary"
//	mode.
//-----------------------------------------------------------------------------
static soko_bool load_save_file( SokoPuzzle p, SokoPool pool )
    {
    soko_bool ok = soko_false;
    FILE* fp;
    char const* errmsg = 0;
    char const* path;
    int version = -1;

    path = soko_denormalize_path( pool, p->data->save_file_name );
    fp = fopen( path, "rb" );					// *NOTE-CR*
    if (fp == 0)
	errmsg = "Cannot open save file.";
    else if ((version = read_version(fp)) < 0 || SAVE_FILE_VERSION < version)
	errmsg = "Invalid file version.";
    else if ((p->data->puzzle_file_name = read_puzzle_name(fp, version)) == 0)
	errmsg = "Cannot read puzzle name.";
    else if ((p->data->puzzle_style = read_puzzle_style(fp, version)) == -1)
	errmsg = "Cannot parse style name.";
    else if (!read_history(fp, version, p))
	errmsg = "Cannot decode history.";
    else if (!read_scores(fp, version, p))
	errmsg = "Error reading recorded scores.";
    else if (!read_saved_puzzle(fp, version, p, path))
	errmsg = "Corrupted puzzle.";
    if (errmsg == 0)
	ok = validate_puzzle( p, version ) && compute_scores(p);
    else
	errmsg = "Error reading file.";

    if (fp != 0)
	fclose(fp);

    if (errmsg != 0)
	send_alert( p, SOKO_ALERT_ERROR, "Error", errmsg );

    return ok;
    }


//-----------------------------------------------------------------------------
// init_saved_puzzle
//-----------------------------------------------------------------------------
static soko_bool init_saved_puzzle(
    SokoPuzzle p,
    SokoPuzzleDelegate delegate,
    char const* save_name,
    SokoPool pool )
    {
    soko_bool okay = soko_false;
    common_init( p, delegate );
    p->data->save_file_name = soko_strdup(
	soko_normalize_path(pool, soko_expand_path(pool, save_name)) );

    if (load_save_file( p, pool ))
	{
	preallocate_buffers( p->data->row_count, p->data->col_count );
	send_solved_state(p);
	okay = soko_true;
	}

    return okay;
    }


//-----------------------------------------------------------------------------
// allocate_puzzle
//-----------------------------------------------------------------------------
static SokoPuzzle allocate_puzzle( void* info )
    {
    SokoPuzzle p = (SokoPuzzle)malloc( sizeof(struct _SokoPuzzle) );
    p->data = (struct SokoPuzzleData*)malloc( sizeof(struct SokoPuzzleData) );
    p->info = info;
    return p;
    }


//-----------------------------------------------------------------------------
// free_puzzle
//-----------------------------------------------------------------------------
static void free_puzzle( SokoPuzzle p )
    {
    free(p->data);
    free(p);
    }


//-----------------------------------------------------------------------------
// open_new
//-----------------------------------------------------------------------------
static SokoPuzzle open_new(
    SokoPuzzleDelegate delegate,
    SokoPuzzleStyle style,
    char const* path,
    void* info,
    SokoPool pool )
    {
    SokoPuzzle p;
    char const* name = SokoPuzzle_save_name_for_puzzle_name( path, pool );
    p = allocate_puzzle( info );
    if (init_new_puzzle( p, delegate, style, path, name, pool ))
	remember_puzzle(p);
    else
	{
	free_puzzle(p);
	p = 0;
	}
    return p;
    }


//-----------------------------------------------------------------------------
// open_old
//-----------------------------------------------------------------------------
static SokoPuzzle open_old(
    SokoPuzzleDelegate delegate,
    char const* path,
    void* info,
    SokoPool pool )
    {
    SokoPuzzle p = allocate_puzzle( info );
    if (init_saved_puzzle( p, delegate, path, pool ))
	remember_puzzle(p);
    else
	{
	free_puzzle(p);
	p = 0;
	}
    return p;
    }


//-----------------------------------------------------------------------------
// SokoPuzzle_new
//-----------------------------------------------------------------------------
SokoPuzzle SokoPuzzle_new(
    SokoPuzzleDelegate delegate, char const* path, void* info )
    {
    SokoPuzzle p = 0;
    if (path != 0)
	{
	SokoPool pool = SokoPool_new(0);
	if (SokoPuzzle_file_is_save_game( path ))
	    p = open_old( delegate, path, info, pool );
	else
	    {
	    SokoPuzzleExtension const* e = match_puzzle_extension( path );
	    if (e != 0)
		p = open_new( delegate, e->style, path, info, pool );
	    }
	SokoPool_destroy( pool );
	}
    return p;
    }


//-----------------------------------------------------------------------------
// SokoPuzzle_destroy
//-----------------------------------------------------------------------------
void SokoPuzzle_destroy( SokoPuzzle p )
    {
    SOKO_ASSERT( !SokoPuzzle_playback_active(p) );
    forget_puzzle(p);
    if (p->data->save_file_name != 0)
	free( p->data->save_file_name );
    if (p->data->puzzle_file_name != 0)
	free( p->data->puzzle_file_name );
    if (p->data->board != 0)
	free( p->data->board );
    if (p->data->history != 0)
	free( p->data->history );
    if (p->data->meta_data.raw != 0)
	free( p->data->meta_data.raw );
    if (p->data->meta_data.tuple_max != 0)
	{
	int i;
	for (i = 0; i < p->data->meta_data.tuple_count; i++)
	    free(p->data->meta_data.tuples[i].key); // Value is freed with key.
	free( p->data->meta_data.tuples );
	}
    free(p);
    }


//-----------------------------------------------------------------------------
// translate_out
//	Translate an internal puzzle identifier to external save file character
//	for save file version 2.
//-----------------------------------------------------------------------------
static char translate_out( char c )
    {
    soko_bool const safe = IS_SAFE(c);
    char o = EMPTY_CHAR;
    if (IS_WALL(c))
	o = WALL_CHAR;
    else if (safe && IS_CRATE(c))
	o = SAFE_CRATE_CHAR;
    else if (IS_CRATE(c))
	o = CRATE_CHAR;
    else if (safe && IS_PLAYER(c))
	o = SAFE_PLAYER_CHAR;
    else if (IS_PLAYER(c))
	o = PLAYER_CHAR;
    else if (safe && IS_EMPTY(c))
	o = SAFE_EMPTY_CHAR;
    else if (IS_EMPTY(c))
	o = EMPTY_CHAR;
    else if (IS_NULL(c))
	o = EMPTY_CHAR;
    return o;
    }


//-----------------------------------------------------------------------------
// save_to_file
//-----------------------------------------------------------------------------
static soko_bool save_to_file(
    SokoPuzzle p, char const* filename, SokoPool pool )
    {
    soko_bool ok = soko_false;
    FILE* fp;
    char const* const path =
	soko_denormalize_path( pool, soko_expand_path(pool, filename) );
    char const* const dir = soko_directory_part( pool, path );
    if (!soko_mkdirs( pool, dir ))
	send_alert( p, SOKO_ALERT_ERROR, "Sorry",
	    "Cannot create save directory: \"%s\" [%s].", dir,strerror(errno));
    else if ((fp = fopen( path, "w" )) == 0)
	send_alert( p, SOKO_ALERT_ERROR, "Sorry",
	    "Cannot create save file: \"%s\" [%s].", path, strerror(errno) );
    else
	{
	int r,c;
	soko_bool const hex = (STYLE(p) == SOKO_STYLE_HEXAGON);
	int const skip = (hex ? 2 : 1);
	int const rc = p->data->row_count;
	int const nc = p->data->col_count;
	int const sz = nc * (hex ? 2 : 1);
	char* const buff = (char*)malloc( sz );
	fprintf( fp, "%d\n", SAVE_FILE_VERSION );
	fprintf( fp, "%s\n", soko_normalize_path(
	    pool, soko_collapse_path(pool, p->data->puzzle_file_name)) );
	fprintf( fp, "%s\n", StyleName[STYLE(p)] );
	fprintf( fp, "%d %d\n", p->data->history_len, p->data->history_pos );
	fwrite( p->data->history, p->data->history_len, 1, fp );
	fputc( '\n', fp );
	fprintf( fp, "%d %d %d %d\n",
	    p->data->recorded_move_count,
	    p->data->recorded_push_count,
	    p->data->recorded_run_count,
	    p->data->recorded_focus_count );

	for (r = 0; r < rc; r++)
	    {
	    char* b = buff + (hex ? (r & 1) : 0);
	    memset( buff, EMPTY_CHAR, sz );
	    for (c = 0; c < nc; c++, b += skip)
		*b = translate_out( BOARD(r,c) );
	    fwrite( buff, sz, 1, fp );
	    fputc( '\n', fp );
	    }

	if (p->data->meta_data.raw != 0)
	    fwrite( p->data->meta_data.raw, p->data->meta_data.raw_count - 1,
		1, fp ); // Writes (raw_count-1) to avoid emitting null.

	fclose(fp);
	free( buff );

	p->data->history_dirty = soko_false;
	DELEGATE(p)->set_dirty_callback( p, DELEGATE(p), soko_false );
	ok = soko_true;
	}
    return ok;
    }


//-----------------------------------------------------------------------------
// SokoPuzzle_save
//-----------------------------------------------------------------------------
soko_bool SokoPuzzle_save( SokoPuzzle p )
    {
    soko_bool ok;
    SokoPool pool = SokoPool_new(0);
    ok = save_to_file( p, p->data->save_file_name, pool );
    SokoPool_destroy( pool );
    return ok;
    }


//-----------------------------------------------------------------------------
// SokoPuzzle_save_if_dirty
//-----------------------------------------------------------------------------
soko_bool SokoPuzzle_save_if_dirty( SokoPuzzle p )
    {
    if (p->data->history_dirty)
	return SokoPuzzle_save(p);
    return soko_true;
    }


//-----------------------------------------------------------------------------
// SokoPuzzle_save_as
//-----------------------------------------------------------------------------
soko_bool SokoPuzzle_save_as( SokoPuzzle p, char const* path )
    {
    soko_bool ok;
    SokoPool pool = SokoPool_new(0);
    path = soko_add_path_extension( pool,
	soko_basename(pool, path, SAVE_EXTENSION), SAVE_EXTENSION );
    ok = save_to_file( p, path, pool );
    if (ok)
	{
	free( p->data->save_file_name );
	p->data->save_file_name = soko_strdup(
	    soko_normalize_path(pool, soko_expand_path(pool, path)) );
	}
    SokoPool_destroy( pool );
    return ok;
    }


//-----------------------------------------------------------------------------
// SokoPuzzle_save_all_puzzles
//-----------------------------------------------------------------------------
void SokoPuzzle_save_all_puzzles( void )
    {
    int i;
    for (i = 0; i < OPEN_PUZZLES_COUNT; i++)
	SokoPuzzle_save_if_dirty( OPEN_PUZZLES[i] );
    }


//=============================================================================
// Dirty Cell Support
//=============================================================================
//-----------------------------------------------------------------------------
// collect_dirty
//-----------------------------------------------------------------------------
static void collect_dirty( SokoPuzzle p )
    {
    SOKO_ASSERT( DIRTY_MAP != 0 && DIRTY_LIST != 0 );
    SOKO_ASSERT( DIRTY_PUZZLE == 0 || DIRTY_PUZZLE == p );
    if (DIRTY_DEPTH++ == 0)
	{
	DIRTY_PUZZLE = p;
	DIRTY_COUNT = 0;
	memset( DIRTY_MAP, soko_false,
	    p->data->row_count * p->data->col_count * sizeof(DIRTY_MAP[0]) );
	}
    }


//-----------------------------------------------------------------------------
// report_dirty
//-----------------------------------------------------------------------------
static void report_dirty( SokoPuzzle p )
    {
    SOKO_ASSERT( DIRTY_DEPTH > 0 );
    SOKO_ASSERT( DIRTY_PUZZLE == p );
    if (--DIRTY_DEPTH == 0)
	{
	if (DIRTY_COUNT != 0)
	    {
	    int i;
	    for (i = 0; i < DIRTY_COUNT; i++)
		{
		SokoCell* c = DIRTY_LIST + i;
		c->type = CELL_TYPE_FOR_CH[ (int)BOARD(c->row, c->col) ];
		}
	    DELEGATE(p)->refresh_cells_callback(
		p, DELEGATE(p), DIRTY_LIST, DIRTY_COUNT );
	    }
	DIRTY_PUZZLE = 0;
	}
    }


//-----------------------------------------------------------------------------
// mark_dirty
//-----------------------------------------------------------------------------
static void mark_dirty( SokoPuzzle p, int row, int col )
    {
    char* cell;
    SOKO_ASSERT( DIRTY_DEPTH > 0 );
    SOKO_ASSERT( DIRTY_PUZZLE == p );
    cell = &CELL_REF( DIRTY_MAP, row, col );
    if (!*cell) // Cell not already marked dirty.
	{
	SokoCell* dirty = DIRTY_LIST + DIRTY_COUNT++;
	dirty->row = row;
	dirty->col = col;
	*cell = soko_true;
	}
    }


//=============================================================================
// Animation Sequence Support
//
//	Support animation sessions.  During an animation session, refresh of
//	the various controls, such as the score fields, the slider, and the
//	various button states, is temporarily suspended.  Normally these
//	controls are updated after every move, whereas with an animation
//	sequence, they are refreshed only when the animation is complete.
//	Suspending control refresh during animation sequences eliminates a lot
//	of extra drawing, consequently improving animation performance.
//
//=============================================================================
static void disable_control_refresh( SokoPuzzle p )
    {
    SOKO_ASSERT( p->data->animating >= 0);
    p->data->animating++;
    }

static void enable_control_refresh( SokoPuzzle p )
    {
    SOKO_ASSERT( p->data->animating > 0 );
    if (--p->data->animating == 0)
	refresh_controls(p);
    }

static soko_bool should_animate_playback( SokoPuzzle p )
    {
    return DELEGATE(p)->should_animate_playback_callback( p, DELEGATE(p) );
    }

static void begin_animation_sequence( SokoPuzzle p )
    {
    disable_control_refresh(p);
    if (p->data->animating == 1)
	DELEGATE(p)->begin_animation_callback( p, DELEGATE(p) );
    if (!should_animate_playback(p))
	collect_dirty(p);
    }

static void end_animation_sequence( SokoPuzzle p )
    {
    enable_control_refresh(p);
    if (p->data->animating == 0)
	DELEGATE(p)->end_animation_callback( p, DELEGATE(p) );
    if (!should_animate_playback(p))
	report_dirty(p);
    }

soko_bool SokoPuzzle_animation_active( SokoPuzzle p )
    {
    return p->data->animating;
    }


//=============================================================================
// History Playback Session Support
//
//	The redo-push and undo-push buttons, along with the playback-slider,
//	result in the playback of one or more moves from the movement history.
//	When animation is enabled (via an "Animate" switch on the user
//	interface), such playback can become quite time-consuming.  Not only
//	can a lengthy playback become annoying for the user (who must wait for
//	the playback sequence to exhaust itself), but it can also stall the
//	application's run-loop long enough that the underlying windowing system
//	may think that the application has become unresponsive.
//
//	In order to address these problems, SokoPuzzle invokes the delegate's
//	`begin_playback' callback.  It is the responsibility of this method to
//	disable the movement controls on the user interface, and to set up some
//	mechanism by which SokoPuzzle_advance_playback() will be called on a
//	periodic basis to actually perform the playback animation.  The
//	playback session terminates when the playback exhausts itself after a
//	call to SokoPuzzle_advance_playback(); when the client application
//	makes an explicit call to SokoPuzzle_abort_playback(); or when a mouse
//	or keyboard event received from the client merits playback
//	cancellation.  When the playback session terminates (prematurely or
//	not), the delegate's `end_playback' callback is invoked.
//
//=============================================================================
//-----------------------------------------------------------------------------
// begin_playback
//-----------------------------------------------------------------------------
static void begin_playback( SokoPuzzle p, int target )
    {
    SOKO_ASSERT( !p->data->playback_active );
    SOKO_ASSERT( target != p->data->history_pos );
    SOKO_ASSERT( should_animate_playback(p) );
    p->data->playback_active = soko_true;
    p->data->playback_target = target;
    DELEGATE(p)->begin_playback_callback( p, DELEGATE(p) );
    begin_animation_sequence(p);
    }


//-----------------------------------------------------------------------------
// end_playback
//-----------------------------------------------------------------------------
static void end_playback( SokoPuzzle p )
    {
    SOKO_ASSERT( p->data->playback_active );
    p->data->playback_active = soko_false;
    DELEGATE(p)->end_playback_callback( p, DELEGATE(p) );
    end_animation_sequence(p);
    }


//-----------------------------------------------------------------------------
// SokoPuzzle_advance_playback
//-----------------------------------------------------------------------------
void SokoPuzzle_advance_playback( SokoPuzzle p )
    {
    SOKO_ASSERT( p->data->playback_active );
    SOKO_ASSERT( p->data->history_pos != p->data->playback_target );
    if (p->data->history_pos > p->data->playback_target)
	undo_move(p);
    else // (p->data->history_pos < p->data->playback_target)
	redo_move(p);
    if (p->data->history_pos == p->data->playback_target)
	end_playback(p);
    }


//-----------------------------------------------------------------------------
// SokoPuzzle_playback_active
//-----------------------------------------------------------------------------
soko_bool SokoPuzzle_playback_active( SokoPuzzle p )
    {
    return p->data->playback_active;
    }


//-----------------------------------------------------------------------------
// SokoPuzzle_abort_playback
//-----------------------------------------------------------------------------
void SokoPuzzle_abort_playback( SokoPuzzle p )
    {
    if (SokoPuzzle_playback_active(p))
	end_playback(p);
    }


//=============================================================================
// Crate Selection Support
//=============================================================================
//-----------------------------------------------------------------------------
// deselect_crate
//-----------------------------------------------------------------------------
static void deselect_crate( SokoPuzzle p )
    {
    if (IS_CRATE_SELECTED)
	{
	char* const b = &BOARD(p->data->selected_row, p->data->selected_col);
	SOKO_ASSERT( IS_CRATE(*b) );
	collect_dirty(p);
	*b &= ~(SELECTED_BIT);
	mark_dirty( p, p->data->selected_row, p->data->selected_col );
	p->data->selected_row = -1;
	p->data->selected_col = -1;
	report_dirty(p);
	}
    }


//-----------------------------------------------------------------------------
// select_crate
//-----------------------------------------------------------------------------
static void select_crate( SokoPuzzle p, int row, int col )
    {
    char* const b = &BOARD(row, col);
    SOKO_ASSERT( IS_CRATE(*b) );
    collect_dirty(p);
    deselect_crate(p);
    *b |= SELECTED_BIT;
    mark_dirty( p, row, col );
    p->data->selected_row = row;
    p->data->selected_col = col;
    report_dirty(p);
    }


//=============================================================================
// History Support
//=============================================================================
//-----------------------------------------------------------------------------
// undo_move
//-----------------------------------------------------------------------------
static void undo_move( SokoPuzzle p )
    {
    SokoPuzzleStyle const style = STYLE(p);
    int src_row,src_col, dst_row,dst_col, ddst_row,ddst_col;
    char ch,src_ch,dst_ch,ddst_ch,move_ch;
    soko_bool is_push;
    int oddrow, oddcol;
    SokoDirection dir, odir;

    collect_dirty(p);
    deselect_crate(p);

    SOKO_ASSERT( p->data->history_pos <= p->data->history_len );
    SOKO_ASSERT( p->data->history_pos > 0 );
    move_ch = p->data->history[ --p->data->history_pos ];
    is_push = IS_PUSH(move_ch);
    dir = dir_from_move( move_ch, soko_true );
    odir = DirOpposite[ dir ];

    dst_row  = p->data->player_row;
    dst_col  = p->data->player_col;
    SOKO_ASSERT( GOOD_ROW(dst_row) && GOOD_COL(dst_col) );
    dst_ch   = BOARD(dst_row, dst_col);
    SOKO_ASSERT( IS_PLAYER(dst_ch) );

    oddrow   = (dst_row & 1);
    oddcol   = (dst_col & 1);
    src_row  = dst_row + DirDeltaRow[ style ][ oddrow ][ oddcol ][ odir ];
    src_col  = dst_col + DirDeltaCol[ style ][ oddrow ][ oddcol ][ odir ];
    SOKO_ASSERT( GOOD_ROW(src_row) && GOOD_COL(src_col) );
    src_ch   = BOARD(src_row, src_col);
    SOKO_ASSERT( IS_EMPTY(src_ch) );

    ddst_row  = dst_row + DirDeltaRow[ style ][ oddrow ][ oddcol ][ dir ];
    ddst_col  = dst_col + DirDeltaCol[ style ][ oddrow ][ oddcol ][ dir ];
    ddst_ch   = WALL_CH;
    if (is_push)
	{
	SOKO_ASSERT( GOOD_ROW(ddst_row) && GOOD_COL(ddst_col) );
	ddst_ch   = BOARD(ddst_row, ddst_col);
	SOKO_ASSERT( IS_CRATE(ddst_ch) );
	}

    ch = IS_SAFE(src_ch) ? SAFE_PLAYER_CH : PLAYER_CH;
    BOARD(src_row, src_col) = ch;
    mark_dirty( p, src_row, src_col );

    if (is_push)
	{
	ch = IS_SAFE(dst_ch) ? SAFE_CRATE_CH : CRATE_CH;
	BOARD(dst_row, dst_col) = ch;
	mark_dirty( p, dst_row, dst_col );

	ch = IS_SAFE(ddst_ch) ? SAFE_EMPTY_CH : EMPTY_CH;
	BOARD(ddst_row, ddst_col) = ch;
	mark_dirty( p, ddst_row, ddst_col );
	}
    else
	{
	ch = IS_SAFE(dst_ch) ? SAFE_EMPTY_CH : EMPTY_CH;
	BOARD(dst_row, dst_col) = ch;
	mark_dirty( p, dst_row, dst_col );
	}

    p->data->player_row = src_row;
    p->data->player_col = src_col;

    p->data->move_count--;
    if (is_push)
	{
	p->data->push_count--;

	if (p->data->history_pos == 0 ||
	    move_ch != p->data->history[ p->data->history_pos - 1 ])
	    {
	    int i, r, c;
	    p->data->run_count--;

	    r = src_row;
	    c = src_col;
	    for (i = p->data->history_pos - 1; i >= 0; i--)
		{
		oddrow = (r & 1);
		oddcol = (c & 1);
		ch = p->data->history[i];
		dir = dir_from_move( ch, soko_true );
		if (IS_PUSH(ch))
		    {
		    r += DirDeltaRow[ style ][ oddrow ][ oddcol ][ dir ];
		    c += DirDeltaCol[ style ][ oddrow ][ oddcol ][ dir ];
		    if (r != dst_row || c != dst_col)
			p->data->focus_count--;
		    p->data->last_push_row = r;
		    p->data->last_push_col = c;
		    break;
		    }
		odir = DirOpposite[ dir ];
		r += DirDeltaRow[ style ][ oddrow ][oddcol][ odir ];
		c += DirDeltaCol[ style ][ oddrow ][oddcol][ odir ];
		}

	    if (i < 0)
		{
		p->data->focus_count--;
		p->data->last_push_row = -1;
		p->data->last_push_col = -1;
		}
	    }
	else
	    {
	    p->data->last_push_row = dst_row;
	    p->data->last_push_col = dst_col;
	    }
	}

    if (p->data->animating == 0)
	refresh_controls(p);
    report_dirty(p);

    if (is_push)
	solved_check( p, ddst_ch, dst_ch );
    }


//-----------------------------------------------------------------------------
// SokoPuzzle_undo_move
//-----------------------------------------------------------------------------
void SokoPuzzle_undo_move( SokoPuzzle p )
    {
    SOKO_ASSERT( !SokoPuzzle_playback_active(p) );
    if (p->data->history_pos > 0)
	undo_move(p);
    }


//-----------------------------------------------------------------------------
// redo_move
//-----------------------------------------------------------------------------
static void redo_move( SokoPuzzle p )
    {
    SokoPuzzleStyle const style = STYLE(p);
    int src_row,src_col, dst_row,dst_col, ddst_row,ddst_col;
    char ch,src_ch,dst_ch,ddst_ch,move_ch;
    soko_bool is_push;
    int oddrow, oddcol;
    SokoDirection dir;

    collect_dirty(p);
    deselect_crate(p);

    SOKO_ASSERT( p->data->history_pos >= 0 );
    SOKO_ASSERT( p->data->history_pos < p->data->history_len );
    move_ch = p->data->history[ p->data->history_pos++ ];
    is_push = IS_PUSH(move_ch);
    dir = dir_from_move( move_ch, soko_true );

    src_row  = p->data->player_row;
    src_col  = p->data->player_col;
    SOKO_ASSERT( GOOD_ROW(src_row) && GOOD_COL(src_col) );
    src_ch   = BOARD(src_row, src_col);
    SOKO_ASSERT( IS_PLAYER(src_ch) );

    oddrow   = (src_row & 1);
    oddcol   = (src_col & 1);
    dst_row  = src_row + DirDeltaRow[ style ][ oddrow ][ oddcol ][ dir ];
    dst_col  = src_col + DirDeltaCol[ style ][ oddrow ][ oddcol ][ dir ];
    SOKO_ASSERT( GOOD_ROW(dst_row) && GOOD_COL(dst_col) );
    dst_ch   = BOARD(dst_row, dst_col);
    SOKO_ASSERT( (is_push && IS_CRATE(dst_ch)) ||
		(!is_push && IS_EMPTY(dst_ch)) );

    oddrow    = (dst_row & 1);
    oddcol    = (dst_col & 1);
    ddst_row  = dst_row + DirDeltaRow[ style ][ oddrow ][ oddcol ][ dir ];
    ddst_col  = dst_col + DirDeltaCol[ style ][ oddrow ][ oddcol ][ dir ];
    ddst_ch   = WALL_CH;
    if (is_push)
	{
	SOKO_ASSERT( GOOD_ROW(ddst_row) && GOOD_COL(ddst_col) );
	ddst_ch   = BOARD(ddst_row, ddst_col);
	SOKO_ASSERT( IS_EMPTY(ddst_ch) );
	}

    ch = IS_SAFE(src_ch) ? SAFE_EMPTY_CH : EMPTY_CH;
    BOARD(src_row, src_col) = ch;
    mark_dirty( p, src_row, src_col );

    ch = IS_SAFE(dst_ch) ? SAFE_PLAYER_CH : PLAYER_CH;
    BOARD(dst_row, dst_col) = ch;
    mark_dirty( p, dst_row, dst_col );

    if (is_push)
	{
	ch = IS_SAFE(ddst_ch) ? SAFE_CRATE_CH : CRATE_CH;
	BOARD(ddst_row, ddst_col) = ch;
	mark_dirty( p, ddst_row, ddst_col );
	}

    p->data->player_row = dst_row;
    p->data->player_col = dst_col;

    p->data->move_count++;
    if (is_push)
	{
	p->data->push_count++;

	if (p->data->history_pos == 1 ||
	    move_ch != p->data->history[ p->data->history_pos - 2 ])
	    {
	    p->data->run_count++;
	    if (dst_row != p->data->last_push_row ||
		dst_col != p->data->last_push_col)
		p->data->focus_count++;
	    }

	p->data->last_push_row = ddst_row;
	p->data->last_push_col = ddst_col;
	}

    if (p->data->animating == 0)
	refresh_controls(p);
    report_dirty(p);

    if (is_push)
	solved_check( p, dst_ch, ddst_ch );
    }


//-----------------------------------------------------------------------------
// SokoPuzzle_redo_move
//-----------------------------------------------------------------------------
void SokoPuzzle_redo_move( SokoPuzzle p )
    {
    SOKO_ASSERT( !SokoPuzzle_playback_active(p) );
    if (p->data->history_pos < p->data->history_len)
	redo_move(p);
    }


//-----------------------------------------------------------------------------
// SokoPuzzle_reenact_history
//	Play back, and optionally animate, a portion of the game history stack.
//	If the playback session is longer than the delegate-selected threshold,
//	then begin an interruptable playback session.  Otherwise, perform the
//	entire playback without giving the user a chance to interrupt it.  For
//	interruptable sessions, the client program must call
//	SokoPuzzle_advance_playback() on a periodic basis for the animation to
//	occur.  SokoPuzzle_abort_playback() can also be called to prematurely
//	abort the playback session.
//-----------------------------------------------------------------------------
void SokoPuzzle_reenact_history( SokoPuzzle p, int new_pos )
    {
    SOKO_ASSERT( new_pos >=0 );
    SOKO_ASSERT( new_pos <= p->data->history_len );
    SOKO_ASSERT( !SokoPuzzle_playback_active(p) );

    if (new_pos != p->data->history_pos)
	{
	soko_bool interruptable = soko_false;
	if (should_animate_playback(p))
	    {
	    int threshold =
		DELEGATE(p)->playback_threshold_callback( p, DELEGATE(p) );
	    SOKO_ASSERT( threshold >= 0 );
	    interruptable = (threshold <= abs(new_pos - p->data->history_pos));
	    }

	if (interruptable)
	    begin_playback( p, new_pos );
	else
	    {
	    begin_animation_sequence(p); // Defer control refresh.
	    if (p->data->history_pos > new_pos)
		{
		while (p->data->history_pos > new_pos)
		    undo_move(p);
		}
	    else // (p->data->history_pos < new_pos)
		{
		while (p->data->history_pos < new_pos)
		    redo_move(p);
		}
	    end_animation_sequence(p); // Refresh controls.
	    }
	}
    }


//-----------------------------------------------------------------------------
// undo_push
//	Smart undo-push.  If the last action was a push, undo all pushes in
//	same direction until a push in a different direction, a move, or the
//	history stack is exhausted.  If the last action was a move, undo all
//	moves until a push is encountered or the history stack is exhausted.
//-----------------------------------------------------------------------------
static void undo_push( SokoPuzzle p )
    {
    int new_pos = -1;
    SOKO_ASSERT( p->data->history_pos > 0 );
    if (p->data->history_pos > 1)
	{
	char const ch = p->data->history[ p->data->history_pos - 1 ];
	if (IS_MOVE(ch))
	    {
	    for (new_pos = p->data->history_pos - 2; new_pos >= 0; new_pos--)
		if (IS_PUSH( p->data->history[new_pos] ))
		    break;
	    }
	else // IS_PUSH(ch)
	    {
	    for (new_pos = p->data->history_pos - 2; new_pos >= 0; new_pos--)
		if (ch != p->data->history[new_pos])
		    break;
	    }
	}
    SokoPuzzle_reenact_history( p, new_pos + 1 );
    }


//-----------------------------------------------------------------------------
// SokoPuzzle_undo_push
//-----------------------------------------------------------------------------
void SokoPuzzle_undo_push( SokoPuzzle p )
    {
    if (!SokoPuzzle_playback_active(p) && p->data->history_pos > 0)
	undo_push(p);
    }


//-----------------------------------------------------------------------------
// redo_push
//	Smart redo-push.  If the next action is a push, redo all pushes in same
//	direction until a push in a different direction, a move, or the history
//	stack is exhausted.  If the next action is a move, redo all moves until
//	a push is encountered or the history stack is exhausted.
//-----------------------------------------------------------------------------
static void redo_push( SokoPuzzle p )
    {
    int new_pos = p->data->history_len;
    SOKO_ASSERT( p->data->history_pos >= 0 );
    SOKO_ASSERT( p->data->history_pos < p->data->history_len );
    if (p->data->history_pos < p->data->history_len - 1)
	{
	char const ch = p->data->history[ p->data->history_pos ];
	if (IS_MOVE(ch))
	    {
	    for (new_pos=p->data->history_pos+1;
		new_pos < p->data->history_len; new_pos++)
		if (IS_PUSH( p->data->history[new_pos] ))
		    break;
	    }
	else // IS_PUSH(c)
	    {
	    for (new_pos=p->data->history_pos+1;
		new_pos < p->data->history_len; new_pos++)
		if (ch != p->data->history[new_pos])
		    break;
	    }
	}
    SokoPuzzle_reenact_history( p, new_pos );
    }


//-----------------------------------------------------------------------------
// SokoPuzzle_redo_push
//-----------------------------------------------------------------------------
void SokoPuzzle_redo_push( SokoPuzzle p )
    {
    if (!SokoPuzzle_playback_active(p) &&
	p->data->history_pos < p->data->history_len)
	redo_push(p);
    }


//=============================================================================
// Drag Support
//=============================================================================
//-----------------------------------------------------------------------------
// begin_drag
//-----------------------------------------------------------------------------
static void begin_drag( SokoPuzzle p, int row, int col, SokoDragCallback c )
    {
    SOKO_ASSERT( !p->data->drag_state.drag_active );
    p->data->drag_state.drag_active = soko_true;
    p->data->drag_state.drag_abort = soko_false;
    p->data->drag_state.drag_attempted = soko_false;
    p->data->drag_state.row = row;
    p->data->drag_state.col = col;
    p->data->drag_state.callback = c;
    DELEGATE(p)->begin_drag_callback( p, DELEGATE(p) );
    }


//-----------------------------------------------------------------------------
// end_drag
//-----------------------------------------------------------------------------
static void end_drag( SokoPuzzle p )
    {
    SOKO_ASSERT( p->data->drag_state.drag_active );
    p->data->drag_state.drag_active = soko_false;
    p->data->drag_state.callback = 0;
    DELEGATE(p)->end_drag_callback( p, DELEGATE(p) );
    }


//-----------------------------------------------------------------------------
// abort_drag
//-----------------------------------------------------------------------------
static void abort_drag( SokoPuzzle p )
    {
    SOKO_ASSERT( p->data->drag_state.drag_active );
    p->data->drag_state.drag_abort = soko_true;
    }


//-----------------------------------------------------------------------------
// drag_aborted
//-----------------------------------------------------------------------------
static soko_bool drag_aborted( SokoPuzzle p )
    {
    return p->data->drag_state.drag_abort;
    }


//-----------------------------------------------------------------------------
// update_drag
//-----------------------------------------------------------------------------
static void update_drag( SokoPuzzle p, int row, int col )
    {
    SOKO_ASSERT( p->data->drag_state.drag_active );
    if (GOOD_ROW( row ) && GOOD_COL( col ) &&
	(row != p->data->drag_state.row || col != p->data->drag_state.col))
	{
	p->data->drag_state.drag_attempted = soko_true;
	if (p->data->drag_state.callback( p, row, col ))
	    {
	    p->data->drag_state.row = row;
	    p->data->drag_state.col = col;
	    }
	}
    }


//-----------------------------------------------------------------------------
// SokoPuzzle_drag_active
//-----------------------------------------------------------------------------
soko_bool SokoPuzzle_drag_active( SokoPuzzle p )
    {
    return p->data->drag_state.drag_active;
    }


//-----------------------------------------------------------------------------
// await_drag
//	Provide the user with the opportunity to drag an object around.  The
//	specified callback function is invoked each time the mouse moves to a
//	different cell.  Returns true if user attempted drag.
//-----------------------------------------------------------------------------
static soko_bool await_drag( SokoPuzzle p, int r, int c, SokoDragCallback f )
    {
    SOKO_ASSERT( IS_CRATE(BOARD(r,c)) || IS_PLAYER(BOARD(r,c)) );
    disable_control_refresh(p);
    begin_drag( p, r, c, f );
    while (!drag_aborted(p))
	DELEGATE(p)->process_events_callback( p, DELEGATE(p), soko_true );
    end_drag(p);
    enable_control_refresh(p);
    return p->data->drag_state.drag_attempted;
    }


//=============================================================================
// Movement Support
//=============================================================================
//-----------------------------------------------------------------------------
// SokoPuzzle_move
//	Move the player and optionally push a single crate in the indicated
//	direction.  Returns soko_true if movement succeeded, else soko_false.
//-----------------------------------------------------------------------------
soko_bool SokoPuzzle_move( SokoPuzzle p, SokoDirection dir )
    {
    SokoPuzzleStyle const style = STYLE(p);
    char ch,dst_ch,ddst_ch;
    int row,col, dst_row,dst_col, ddst_row, ddst_col;
    int oddrow, oddcol;
    soko_bool ok = soko_true;

    SOKO_ASSERT( !SokoPuzzle_playback_active(p) );

    row = p->data->player_row;
    col = p->data->player_col;

    oddrow = (row & 1);
    oddcol = (col & 1);
    dst_row = row + DirDeltaRow[ style ][ oddrow ][ oddcol ][ dir ];
    dst_col = col + DirDeltaCol[ style ][ oddrow ][ oddcol ][ dir ];

    oddrow = (dst_row & 1);
    oddcol = (dst_col & 1);
    ddst_row = dst_row + DirDeltaRow[ style ][ oddrow ][ oddcol ][ dir ];
    ddst_col = dst_col + DirDeltaCol[ style ][ oddrow ][ oddcol ][ dir ];

    collect_dirty(p);
    deselect_crate(p);

    SOKO_ASSERT( GOOD_ROW(row) && GOOD_COL(col) );
    ch = BOARD(row, col);

    SOKO_ASSERT( GOOD_ROW(dst_row) && GOOD_COL(dst_col) );
    dst_ch = BOARD(dst_row, dst_col);

    if (GOOD_ROW(ddst_row) && GOOD_COL(ddst_col))
	ddst_ch = BOARD(ddst_row, ddst_col);
    else
	ddst_ch = WALL_CH;

    if (IS_EMPTY(dst_ch))	// It's a move (not a push).
	{
	if (p->data->history_pos > 0 // Is it the opposite of the last move?
	&&  p->data->history[ p->data->history_pos - 1 ] ==
	    ch_for_move( p, DirOpposite[dir], soko_false ))
	    {
	    undo_move(p);	// Yes, implicitly undo the last move.
	    ok = soko_true;
	    }
	else
	    {
	    if (ch == SAFE_PLAYER_CH)
		BOARD(row, col) = SAFE_EMPTY_CH;
	    else
		{
		SOKO_ASSERT( ch == PLAYER_CH );
		BOARD(row, col) = EMPTY_CH;
		}
	    mark_dirty( p, row, col );

	    if (dst_ch == SAFE_EMPTY_CH)
		BOARD(dst_row, dst_col) = SAFE_PLAYER_CH;
	    else
		{
		SOKO_ASSERT( dst_ch == EMPTY_CH );
		BOARD(dst_row, dst_col) = PLAYER_CH;
		}
	    mark_dirty( p, dst_row, dst_col );

	    p->data->player_row = dst_row;
	    p->data->player_col = dst_col;
	    append_move_to_history( p, dir, soko_false );
	    }
	}
    else if (IS_CRATE(dst_ch) && IS_EMPTY(ddst_ch))
	{
	if (ch == SAFE_PLAYER_CH)
	    BOARD(row, col) = SAFE_EMPTY_CH;
	else
	    {
	    SOKO_ASSERT( ch == PLAYER_CH );
	    BOARD(row, col) = EMPTY_CH;
	    }
	mark_dirty( p, row, col );

	if (dst_ch == SAFE_CRATE_CH)
	    BOARD(dst_row, dst_col) = SAFE_PLAYER_CH;
	else
	    {
	    SOKO_ASSERT( dst_ch == CRATE_CH );
	    BOARD(dst_row, dst_col) = PLAYER_CH;
	    }
	mark_dirty( p, dst_row, dst_col );

	if (ddst_ch == SAFE_EMPTY_CH)
	    BOARD(ddst_row, ddst_col) = SAFE_CRATE_CH;
	else
	    {
	    SOKO_ASSERT( ddst_ch == EMPTY_CH );
	    BOARD(ddst_row, ddst_col) = CRATE_CH;
	    }
	mark_dirty( p, ddst_row, ddst_col );

	p->data->player_row = dst_row;
	p->data->player_col = dst_col;
	append_move_to_history( p, dir, soko_true );
	solved_check( p, dst_ch, ddst_ch );
	}
    else
	{
	ok = soko_false;		// Illegal move.
	}

    report_dirty(p);
    return ok;
    }


//-----------------------------------------------------------------------------
// can_move_from
//	Raw movement test from a specified position.  This tests only if a move
//	is allowed.
//-----------------------------------------------------------------------------
static soko_bool can_move_from( SokoPuzzle p, SokoDirection dir,
    int row, int col, soko_bool* is_push )
    {
    soko_bool ok = soko_false;
    int ch;
    SokoPuzzleStyle const style = STYLE(p);
    int oddrow = (row & 1);
    int oddcol = (col & 1);
    row += DirDeltaRow[style][oddrow][oddcol][dir];
    col += DirDeltaCol[style][oddrow][oddcol][dir];
    ch = BOARD(row, col);
    if (IS_EMPTY(ch))
	{
	ok = soko_true;
	*is_push = soko_false;
	}
    else if (IS_CRATE(ch))
	{
	oddrow = (row & 1);
	oddcol = (col & 1);
	row += DirDeltaRow[style][oddrow][oddcol][dir];
	col += DirDeltaCol[style][oddrow][oddcol][dir];
	ch = BOARD(row, col);
	ok = IS_EMPTY(ch);
	*is_push = soko_true;
	}
    return ok;
    }


//-----------------------------------------------------------------------------
// can_move
//	Raw movement test from the player's position.  This tests only if a
//	move is allowed.
//-----------------------------------------------------------------------------
static soko_bool can_move(SokoPuzzle p, SokoDirection dir, soko_bool* is_push)
    {
    return can_move_from(
	p, dir, p->data->player_row, p->data->player_col, is_push );
    }


//-----------------------------------------------------------------------------
// SokoPuzzle_can_move
//	Raw movement test.  This tests only if a move is allowed.  For a test
//	which also takes user-interface constraints into account, see
//	SokoPuzzle_can_move_constrained().
//-----------------------------------------------------------------------------
soko_bool SokoPuzzle_can_move( SokoPuzzle p, SokoDirection dir )
    {
    soko_bool is_push;
    return can_move( p, dir, &is_push );
    }


//-----------------------------------------------------------------------------
// SokoPuzzle_can_move_constrained
//	Take user-interface constraints into account when checking if movement
//	in a particular direction is allowed.
// *SQUARE*
//	No special constraints.
// *HEXAGON*
//	Because of the orientation of the tiles, literal up and down movements
//	never make sense, so the "Up" and "Down" direction buttons in the
//	user-interface should always be disabled.
// *TRIANGLE*
//	For some directions, the orientation of a triangular tile determines
//	which moves are allowed.  This may be further constrained by whether
//	this is a move or a push.  For moves, only literal up and down make
//	sense.  Up is only used for a south-pointing triangle, and down for a
//	north-pointing triangle.  Left and right are always enabled so that the
//	user can repeatedly press the same button to move the player in an
//	apparent straight line.  When pushing, the crate moves along in front
//	of the player in that same apparent line.
//-----------------------------------------------------------------------------
soko_bool SokoPuzzle_can_move_constrained( SokoPuzzle p, SokoDirection dir )
    {
    soko_bool ok = soko_false;
    soko_bool is_push;
    switch (STYLE(p))
	{
	case SOKO_STYLE_SQUARE:
	    ok = can_move( p, dir, &is_push );
	    break;
	case SOKO_STYLE_HEXAGON:
	    if (dir != SOKO_DIR_UP && dir != SOKO_DIR_DOWN)
		ok = can_move( p, dir, &is_push );
	    break;
	case SOKO_STYLE_TRIANGLE:
	    if (can_move( p, dir, &is_push ))
		{
		if (dir == SOKO_DIR_LEFT || dir == SOKO_DIR_RIGHT)
		    ok = soko_true;
		else if (!is_push)
		    {
		    if (((p->data->player_row&1)^(p->data->player_col&1))==0)
			ok = (dir == SOKO_DIR_UP);
		    else
			ok = (dir == SOKO_DIR_DOWN);
		    }
		}
	    break;
	default:
	    break;
	}
    return ok;
    }


//-----------------------------------------------------------------------------
// try_move_diagonal_square
//	Check if the player can move diagonally in a square-tiled puzzle.  A
//	diagonal movement is composed of either a horizontal movement followed
//	by a vertical movement, or a vertical movement followed by a horizontal
//	movement, depending upon which cells are empty.  If neither option
//	succeeds, then the player can not be moved.
//-----------------------------------------------------------------------------
static soko_bool try_move_diagonal_square(
    SokoPuzzle p, SokoDirection a, SokoDirection b, soko_bool* is_push )
    {
    soko_bool ok = soko_false;
    int r = p->data->player_row;
    int c = p->data->player_col;
    if (can_move_from( p, a, r, c, is_push ))
	{
	soko_bool push_test;
	SokoPuzzleStyle const style = STYLE(p);
	int const oddrow = (r & 1);
	int const oddcol = (c & 1);
	SOKO_ASSERT( style == SOKO_STYLE_SQUARE );
	r += DirDeltaRow[ style ][ oddrow ][ oddcol ][a];
	c += DirDeltaCol[ style ][ oddrow ][ oddcol ][a];
	ok = can_move_from( p, b, r, c, &push_test );
	*is_push |= push_test;
	}
    return ok;
    }


//-----------------------------------------------------------------------------
// can_move_diagonal_square
//	Diagonal movement is simulated by horizontal followed by vertical
//	movement, or vice versa.  If either approach works, but one involes a
//	push, while the other does not, choose the one which does not.
//-----------------------------------------------------------------------------
static soko_bool can_move_diagonal_square(
    SokoPuzzle p, SokoDiagonal dir, soko_bool* is_push, soko_bool do_move )
    {
    soko_bool ok;
    SokoDirection d1 = SOKO_DIR_ILLEGAL, d2 = SOKO_DIR_ILLEGAL;
    SokoDirection const a = DirDiagonalSquare[ dir ][0];
    SokoDirection const b = DirDiagonalSquare[ dir ][1];
    soko_bool ok1, ok2, push1, push2;
    ok1 = try_move_diagonal_square( p, a, b, &push1 );
    ok2 = try_move_diagonal_square( p, b, a, &push2 );
    ok = ok1 || ok2;
    if (ok1 && ok2 && push1) // Choose move over push.
	ok1 = soko_false;
    if (ok1)
	{
	d1 = a;
	d2 = b;
	*is_push = push1;
	}
    else if (ok2)
	{
	d1 = b;
	d2 = a;
	*is_push = push2;
	}
    if (ok && do_move)
	{
	SokoPuzzle_move( p, d1 );
	SokoPuzzle_move( p, d2 );
	}
    return ok;
    }


//-----------------------------------------------------------------------------
// can_move_diagonal
//	Raw diagonal movement test and optional operation.  This tests only if
//	a move is allowed.
// *SQUARE*
//	Diagonal movement is simulated by horizontal followed by vertical
//	movement, or vice versa.  If either approach works, but one involes a
//	push, while the other does not, choose the one which does not.
// *HEXAGON*
//	Diagonal movement maps directly to the standard movements directions
//	up, down, north, and south.
// *TRIANGLE*
//	Diagonal movement maps directly to the standard movements directions
//	up, down, north, south, left, and right.
//-----------------------------------------------------------------------------
static soko_bool can_move_diagonal(
    SokoPuzzle p, SokoDiagonal dir, soko_bool* is_push, soko_bool do_move )
    {
    soko_bool ok;
    SokoPuzzleStyle const style = STYLE(p);
    if (style == SOKO_STYLE_SQUARE)
	{
	ok = can_move_diagonal_square( p, dir, is_push, do_move );
	}
    else // (style == SOKO_STYLE_HEXAGON || style == SOKO_STYLE_TRIANGLE)
	{
	SokoDirection const d = DirDiagonalHexTri[ dir ];
	ok = can_move( p, d, is_push );
	if (ok && do_move)
	    SokoPuzzle_move( p, d );
	}
    return ok;
    }


//-----------------------------------------------------------------------------
// SokoPuzzle_can_move_diagonal
//	Raw diagonal movement test.  This tests only if a move is allowed.  For
//	a test which also takes user-interface constraints into account, see
//	SokoPuzzle_can_move_diagonal_constrained().
//-----------------------------------------------------------------------------
soko_bool SokoPuzzle_can_move_diagonal( SokoPuzzle p, SokoDiagonal dir )
    {
    soko_bool is_push;
    return can_move_diagonal( p, dir, &is_push, soko_false );
    }


//-----------------------------------------------------------------------------
// SokoPuzzle_can_move_diagonal_constrained
//	Take user-interface constraints into account when checking if movement
//	in a diagonal direction is allowed.
// *SQUARE*
//	No special constraints.
// *HEXAGON*
//	No special constraints.
// *TRIANGLE*
//	No special constraints.  Diagonal directions are appropriate in all
//	cases, and are not affected by triangle orientation or whether this is
//	a move or a push.  By remaining enabled, the user can repeatedly press
//	the same diagonal button in order to move in an apparent straight line.
//	When pushing, the crate moves along in front of the player in that same
//	apparent line.
//-----------------------------------------------------------------------------
soko_bool SokoPuzzle_can_move_diagonal_constrained(
    SokoPuzzle p, SokoDiagonal dir )
    {
    soko_bool is_push;
    return can_move_diagonal( p, dir, &is_push, soko_false );
    }


//-----------------------------------------------------------------------------
// SokoPuzzle_move_diagonal
//-----------------------------------------------------------------------------
soko_bool SokoPuzzle_move_diagonal( SokoPuzzle p, SokoDiagonal dir )
    {
    soko_bool is_push;
    SOKO_ASSERT( !SokoPuzzle_playback_active(p) );
    return can_move_diagonal( p, dir, &is_push, soko_true );
    }


//-----------------------------------------------------------------------------
// shortest_player_path
//
//	Compute the shortest path by which the player can move to a target
//	location.
//
//	The algorithm operates by performing a flood-fill of all cells
//	accessible to the player.  Cells not accessible to the player are those
//	occupied by crates or walls, or those to which no clear path exists.
//	If the flood-fill touches the target cell, then the target is
//	considered reachable and the shortest path is determined by walking
//	backward from the destination to the starting location.
//
//	If the `path' argument is null, then the caller is only interested in
//	knowing whether or not a path exists to the target location, rather
//	than knowing the actual shortest path.
//
//	The following discussion outlines the overall flow of the algorithm.
//	The numbered sections annotate important areas of the code, discussing
//	the code's purpose and pointing out any special cases.
//
//	-1-
//	Clear the map prior to the flood-fill.  The value of the map at any
//	cell represents the distance of that cell from the starting location.
//	The value (INT_MAX - 1) is used to initialize empty cells to indicate
//	that they have not yet been visited; and INT_MAX is used to initialize
//	occupied cells, or those which are walls, to indicate that they can not
//	be visited.
//
//	-2-
//	Initialize the flood-fill stacks.  Two stacks are used for the
//	flood-fill.  The "current" stack contains the cells which are being
//	examined during this iteration.  The examination process may push cells
//	onto the "next" stack for examination during the next iteration.  Each
//	cell on a stack is at the same distance from the starting location as
//	every other cell on the same stack.
//
//	-3-
//	If there are still cells to visit, then perform the following steps.
//
//	-3.1-
//	Each item on the stack represents a reachable cell which has not yet
//	been visited, and which is at the same distance from the starting
//	location.  For each cell on the stack, perform the following steps.
//
//	-3.2-
//	Determine if the player, when at the given cell, can move to each of
//	the neighboring cells.  The player can move to a particular neighboring
//	cell if the cell is empty.  A cell is only considered if it has not
//	already been visited.
//
//	-3.3-
//	If the neighboring cell in question is also the final destination, then
//	a path exists to the target and the search is complete.
//
//	-3.4-
//	Otherwise, push the cell onto the "next" stack for later examination.
//
//	-4-
//	The flood-fill never touched the target cell, so signal failure.
//
//	-5-
//	If the destination cell is reachable and if the caller is interested in
//	the shortest path to that destination, then compute that path.  If
//	several shortest paths exist, then choose any single path.
//
//	-6-
//	Walk backward from the final destination to the starting player
//	location.  For each step back, choose any direction which decreases the
//	distance from the current location, thus resulting in the shortest
//	path.
//-----------------------------------------------------------------------------
static soko_bool shortest_player_path( SokoPuzzle p, SokoDirection* path,
    int* length, int src_row, int src_col, int dst_row, int dst_col )
    {
    soko_bool reachable = soko_true;
    SokoPuzzleStyle const style = STYLE(p);
    int const ndirs = DirMoveCount[ style ];
    int row, col, r, c, oddrow, oddcol, distance, dirs;
    int* map = PLAYER_MAP;
    Coord coord,temp_coord;
    Coord* p_curr_stack;
    Coord* p_next_stack;
    Coord* p_temp_stack;
    int curr_stack_top;
    int next_stack_top;
    int temp_stack_top;

    SOKO_ASSERT( PLAYER_PATH!=0 && PLAYER_STACK_1!=0 && PLAYER_STACK_2!=0 );
    SOKO_ASSERT( GOOD_ROW(src_row) && GOOD_COL(src_col) );
    SOKO_ASSERT( GOOD_ROW(dst_row) && GOOD_COL(dst_col) );
    SOKO_ASSERT( src_row != dst_row || src_col != dst_col );
    SOKO_ASSERT( IS_EMPTY(BOARD(dst_row, dst_col)) );

    for (row = 0;  row < p->data->row_count;  row++)		// -1-
	for (col = 0;  col < p->data->col_count;  col++)
	    MAP(row, col) = (IS_EMPTY(BOARD(row, col)) ? INT_MAX-1 : INT_MAX);

    p_next_stack = PLAYER_STACK_1;				// -2-
    next_stack_top = 0;
    p_curr_stack = PLAYER_STACK_2;
    curr_stack_top = 0;

    distance = 0;
    MAP(src_row, src_col) = distance; // "mark"
    coord.row = src_row;
    coord.col = src_col;
    p_next_stack[ next_stack_top++ ] = coord; // "push"
    while (next_stack_top > 0)					// -3-
	{
	distance++;
	p_temp_stack = p_next_stack; // switch stacks.
	p_next_stack = p_curr_stack;
	p_curr_stack = p_temp_stack;
	temp_stack_top = next_stack_top;
	next_stack_top = curr_stack_top;
	curr_stack_top = temp_stack_top;
	while (curr_stack_top > 0)				// -3.1-
	    {
	    temp_coord = p_curr_stack[ --curr_stack_top ]; // "pop"
	    oddrow = (temp_coord.row & 1);
	    oddcol = (temp_coord.col & 1);
	    for (dirs = 0; dirs < ndirs; dirs++)
		{
		SokoDirection const dir = DirMove[style][oddrow][oddcol][dirs];
		row = temp_coord.row + DirDeltaRow[style][oddrow][oddcol][dir];
		col = temp_coord.col + DirDeltaCol[style][oddrow][oddcol][dir];
		if (MAP(row, col) == INT_MAX-1)			// -3.2-
		    {
		    MAP(row, col) = distance; // "mark"
		    if (row == dst_row && col == dst_col)	// -3.3-
			goto FOUND_DST;
		    coord.row = row;				// -3.4-
		    coord.col = col;
		    p_next_stack[ next_stack_top++ ] = coord; // "push"
		    }
		}
	    }
	}

    reachable = soko_false;					// -4-

FOUND_DST:

    if (length != 0)
	*length = (reachable ? distance : -1);

    if (reachable && path != 0)					// -5-
	{
	row = dst_row;
	col = dst_col;
	while (distance > 0)					// -6-
	    {
	    distance--;
	    oddrow = (row & 1);
	    oddcol = (col & 1);
	    for (dirs = 0; dirs < ndirs; dirs++)
		{
		SokoDirection const dir = DirMove[style][oddrow][oddcol][dirs];
		r = row + DirDeltaRow[style][oddrow][oddcol][dir];
		c = col + DirDeltaCol[style][oddrow][oddcol][dir];
		if (MAP(r, c) == distance)
		    {
		    path[distance] = DirOpposite[dir];
		    row = r;
		    col = c;
		    break;
		    }
		}
	    }
	}

    return reachable;
    }


//-----------------------------------------------------------------------------
// player_path_exists
//	Check if a path exists from one cell to another which the player can
//	navigate.
//-----------------------------------------------------------------------------
static soko_bool player_path_exists(
    SokoPuzzle p, int src_row, int src_col, int dst_row, int dst_col )
    {
    return shortest_player_path( p, 0, 0, src_row, src_col, dst_row, dst_col );
    }


//-----------------------------------------------------------------------------
// shortest_crate_path
//
//	Compute the shortest path by which a crate can be pushed to a target
//	location.  The resulting path may overlap upon itself any number of
//	times, if necessary.  As a secondary goal, if multiple shortest paths
//	exist, choose the one requiring the fewest number of player moves.
//
//	This algorithm is considerably more complex than the algorithm for
//	computing the shortest path for moving the player to a destination
//	location.  There are a couple of reasons for this complexity.
//
//	First, it computes a running minimum cost of moving the player along
//	every path accessible to the crate.  A consequence of this requirement
//	is that, during the flood-fill phase, a particular cell may be visited
//	multiple times at the same distance from the starting point in order to
//	compute the player movement cost for pushing the crate onto that cell
//	from different directions.  This differs from the algorithm for finding
//	the shortest player movement path which only ever visits a cell once.
//
//	Second, the constraints for crate movement are more strict than those
//	for player movement.  To push a crate in some direction along some
//	path, the following two constraints must be met:
//
//	1. The player must be able to reach, from its last location, the side
//	   of the crate from which to push in the given direction.
//	2. The cell into which the crate will be pushed must be empty.
//
//	The algorithm operates by performing a flood-fill of all cells
//	accessible to the crate, given the above constraints.  In parallel with
//	the flood-fill, it also computes a running minimum cost for moving the
//	player into position to push the crate to all cells marked for
//	consideration by the flood-fill.
//
//	Unlike the algorithm for computing the shortest player path, this
//	algorithm may visit a cell multiple times; potentially entering once
//	across each edge.  This departure from the standard visit-once
//	flood-fill is necessary for the following reasons:
//
//	1. In order to satisfy the secondary goal of minimizing the number of
//	   player moves, it is necessary to examine all paths which arrive at a
//	   cell at the same distance from the starting location.  Only in this
//	   way can the path be chosen which requires the fewest player moves.
//	2. To arrive at the destination, crate paths must be allowed to
//	   overlap upon themselves any number of times.  For this to work,
//	   entry to the cell is allowed via edges which have not already been
//	   crossed by earlier visits.
//
//	If the flood-fill touches the target cell, then the target is
//	considered reachable and the shortest path is determined by walking
//	backward from the destination to the starting location while taking
//	into account the player movement costs along each step of the way.
//
//	If the `path' argument is null, then the caller is only interested in
//	knowing whether or not a path exists to the target location, rather
//	than knowing the actual shortest path.  In this case a minor
//	optimization is made during the flood-fill phase.  When the destination
//	location is reached, the search terminates abruptly with a positive
//	result.  In normal circumstances, the search does not terminate
//	abruptly since player movement cost must be computed for every approach
//	to the target location.
//
//	The following discussion outlines the overall flow of the algorithm.
//	The numbered sections annotate important areas of the code, discussing
//	the code's purpose and pointing out any special cases.
//
//	-1-
//	Seed the starting crate position with the cost of moving the player to
//	each side of the crate.  Each element of the `cost[]' array in the
//	flood-fill `map' records the number of moves required to reposition the
//	player from its current location to that side of the crate.  If the
//	player already occupies the cell adjacent to the crate, then the cost
//	is zero.  Otherwise, if the cell next to the crate is empty, then
//	compute the actual cost of moving the player to that cell.  If a
//	particular side of the crate is not reachable by the player, then the
//	cost is tagged with INT_MAX.
//
//	-2-
//	As a side-effect of computing the initial player movement costs, if no
//	side of the crate is reachable by the player, then the crate can not be
//	moved at all, thus all further processing can be skipped.
//
//	-3-
//	Remove player and crate from board to prevent interference during
//	subsequent shortest-path calculations.  The player needs to be removed
//	because the player's current location might end up being a valid
//	position along the crate's path.  The crate needs to be removed because
//	the player may need to move through that position in order to reach or
//	push the crate for subsequent steps along the path.  Furthermore, the
//	crate's path may overlap upon itself, therefore, that position must be
//	available.  Later on, the crate is temporarily added back to the board
//	at each location along every path in order to ensure validity of the
//	paths computed for the player when attempting to move the player
//	adjacent to the crate for a particular push.
//
//	-4-
//	Clear the map prior to the flood-fill.  The player movement cost for
//	each edge of each cell in the map is initialized to INT_MAX for all
//	empty cells to indicate that they have not yet been visited from that
//	particular edge; and to (INT_MAX - 1) for all occupied cells, or those
//	which are walls, to indicate that they can not be visited.  During the
//	reverse map walking phase, a minimum-cost direction of SOKO_DIR_ILLEGAL
//	indicates that a given map location was unreachable from the
//	corresponding direction, in which case, the actual cost will also be
//	tagged with INT_MAX.  Special care is taken to avoid clobbering the
//	seed values which were previously computed for the starting crate
//	location.
//
//	-5-
//	Initialize the flood-fill stacks.  Two stacks are used for the
//	flood-fill.  The "current" stack contains the cells which are being
//	examined during this iteration.  The examination process may push cells
//	onto the "next" stack for examination during the next iteration.  Each
//	cell on a stack is at the same distance from the starting location as
//	every other cell on the same stack.
//
//	-6-
//	If the destination has not been reached and there are still cells to
//	visit, then perform the following steps.
//
//	-6.1-
//	Each item on the stack represents a reachable cell which has not yet
//	been visited from at least one direction, and which is at the same
//	distance from the starting location.  For each cell on the stack,
//	perform the following steps.
//
//	-6.2-
//	Determine if a crate on the current cell can be pushed to each of the
//	neighboring cells.  A crate can be pushed to a neighboring cell if the
//	neighboring cell is empty and if the neighboring cell on the opposite
//	side of the crate is reachable by the player from one of the player's
//	possible positions at this point in the flood-fill.
//
//	-6.3-
//	First check if the neighboring cell into which the crate will be
//	pushed, and the neighboring cell from which the player will push are
//	empty.  If so, then that cell is a candidate as a push destination.
//
//	-6.4-
//	The neighbors are empty, so check if an attempt has already been made
//	to push a crate into the destination cell in this direction.  If not
//	(cost is INT_MAX), then see if the player can reach this cell from one
//	of the player's possible positions and, if reachable, compute the
//	minimum cost of reaching this cell.
//
//	-6.5-
//	Compute the cost of moving the player into position to push the crate
//	in the desired direction.  If this computation succeeds, that means
//	that the player is able to reach the crate's pushing side by at least
//	one path, thus, as a side-effect, this computation also indicates if
//	the crate can even be pushed in the desired direction.
//
//	The overall computed cost at any given map location represents the
//	minimum number of extra moves made by the player to push the crate to
//	that map location.  The computed cost does not include the move which
//	actually pushes the crate; rather, only the number of moves to position
//	the player so that it can push the crate.  Since a crate may
//	potentially be pushed onto a cell from any of its neighbors, the map's
//	`cost[]' array records the overall minimum cost in player moves
//	required to push the crate onto the cell from each direction.  For
//	example, with a square-tiled puzzle, the SOKO_DIR_UP cost at map row 3,
//	column 5 is the minimum number of extra moves by the player to push the
//	crate up from row 4, column 5 to row 3, column 5.
//
//	Computing the minimum cost of repositioning the player so that it can
//	push the crate in the desired direction requires examining the
//	potential directions from which the crate was pushed into its current
//	location, as seen below.
//
//	-6.6-
//	See if the player can reach the pushing cell from any of the positions
//	which it may already be occupying after having pushed the crate this
//	far, and compute the cost of doing so.  For example, for a square
//	puzzle, if the crate was just pushed right and is about to be pushed
//	up, see if the player can be moved from the crate's left side to the
//	bottom side, and compute the cost of that movement.  The overall cost
//	for positioning the player at this point is the sum of the base cost
//	plus the additional cost to reposition the player for this step.  The
//	base cost is the cost of player movement up to, but not including this
//	step.  If the player is already in position to push the crate in the
//	desired direction, then there is no additional cost.  For example, if
//	the crate was just pushed up and is about to be pushed up again, then
//	the player does not need to be repositioned prior to the second push.
//	In this case, although the additional cost is zero there may still be
//	other paths which have lower overall costs, thus all approaches must be
//	checked.  Note that the various costs for the starting crate position
//	were seeded in an earlier step, so this computation is accurate even if
//	the player is not adjacent to the crate at the very first push.
//
//	-6.7-
//	If least-cost computation succeeded, then the player was able to reach
//	the pushing side of the crate from at least one direction.  Therefore,
//	record the minimum cost of pushing the crate into the destination cell
//	in the desired direction.
//
//	-6.8-
//	If the search has reached the final destination, then signal that the
//	target is reachable.  Otherwise, push the cell onto the "next" stack
//	for later examination.  As an optimization, if the caller is only
//	interested in whether or not any path exists to the target, rather than
//	knowing the exact path, then abruptly terminate the search with a
//	positive result if the target cell has been reached.  Under normal
//	circumstances, abrupt termination is unwarranted because the least-cost
//	computation must be performed on the final destination cell from all
//	approaches, just as it is computed for all other accessible cells.
//
//	-7-
//	If the destination cell is reachable and if the caller is interested in
//	the shortest, least-cost path to that destination, then compute that
//	path.  If several shortest least-cost paths exist, then choose any
//	single path.
//
//	-8-
//	Walk backward from the final destination to the starting crate
//	location.  For each step back, choose the direction which required the
//	least overall cost.
//
//	-9-
//	Place the player and crate back on the board.  They were removed in an
//	earlier step to prevent interference during path-finding.
//-----------------------------------------------------------------------------
static soko_bool shortest_crate_path( SokoPuzzle p, SokoDirection* path,
    int* length, int src_row, int src_col, int dst_row, int dst_col )
    {
    soko_bool reachable;
    SokoPuzzleStyle const style = STYLE(p);
    int const ndirs = DirPushCount[ style ];
    SokoDirection dir, odir;
    int row, col, oddrow, oddcol, distance, dirs;
    CrateMapRec* map = CRATE_MAP;
    CrateMapRec* q;
    Coord coord;
    Coord* p_curr_stack;
    Coord* p_next_stack;
    Coord* p_temp_stack;
    int curr_stack_top;
    int next_stack_top;
    int temp_stack_top;
    char player_ch,crate_ch;

    SOKO_ASSERT( CRATE_PATH != 0 && CRATE_STACK_1 != 0 && CRATE_STACK_2 != 0 );
    SOKO_ASSERT( GOOD_ROW(src_row) && GOOD_COL(src_col) );
    SOKO_ASSERT( GOOD_ROW(dst_row) && GOOD_COL(dst_col) );
    SOKO_ASSERT( src_row != dst_row || src_col != dst_col );
    SOKO_ASSERT( IS_CRATE(BOARD(src_row, src_col)) );
    SOKO_ASSERT( IS_EMPTY (BOARD(dst_row, dst_col)) ||
		 IS_PLAYER(BOARD(dst_row, dst_col)) );

    reachable = soko_false;
    oddrow = (src_row & 1);
    oddcol = (src_col & 1);
    q = &MAP(src_row, src_col);
    for (dirs = 0; dirs < ndirs; dirs++)			// -1-
	{
	int cost = 0;
	dir = DirPush[ style ][ dirs ];
	odir = DirOpposite[ dir ];
	q->cost[ dir ].n = INT_MAX;
	q->cost[ dir ].d = SOKO_DIR_ILLEGAL;
	q->cost[ dir ].l = INT_MAX;
	row = src_row + DirDeltaRow[ style ][ oddrow ][ oddcol ][ odir ];
	col = src_col + DirDeltaCol[ style ][ oddrow ][ oddcol ][ odir ];
	if ((row == p->data->player_row && col == p->data->player_col) ||
	    (IS_EMPTY(BOARD(row, col)) &&
	    shortest_player_path( p, 0, &cost,
	    p->data->player_row, p->data->player_col, row, col )))
	    {
	    q->cost[ dir ].n = cost; // "mark"
	    q->cost[ dir ].l = 0;
	    reachable = soko_true;
	    }
	}

    if (!reachable)						// -2-
	{
	if (length != 0)
	    *length = -1;
	return soko_false;				// *** RETURN ***
	}

    player_ch = BOARD(p->data->player_row, p->data->player_col);
    BOARD(p->data->player_row, p->data->player_col) = EMPTY_CH;	// -3-
    crate_ch = BOARD(src_row, src_col);
    BOARD(src_row, src_col) = EMPTY_CH;

    for (row = 0;  row < p->data->row_count;  row++)		// -4-
	for (col = 0;  col < p->data->col_count;  col++)
	    {
	    if (row != src_row || col != src_col) // Source cell already seeded
		for (dirs = 0; dirs < ndirs; dirs++)
		    {
		    CrateMapRec* r = &MAP(row, col);
		    dir = DirPush[style][dirs];
		    r->cost[ dir ].n =
			(IS_EMPTY(BOARD(row, col)) ? INT_MAX : INT_MAX-1);
		    r->cost[ dir ].d = SOKO_DIR_ILLEGAL;
		    r->cost[ dir ].l = INT_MAX;
		    }
	    }

    p_next_stack = CRATE_STACK_1;				// -5-
    next_stack_top = 0;
    p_curr_stack = CRATE_STACK_2;
    curr_stack_top = 0;

    reachable = soko_false;
    distance = 0;
    coord.row = src_row;
    coord.col = src_col;
    p_next_stack[ next_stack_top++ ] = coord;  // "push"
    while (!reachable && next_stack_top > 0)			// -6-
	{
	distance++;
	p_temp_stack = p_next_stack; // switch stacks.
	p_next_stack = p_curr_stack;
	p_curr_stack = p_temp_stack;
	temp_stack_top = next_stack_top;
	next_stack_top = curr_stack_top;
	curr_stack_top = temp_stack_top;
	while (curr_stack_top > 0)				// -6.1-
	    {
	    Coord temp_coord = p_curr_stack[--curr_stack_top]; // "pop"
	    row = temp_coord.row;
	    col = temp_coord.col;
	    for (dirs = 0; dirs < ndirs; dirs++)
		{						// -6.2-
		int dr, dc, odr, odc, r1, c1, r2, c2;
		oddrow = (row & 1);
		oddcol = (col & 1);
		dir = DirPush[ style ][ dirs ];
		odir = DirOpposite[ dir ];
		dr  = DirDeltaRow[ style ][ oddrow ][ oddcol ][ dir  ];
		dc  = DirDeltaCol[ style ][ oddrow ][ oddcol ][ dir  ];
		odr = DirDeltaRow[ style ][ oddrow ][ oddcol ][ odir ];
		odc = DirDeltaCol[ style ][ oddrow ][ oddcol ][ odir ];
		r1 = row + odr; // Cell from which player
		c1 = col + odc; //   will push crate.
		r2 = row + dr;  // Cell into which crate
		c2 = col + dc;  //   will be pushed.
		if (GOOD_ROW(r1) && GOOD_COL(c1) &&		// -6.3-
		    GOOD_ROW(r2) && GOOD_COL(c2) &&
		    IS_EMPTY(BOARD(r1, c1)) && IS_EMPTY(BOARD(r2, c2)))
		    {
		    CrateMapRec* md = &MAP(r2, c2); // Destination map record.
		    if (md->cost[dir].n == INT_MAX)		// -6.4-
			{
			int i;					// -6.5-
			int least_cost = INT_MAX;
			SokoDirection least_cost_dir = SOKO_DIR_ILLEGAL;
			CrateMapRec* ms = &MAP(row, col); // Source map record.
			for (i = 0; i < ndirs; i++)		// -6.6-
			    {
			    SokoDirection const d = DirPush[style][i];
			    int base_cost = ms->cost[d].n;
			    if (base_cost != INT_MAX &&
				ms->cost[d].l == distance - 1)
				{
				SokoDirection o = DirOpposite[d];
				soko_bool ok;
				int cost = 0;
				int const r = row +
				    DirDeltaRow[style][oddrow][oddcol][o];
				int const c = col +
				    DirDeltaCol[style][oddrow][oddcol][o];
				if (r == r1 && c == c1)
				    ok = soko_true;
				else
				    {
				    char save_ch = BOARD(row, col);
				    BOARD(row,col) = CRATE_CH;
				    ok = shortest_player_path(
					p, 0, &cost, r, c, r1, c1);
				    BOARD(row,col) = save_ch;
				    }
				if (ok && (base_cost + cost < least_cost ||
				   (d == dir &&
				    base_cost + cost <= least_cost)))
				    {
				    least_cost = base_cost + cost;
				    least_cost_dir = d;
				    }
				}
			    }

			if (least_cost != INT_MAX)		// -6.7-
			    {
			    SOKO_ASSERT( least_cost_dir != SOKO_DIR_ILLEGAL );
			    md->cost[dir].n = least_cost; // "mark"
			    md->cost[dir].d = least_cost_dir;
			    md->cost[dir].l = distance;
			    if (r2 == dst_row && c2 == dst_col)	// -6.8-
				{
				reachable = soko_true;
				if (path == 0)
				    goto FOUND_DST;
				}
			    else
				{
				coord.row = r2;
				coord.col = c2; // "push"
				p_next_stack[ next_stack_top++ ] = coord;
				}
			    }
			}
		    }
		}
	    }
	}

FOUND_DST:

    if (length != 0)
	*length = (reachable ? distance : -1);

    if (reachable && path != 0)					// -7-
	{
	int cost, least_cost = INT_MAX;
	q = &MAP(dst_row, dst_col);
	dir = SOKO_DIR_ILLEGAL;
	for (dirs = 0; dirs < ndirs; dirs++)
	    {
	    SokoDirection const d = DirPush[ style ][ dirs ];
	    if (q->cost[d].l == distance)
		{
		cost = q->cost[d].n;
		if (cost < least_cost)
		    {
		    least_cost = cost;
		    dir = d;
		    }
		}
	    }
	row = dst_row;
	col = dst_col;
	while (distance > 0)					// -8-
	    {
	    q = &MAP(row,col);
	    SOKO_ASSERT( dir != SOKO_DIR_ILLEGAL );
	    SOKO_ASSERT( q->cost[ dir ].l == distance );
	    distance--;
	    path[ distance ] = dir;
	    odir = DirOpposite[ dir ];
	    dir = q->cost[ dir ].d;
	    oddrow = (row & 1);
	    oddcol = (col & 1);
	    row += DirDeltaRow[ style ][ oddrow ][ oddcol ][ odir ];
	    col += DirDeltaCol[ style ][ oddrow ][ oddcol ][ odir ];
	    }
	}

    BOARD(p->data->player_row, p->data->player_col) = player_ch;
    BOARD(src_row, src_col) = crate_ch;				// -9-

    return reachable;
    }


//-----------------------------------------------------------------------------
// move_player
//	Try repositioning the player to a new cell while minimizing the number
//	of moves.  If a path exists from the player's current location to the
//	new location, move the player along that path.
//-----------------------------------------------------------------------------
static soko_bool move_player( SokoPuzzle p, int dst_row, int dst_col )
    {
    int distance;
    soko_bool reachable = shortest_player_path( p, PLAYER_PATH, &distance,
	p->data->player_row, p->data->player_col, dst_row, dst_col );
    if (reachable)
	{
	int i;
	begin_animation_sequence(p);
	for (i = 0; i < distance; i++)
	    {
	    SokoDirection const dir = PLAYER_PATH[i];
	    SOKO_ASSERT( SokoPuzzle_can_move(p, dir) );
	    SokoPuzzle_move( p, dir );
	    }
	end_animation_sequence(p);
	}
    return reachable;
    }


//-----------------------------------------------------------------------------
// move_crate
//
//	Try moving a crate from one cell to another while minimizing the number
//	of pushes.  If a path exists from the crate's current location to the
//	new location, then for each push required to move the crate to the
//	destination, perform these steps:
//
//	* Move the player to a cell adjacent to the crate such that it is in
//	  position to push the crate in the desired direction.  If the player
//	  is already at the appropriate cell, then skip this step.
//
//	* Move the player in the desired direction.  This pushes the crate as a
//	  side-effect.
//
//	If crate selection mode is enabled, it is automatically cancelled as a
//	side-effect of moving the player.
//-----------------------------------------------------------------------------
static soko_bool move_crate(
    SokoPuzzle p, int src_row, int src_col, int dst_row, int dst_col )
    {
    int distance;
    soko_bool reachable = shortest_crate_path( p, CRATE_PATH, &distance,
	src_row, src_col, dst_row, dst_col );
    if (reachable)
	{
	int i;
	int crate_row = src_row;
	int crate_col = src_col;
	SokoPuzzleStyle const style = STYLE(p);
	begin_animation_sequence(p);
	for (i = 0; i < distance; i++)
	    {
	    SokoDirection const dir = CRATE_PATH[i];
	    SokoDirection const odir = DirOpposite[ dir ];
	    int oddrow = (crate_row & 1);
	    int oddcol = (crate_col & 1);
	    int const r = crate_row + DirDeltaRow[style][oddrow][oddcol][odir];
	    int const c = crate_col + DirDeltaCol[style][oddrow][oddcol][odir];
	    if (r != p->data->player_row || c != p->data->player_col)
		move_player( p, r, c );
	    SokoPuzzle_move( p, dir );
	    crate_row += DirDeltaRow[style][oddrow][oddcol][dir];
	    crate_col += DirDeltaCol[style][oddrow][oddcol][dir];
	    }
	end_animation_sequence(p);
	}
    return reachable;
    }


//-----------------------------------------------------------------------------
// can_reach_and_push_crate_in_direction
//	Check if a crate is a potential candidate for pushing in a particular
//	direction.  To be a candidate, the appropriate edge of the crate must
//	be reachable by the player and an empty cell must be present on the
//	opposite side of the crate.
//-----------------------------------------------------------------------------
static soko_bool can_reach_and_push_crate_in_direction(
    SokoPuzzle p, int row, int col, SokoDirection dir )
    {
    SokoPuzzleStyle const style = STYLE(p);
    SokoDirection const odir = DirOpposite[ dir ];
    int const oddrow = (row & 1);
    int const oddcol = (col & 1);
    int const rdst = row + DirDeltaRow[ style ][ oddrow ][ oddcol ][ dir  ];
    int const cdst = col + DirDeltaCol[ style ][ oddrow ][ oddcol ][ dir  ];
    int const rsrc = row + DirDeltaRow[ style ][ oddrow ][ oddcol ][ odir ];
    int const csrc = col + DirDeltaCol[ style ][ oddrow ][ oddcol ][ odir ];
    char const chdst = BOARD(rdst,cdst);
    char const chsrc = BOARD(rsrc,csrc);
    return (IS_EMPTY(chsrc) || IS_PLAYER(chsrc)) && // Source accessible.
	   (IS_EMPTY(chdst) || IS_PLAYER(chdst)) && // Destination accessible.
	   ((rsrc == p->data->player_row && csrc == p->data->player_col) ||
	     player_path_exists(
	         p, p->data->player_row, p->data->player_col, rsrc, csrc ));
    }


//-----------------------------------------------------------------------------
// can_reach_and_push_crate
//	Check if a crate is a potential candidate for pushing.  To be a
//	candidate, the crate must be reachable by the player and must be
//	capable of being pushed by one of the edges reachable by the player.
//-----------------------------------------------------------------------------
static soko_bool can_reach_and_push_crate( SokoPuzzle p, int r, int c )
    {
    SokoPuzzleStyle const style = STYLE(p);
    int d, ndirs = DirPushCount[ style ];
    for (d = 0; d < ndirs; d++)
	if (can_reach_and_push_crate_in_direction(p, r, c, DirPush[style][d]))
	    return soko_true;
    return soko_false;
    }


//-----------------------------------------------------------------------------
// crate_dragged
//
//	User attempted to drag a crate to a new cell.  Respond as follows:
//
//	* If the destination cell is empty or is occupied by the player, try
//	  moving the crate to that cell via the shortest path.  If the crate
//	  moves, crate selection mode is automatically cancelled.
//-----------------------------------------------------------------------------
static soko_bool crate_dragged( SokoPuzzle p, int r, int c )
    {
    soko_bool ok = soko_false;
    int const src_row = p->data->drag_state.row;
    int const src_col = p->data->drag_state.col;
    char const ch = BOARD(r, c);
    if (IS_EMPTY(ch) || IS_PLAYER(ch))
	ok = move_crate( p, src_row, src_col, r, c );
    return ok;
    }


//-----------------------------------------------------------------------------
// crate_clicked
//
//	A crate was clicked.  Respond as follows:
//
//	* Provide the user with the opportunity to drag the crate around.
//
//	If the mouse was clicked and released on the crate but not dragged to
//	another cell (that is, the player elected to not drag the crate),
//	respond as follows:
//
//	* If the clicked crate was selected, or if some other crate was
//	  selected, then cancel the selection.
//
//	* If the clicked crate was not originally selected, and if it is
//	  accessible then select it.  To be accessible, the crate must be
//	  reachable by the player and must be capable of being pushed by one of
//	  the edges reachable by the player (that is, an empty cell must be
//	  present on the side opposite the player).
//-----------------------------------------------------------------------------
static void crate_clicked( SokoPuzzle p, int row, int col )
    {
    if (!await_drag( p, row, col, crate_dragged ))
	{
	if (IS_CRATE_SELECTED &&
	    row == p->data->selected_row && col == p->data->selected_col)
	    deselect_crate(p);
	else
	    {
	    if (IS_CRATE_SELECTED)
		deselect_crate(p);		// First deselect old cell.
	    if (can_reach_and_push_crate( p, row, col ))
		select_crate( p, row, col );	// Then select new cell.
	    }
	}
    }


//-----------------------------------------------------------------------------
// player_dragged
//
//	User attempted to drag the player to a new cell.  Respond as follows:
//
//	* If the destination cell is empty, try moving the player to that cell
//	  via the shortest path.
//
//	* If the destination cell is occupied by a crate and if the player is
//	  directly adjacent to the crate then try pushing the crate.
//
//	* In any event, even if the mouse was dragged to a useless or
//	  unreachable cell, cancel crate selection mode.  (Crate selection mode
//	  is also cancelled automatically for the above two cases.)
//-----------------------------------------------------------------------------
static soko_bool player_dragged( SokoPuzzle p, int r, int c )
    {
    soko_bool ok = soko_false;
    char const ch = BOARD(r, c);
    if (IS_EMPTY(ch))
	ok = move_player( p, r, c );
    else if (IS_CRATE(ch))
	{
	SokoPuzzleStyle const style = STYLE(p);
	int d, ndirs = DirMoveCount[ style ];
	for (d = 0; d < ndirs; d++)
	    { // Is player directly adjacent to crate?
	    int const oddrow = (p->data->player_row & 1);
	    int const oddcol = (p->data->player_col & 1);
	    SokoDirection const dir = DirMove[style][oddrow][oddcol][d];
	    if (r == p->data->player_row +
		DirDeltaRow[style][oddrow][oddcol][dir] &&
		c == p->data->player_col +
		DirDeltaCol[style][oddrow][oddcol][dir])
		{
		ok = SokoPuzzle_move( p, dir );
		break;
		}
	    }
	}
    else if (IS_CRATE_SELECTED)
	deselect_crate(p);
    return ok;
    }


//-----------------------------------------------------------------------------
// player_clicked
//
//	The player was clicked.  Respond as follows:
//
//	* Provide the user with the opportunity to drag the player around.
//
//	If the mouse was clicked and released on the player cell but not
//	dragged to another cell (that is, the user elected to not drag the
//	player) and if a crate is selected, then respond as follows:
//
//	* Try moving the crate to the cell currently occupied by the player.
//	  This allows crate movement via point-and-click to work even when the
//	  destination cell is currently occupied by the player.  If movement
//	  occurs, then crate selection mode is automatically cancelled.
//
//	* Otherwise cancel crate selection mode in order to indicate that the
//	  requested move was invalid.
//-----------------------------------------------------------------------------
static void player_clicked( SokoPuzzle p )
    {
    int const r = p->data->player_row;
    int const c = p->data->player_col;
    if (!await_drag( p, r, c, player_dragged ) && IS_CRATE_SELECTED &&
	!move_crate( p, p->data->selected_row, p->data->selected_col, r, c ))
	deselect_crate(p);
    }


//-----------------------------------------------------------------------------
// empty_clicked
//
//	An empty cell was clicked.  Respond as follows:
//
//	* If no crate is selected, then try moving the player to the clicked
//	  cell via the shortest path.  If moving the player to the clicked cell
//	  was successful, then allow the mouse to subsequently drag player from
//	  that point.
//
//	* If a crate was selected, try moving the crate to the clicked cell via
//	  the shortest path.  If moving the crate to the clicked cell was
//	  successful, then allow the mouse to subsequently drag the crate from
//	  that point.  Crate selection mode is automatically cancelled if crate
//	  moves.
//
//	* If crate did not move, then manually cancel crate selection mode.
//-----------------------------------------------------------------------------
static void empty_clicked( SokoPuzzle p, int row, int col )
    {
    if (!IS_CRATE_SELECTED)
	{
	if (move_player( p, row, col ))
	    player_clicked(p);
	}
    else if (move_crate( p, p->data->selected_row, p->data->selected_col,
	     row, col ))
	await_drag( p, row, col, crate_dragged );
    else
	deselect_crate(p);
    }


//-----------------------------------------------------------------------------
// push_crate
//	If the player is in line with a moveable crate, try pushing the crate
//	to destination cell.  If no crate exists between player and destination
//	cell, then move the player in a straight line to the destination,
//	instead.
//-----------------------------------------------------------------------------
static void push_crate( SokoPuzzle p, int row, int col )
    {
    SokoPuzzleStyle const style = STYLE(p);
    int dirs, ndirs = DirPushCount[ style ];
    soko_bool crate_seen = soko_false;
    int num_moves = 0;
    for (dirs = 0; dirs < ndirs; dirs++)
	{
	int r = p->data->player_row;
	int c = p->data->player_col;
	SokoDirection const dir = DirPush[ style ][ dirs ];
	crate_seen = soko_false;
	for (num_moves = 0; ; num_moves++)
	    {
	    char ch;
	    int const oddrow = (r & 1);
	    int const oddcol = (c & 1);
	    r += DirDeltaRow[ style ][ oddrow ][ oddcol ][ dir ];
	    c += DirDeltaCol[ style ][ oddrow ][ oddcol ][ dir ];
	    if (!GOOD_ROW(r) || !GOOD_COL(c))
		break;			// *** BREAK ***  Beyond puzzle bounds.
	    if (r == row && c == col)
		goto FOUND_DIR;		// *** GOTO ***   Found straight line.
	    ch = BOARD(r, c);
	    if (IS_CRATE(ch))
		{
		if (crate_seen)
		    break;		// *** BREAK ***  Too many crates.
		crate_seen = soko_true;
		}
	    else if (!IS_EMPTY(ch))
		break;			// *** BREAK ***  Ran into wall.
	    }
	}

FOUND_DIR:

    if (dirs < ndirs)	// Found good path.
	{
	SokoDirection const dir = DirPush[ style ][ dirs ];

	if (!crate_seen)	// If player was not actually pushing a crate,
	    num_moves++;	// then move player to goal cell, instead.

	SOKO_ASSERT(num_moves != 0);

	begin_animation_sequence(p);
	while (num_moves-- > 0)
	    SokoPuzzle_move( p, dir );
	end_animation_sequence(p);
	}
    }


//-----------------------------------------------------------------------------
// secondary_mouse_action
//
//	Secondary mouse button was pressed or released.  Respond as follows:
//
//	* In all cases, for a mouse down, cancel crate selection mode.
//
//	* If the mouse is pressed and the target cell is empty, try pushing a
//	  single crate in a straight line with the player.
//-----------------------------------------------------------------------------
static void secondary_mouse_action(
    SokoPuzzle p, soko_bool down, int row, int col )
    {
    SOKO_ASSERT( !SokoPuzzle_drag_active(p) );
    SOKO_ASSERT( !SokoPuzzle_playback_active(p) );
    if (down)
	{
	deselect_crate(p);
	if (IS_EMPTY(BOARD(row, col)))
	    push_crate( p, row, col );
	}
    }


//-----------------------------------------------------------------------------
// SokoPuzzle_mouse_action
//
//	A mouse button was pressed or released.  Respond as follows:
//
//	* If a history playback session is active, then cancel the playback
//	  session.  No additional processing is performed for the event which
//	  cancels playback.
//
//	If the mouse button was released, then respond as follows:
//
//	* If a drag session is active, end the session.
//
//	If the mouse button was pressed, then respond as follows:
//
//	* If the primary mouse button was pressed, and if a modifier key, such
//	  as Shift, Control, Alternate, or Command, was held down, then pretend
//	  that the secondary button was pressed, instead.  This allows users of
//	  one-button mice to perform the push-crate-in-straight-line function
//	  which is normally tied to the secondary mouse button.
//
//	* Otherwise, respond according to the content of the clicked cell if
//	  the cell is empty, occupied by the player, or occupied by a crate.
//
//	* Otherwise, if the clicked cell can not respond, then cancel crate
//	  selection mode if enabled.
//-----------------------------------------------------------------------------
void SokoPuzzle_mouse_action( SokoPuzzle p, int button, soko_bool down,
    SokoEventFlags flags, int row, int col )
    {
    if (SokoPuzzle_playback_active(p))
	SokoPuzzle_abort_playback(p);
    else
	{
	SokoEventFlags const SIMULATE_SECONDARY =
	    SOKO_FLAG_SHIFT     |
	    SOKO_FLAG_CONTROL   |
	    SOKO_FLAG_ALTERNATE |
	    SOKO_FLAG_COMMAND;
	if (SokoPuzzle_drag_active(p))
	    {
	    if (!down && button == 0)
		abort_drag(p);
	    }
	else if (button==1 || (button==0 && (flags & SIMULATE_SECONDARY) != 0))
	    {
	    secondary_mouse_action( p, down, row, col );
	    }
	else if (down && button == 0)
	    {
	    char const ch = BOARD(row, col);
	    if (IS_EMPTY(ch))
		empty_clicked( p, row, col );
	    else if (IS_PLAYER(ch))
		player_clicked(p);
	    else if (IS_CRATE(ch))
		crate_clicked( p, row, col );
	    else
		deselect_crate(p);
	    }
	}
    }


//-----------------------------------------------------------------------------
// SokoPuzzle_mouse_drag
//
//	The mouse was dragged with a mouse button depressed.  Respond as
//	follows:
//
//	* If a drag session is active, update the drag state, thus allowing
//	  the player or crate to follow the mouse.
//-----------------------------------------------------------------------------
void SokoPuzzle_mouse_drag( SokoPuzzle p, int row, int col )
    {
    if (SokoPuzzle_drag_active(p))
	update_drag( p, row, col );
    }


//-----------------------------------------------------------------------------
// SokoPuzzle_key_action
//
//	A key was pressed or released.  Respond as follows:
//
//	* If a history playback session is active and if the key is Escape,
//	  then cancel playback.  If the key is not Escape, then ignore it since
//	  user-initiated movement commands are not allowed during playback.
//
//	* If a drag session is active, then ignore the key since keyboard-based
//	  movement is contraindicated while the player or a crate is being
//	  dragged.
//
//	* Otherwise, if a directional key was pressed, then attempt to move the
//	  player in the corresponding direction.  Crate selection mode is also
//	  cancelled as a side-effect.
//
//	* Otherwise, if the key is Escape and a crate is selected, then
//	  deselect the crate.
//-----------------------------------------------------------------------------
void SokoPuzzle_key_action(
    SokoPuzzle p, SokoKeyCode code, soko_bool down, SokoEventFlags flags )
    {
    (void)flags;
    if (SokoPuzzle_playback_active(p))
	{
	if (!down && code == SOKO_KEY_ESCAPE)
	    SokoPuzzle_abort_playback(p);
	}
    else if (!SokoPuzzle_drag_active(p))
	{
	if (down)
	    {
	    switch (code)
		{
		case SOKO_KEY_UP:
		    SokoPuzzle_move( p, SOKO_DIR_UP );
		    break;
		case SOKO_KEY_DOWN:
		    SokoPuzzle_move( p, SOKO_DIR_DOWN );
		    break;
		case SOKO_KEY_LEFT:
		    SokoPuzzle_move( p, SOKO_DIR_LEFT );
		    break;
		case SOKO_KEY_RIGHT:
		    SokoPuzzle_move( p, SOKO_DIR_RIGHT );
		    break;
		case SOKO_KEY_UP_LEFT:
		    SokoPuzzle_move_diagonal( p, SOKO_DIAG_UP_LEFT );
		    break;
		case SOKO_KEY_UP_RIGHT:
		    SokoPuzzle_move_diagonal( p, SOKO_DIAG_UP_RIGHT );
		    break;
		case SOKO_KEY_DOWN_LEFT:
		    SokoPuzzle_move_diagonal( p, SOKO_DIAG_DOWN_LEFT );
		    break;
		case SOKO_KEY_DOWN_RIGHT:
		    SokoPuzzle_move_diagonal( p, SOKO_DIAG_DOWN_RIGHT );
		    break;
		default:
		    break;
		}
	    }
	else if (code == SOKO_KEY_ESCAPE) // !down && escape
	    deselect_crate(p);
	}
    }
