//-----------------------------------------------------------------------------
// SokoPool.c
//
//	Memory pool object.
//
// Copyright (c), 2001, Eric Sunshine <sunshine@sunshineco.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoPool.c,v 1.1 2001/12/21 21:57:43 sunshine Exp $
// $Log: SokoPool.c,v $
// Revision 1.1  2001/12/21 21:57:43  sunshine
// v15
// -*- Extracted core game logic out of GUI code and generalized it so that
//     the same core code can be used by any platform.  Logic from
//     SokoBoard.m now resides in SokoPuzzle.c, etc.
//
// -*- Added SokoPool which is a memory pool object.  This is used heavily by
//     the string manipulation functions in SokoUtil to ensure that composed
//     pathnames and strings do not go out of scope before the client is
//     finished with them.  This replaces the old statically allocated,
//     circular array of pathname buffers which was used in the past.
//
//-----------------------------------------------------------------------------
#include "SokoPool.h"
#include "SokoAssert.h"
#include <stdlib.h>
#include <string.h>

#define SOKO_POOL_SLOT_SIZE (64)
#define SOKO_POOL_PAGE_SIZE (1024)

struct SokoPoolData
    {
    char** slots;
    unsigned long slots_count;
    unsigned long slots_max;
    unsigned long pool_next;
    unsigned long pool_max;
    char const* null_string; // Freed along with pool.
    };

//-----------------------------------------------------------------------------
// pool_allocate
//-----------------------------------------------------------------------------
static char* pool_allocate( SokoPool p, unsigned long needed )
    {
    char* mem;
    if (needed + p->data->pool_next >= p->data->pool_max)
	{
	unsigned long nbytes;
	if (p->data->slots_count >= p->data->slots_max)
	    {
	    p->data->slots_max += SOKO_POOL_SLOT_SIZE;
	    nbytes = p->data->slots_max * sizeof(p->data->slots[0]);
	    if (p->data->slots == 0)
		p->data->slots = (char**)malloc( nbytes );
	    else
		p->data->slots = (char**)realloc( p->data->slots, nbytes );
	    }
	p->data->slots_count++;

	nbytes = 0;
	do { nbytes += SOKO_POOL_PAGE_SIZE; } while (nbytes < needed);
	p->data->slots[ p->data->slots_count - 1 ] = (char*)malloc( nbytes );
	p->data->pool_next = 0;
	p->data->pool_max = nbytes;
	}
    mem = p->data->slots[ p->data->slots_count - 1 ] + p->data->pool_next;
    p->data->pool_next += needed;
    return mem;
    }


//-----------------------------------------------------------------------------
// SokoPool_allocate
//-----------------------------------------------------------------------------
char* SokoPool_allocate( SokoPool p, unsigned long nbytes )
    {
    SOKO_ASSERT( nbytes != 0 );
    return pool_allocate( p, nbytes );
    }


//-----------------------------------------------------------------------------
// SokoPool_store
//-----------------------------------------------------------------------------
char const* SokoPool_store( SokoPool p, char const* s )
    {
    char const* interned;
    soko_bool empty = (s == 0 || *s == '\0');
    if (empty) s = "";
    if (empty && p->data->null_string != 0)
	interned = p->data->null_string;
    else
	{
	unsigned long const len = strlen(s) + 1; // Add one for terminator.
	char* tmp = pool_allocate( p, len );
	memcpy( tmp, s, len );
	if (empty && p->data->null_string == 0)
	    p->data->null_string = tmp;
	interned = tmp;
	}
    return interned;
    }


//-----------------------------------------------------------------------------
// SokoPool_store_data
//-----------------------------------------------------------------------------
char const* SokoPool_store_data(
    SokoPool p, char const* data, unsigned long nbytes )
    {
    char* m;
    SOKO_ASSERT( data != 0 );
    SOKO_ASSERT( nbytes != 0 );
    m = pool_allocate( p, nbytes );
    memcpy( m, data, nbytes );
    return m;
    }


//-----------------------------------------------------------------------------
// SokoPool_store_data_and_terminate
//-----------------------------------------------------------------------------
char const* SokoPool_store_data_and_terminate(
    SokoPool p, char const* data, unsigned long nbytes )
    {
    char* m;
    SOKO_ASSERT( data != 0 );
    m = pool_allocate( p, nbytes + 1 );
    memcpy( m, data, nbytes );
    m[ nbytes ] = '\0';
    return m;
    }


//-----------------------------------------------------------------------------
// SokoPool_new
//-----------------------------------------------------------------------------
SokoPool SokoPool_new( void* info )
    {
    SokoPool p = (SokoPool)malloc(sizeof(struct _SokoPool));
    p->data = (struct SokoPoolData*)malloc(sizeof(struct SokoPoolData));
    p->data->slots = 0;
    p->data->slots_count = 0;
    p->data->slots_max = 0;
    p->data->pool_next = 0;
    p->data->pool_max = 0;
    p->data->null_string = 0;
    p->info = info;
    return p;
    }


//-----------------------------------------------------------------------------
// SokoPool_destroy
//-----------------------------------------------------------------------------
void SokoPool_destroy( SokoPool p )
    {
    if (p->data->slots_count != 0)
	{
	unsigned long i, n = p->data->slots_count;
	for (i = 0; i < n; i++)
	    free(p->data->slots[i]);
	free(p->data->slots);
	}
    free(p->data);
    free(p);
    }
