//-----------------------------------------------------------------------------
// SokoGeometry.c
//
//	Geometric utility routines for SokoSave.
//
// Copyright (c), 2002, Eric Sunshine <sunshine@sunshineco.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoGeometry.c,v 1.1 2002/02/19 09:52:49 sunshine Exp $
// $Log: SokoGeometry.c,v $
// Revision 1.1  2002/02/19 09:52:49  sunshine
// v18
// -*- Added support for new triangular-style Trioban puzzles.
//
// -*- Consolidated all of the geometry-related utility code into the new
//     SokoGeometry facility.  This mechanism provides hit-testing and
//     coverage-testing facilities for square-, hexagonal-, and
//     triangular-tiled grids.  All high-level GUI code now utilizes these
//     facilities rather than relying upon their own copy/paste
//     implementations.
//
//-----------------------------------------------------------------------------
#include "SokoGeometry.h"

//-----------------------------------------------------------------------------
// constrain
//-----------------------------------------------------------------------------
static int constrain( int n, int nmin, int nmax )
    {
    return (n < nmin ? nmin : (n > nmax ? nmax : n));
    }


//-----------------------------------------------------------------------------
// soko_hit_test_square
//	Calculate the row and column for a given point on a square grid.
//-----------------------------------------------------------------------------
soko_bool soko_hit_test_square(
    int x,
    int y,
    int rows,
    int cols,
    int cell_width,
    int cell_height,
    int origin_x,
    int origin_y,
    int* pr,
    int* pc )
    {
    if (x >= origin_x && x < origin_x + cols * cell_width &&
	y >= origin_y && y < origin_y + rows * cell_height)
	{
	*pr = y / cell_height;
	*pc = x / cell_width;
	}
    else
	{
	*pr = -1;
	*pc = -1;
	}
    return (*pr >= 0 && *pr < rows && *pc >= 0 && *pc < cols);
    }


//-----------------------------------------------------------------------------
// soko_coverage_square
//	Determine which rows and columns are covered by a rectangle for a
//	square grid.
//-----------------------------------------------------------------------------
void soko_coverage_square(
    int x,
    int y,
    int width,
    int height,
    int rows,
    int cols,
    int cell_width,
    int cell_height,
    int origin_x,
    int origin_y,
    int* r,
    int* c,
    int* nr,
    int* nc )
    {
    int rn, cn;

    soko_hit_test_square( x, y, rows, cols, cell_width, cell_height,
	origin_x, origin_y, r, c );
    *r = constrain( *r, 0, rows - 1 );
    *c = constrain( *c, 0, cols - 1 );

    soko_hit_test_square( x + width - 1, y + height - 1, rows, cols,
	cell_width, cell_height, origin_x, origin_y, &rn, &cn );
    *nr = constrain( rn, 0, rows - 1 ) - *r + 1;
    *nc = constrain( cn, 0, cols - 1 ) - *c + 1;
    }


//-----------------------------------------------------------------------------
// soko_hit_test_hexagon
//
//	Calculate the row and column for a given point on a hexagonal grid.
//	This function works by laying a grid of boxes over the hexagons, as
//	illustrated below.
//	 _____ _____
//	| / \ | / \ |
//	|/   \|/   \|
//	| 0,0 | 0,1 |
//	|_____|_____|__
//	 \ | / \ | / \ |
//	  \|/   \|/   \|
//	   | 1,0 | 1,1 |
//	   |_____|_____|
//	  / \   / \   /
//	 /   \ /   \ /
//
//	Each box is cell_width units wide, and (cell_height - cell_y_inset)
//	units high, and encompasses a large portion of its underlying hexagon,
//	plus small triangular slivers of the hexagons above to the left and
//	right.  To determine which hexagon contains a point, first determine
//	which box contains the point.  Next, check if the point is further
//	contained by one of the triangular slivers associated with that box.
//	If contained in the left sliver, then the point actually belongs to the
//	hexagon above and to the left.  If contained in the right sliver, then
//	the point belongs to the hexagon above and to the right.
//
//	The computed row and/or column may be -1, or may greater than the
//	hexagonal grid dimensions if the point lies in the "empty" area outside
//	of the grid (even though it is still within the rectangular bounds of
//	the view).  Callers of this function must be prepared to deal with
//	these "invalid" values.
//
// *1*	Quick rejection test for out-of-bounds points.
// *2*	Calculations:
//	h => Height of each box.
//	r => Row containing point (counting from zero).
//	odd_row => Row containing point is even/odd.  Because rows are
//	    staggered, we have to perform extra work for odd rows.  For
//	    example, when computing the column containing the point, we must
//	    make allowance for the fact that odd rows are inset horizontally by
//	    cell_x_inset from even rows.  Also, a point to the left of the
//	    first cell on an odd row actually resides in column -1.  Finally,
//	    although the tiles are staggered visually, internally, they are
//	    represented as a simple two-dimensional array.  A consequence of
//	    this is that the notion of upper-left and upper-right neighbors
//	    differs depending upon whether or not the row number is even.  For
//	    instance, on an even row, the upper-left neighbor is actually above
//	    and one column to the left in the two-dimensional array
//	    representation.  Conversely, the upper-left neighbor on an odd row
//	    is the cell directly above in the two-directly array.
//	c => Column containing point (counting from zero).  This may be -1 on
//	    odd rows if the point lies to the left of the very first box.  It
//	    may also be greater than the number of grid columns if the point
//	    lies to the right of the last box on even rows.  If these "invalid"
//	    values make it through all remaining calculations unaltered, then
//	    they will be rejected.
// *3*	Translate point to box {0,0} to simplify triangular sliver checks.
//	y => Relative Y location of point in box {0,0}.
//	x => Relative X location of point in box {0,0}.
// *4*	Use the standard line formula (y = mx + b) to compute the value of Y on
//	the line at point `x'.  If the actual Y (from `y' above) is less than
//	the the Y value on the line at point `x', then the point is contained
//	in the triangle.  Note that the coordinate system is flipped from the
//	usual mathematical representation where (0,0) is at the bottom-left.
//	In this case, (0,0) is at the top-left.  Because the coordinate system
//	is flipped, `m' is the slope of the hypotenuse of the triangular sliver
//	on the right, whereas `-m' is the slope of the hypotenuse of the sliver
//	on the left.  For the sliver on the left, `b' is cell_y_inset; and is
//	-cell_y_inset for the sliver on the right.  It turns out that the value
//	of (y = mx + b) for each hypotenuse is merely the negative of the
//	other.  We take advantage of this by only calculating it once (as
//	`ylim') and then simply changing the sign later on.
// *5*	If the actual `y' value is less than the Y value of the negatively
//	sloped line at `x', then the point actually lies in the upper-left
//	neighbor.
// *6*	Else if the actual `y' value is less than the Y value of the positively
//	sloped line at `x', then the point actually lies in the upper-right
//	neighbor.  If neither, then the point lies within the cell at {r,c}.
//-----------------------------------------------------------------------------
soko_bool soko_hit_test_hexagon(
    int ix,
    int iy,
    int rows,
    int cols,
    int cell_width,
    int cell_height,
    int cell_x_inset,
    int cell_y_inset,
    int origin_x,
    int origin_y,
    int* pr,
    int* pc )
    {
    int const max_x = origin_x + cols * cell_width + cell_x_inset;
    int const max_y =
	origin_y + rows * (cell_height - cell_y_inset) + cell_y_inset;
    if (ix >= origin_x && ix < max_x && iy >= origin_y && iy < max_y)	// *1*
	{
	int const h = cell_height - cell_y_inset;			// *2*
	int r = (int)(iy / h);
	soko_bool const odd_row = ((r & 1) == 0);
	int c = (int)((ix + (odd_row ? 0 : cell_x_inset)) / cell_width) -
	    (odd_row ? 0 : 1);
	float const y = iy - r * h;					// *3*
	float const x = ix - c * cell_width - (odd_row ? 0 : cell_x_inset);
	float const m = (float)cell_y_inset / (float)cell_x_inset;	// *4*
	float const ylim = m * x - cell_y_inset;
	if (y < -ylim)							// *5*
	    {
	    r--;
	    if (odd_row)
		c--;
	    }
	else if (y < ylim)						// *6*
	    {
	    r--;
	    if (!odd_row)
		c++;
	    }
	*pr = r;
	*pc = c;
	}
    else
	{
	*pr = -1;
	*pc = -1;
	}
    return (*pr >= 0 && *pr < rows && *pc >= 0 && *pc < cols);
    }


//-----------------------------------------------------------------------------
// soko_coverage_hexagon
//	Determine which rows and columns are covered by a rectangle for a
//	hexagonal grid.
// *IMPLEMENTATION*
//	This is a poor-man's implementation.  Rather than precisely computing
//	the coverage, it merely expands the rectangle and assumes that the
//	cells touched by the corners of the expanded rectangle are the coverage
//	extents.  In the future, this should be re-implemented to perform a
//	precise coverage computation.
//-----------------------------------------------------------------------------
void soko_coverage_hexagon(
    int x,
    int y,
    int width,
    int height,
    int rows,
    int cols,
    int cell_width,
    int cell_height,
    int cell_x_inset,
    int cell_y_inset,
    int origin_x,
    int origin_y,
    int* r,
    int* c,
    int* nr,
    int* nc )
    {
    int rn, cn;
    int const max_x = origin_x + cols * cell_width + cell_x_inset - 1;
    int const max_y =
	origin_y + rows * (cell_height - cell_y_inset) + cell_y_inset - 1;

    x = constrain( x - cell_width,  origin_x, max_x );
    y = constrain( y - cell_height, origin_y, max_y );
    soko_hit_test_hexagon( x, y, rows, cols, cell_width, cell_height,
	cell_x_inset, cell_y_inset, origin_x, origin_y, r, c );
    *r = constrain( *r, 0, rows - 1 );
    *c = constrain( *c, 0, cols - 1 );

    x = constrain( x + width  + cell_width,  origin_x, max_x );
    y = constrain( y + height + cell_height, origin_y, max_y );
    soko_hit_test_hexagon( x, y, rows, cols, cell_width, cell_height,
	cell_x_inset, cell_y_inset, origin_x, origin_y, &rn, &cn );
    *nr = constrain( rn, 0, rows - 1 ) - *r + 1;
    *nc = constrain( cn, 0, cols - 1 ) - *c + 1;
    }


//-----------------------------------------------------------------------------
// soko_hit_test_triangle
//
//	Calculate the row and column for a given point on a triangular grid.
//	This function works by laying a grid of boxes over the triangles, as
//	illustrated below.
//	 _______________________
//	|\  |  /|\  |  /|\  |  /
//	| \ | / | \ | / | \ | /
//	|__\|/__|__\|/__|__\|/
//	|  /|\  |  /|\  |  /|\ 
//	| / | \ | / | \ | / | \ 
//	|/__|__\|/__|__\|/__|__\ 
//	 \     / \     / \     /
//	  \   /   \   /   \   /
//	   \ /_____\ /_____\ /
//
//	Each box is cell_half_width units wide, and cell_height units high, and
//	is composed of two triangle halves; each half from a different
//	triangular cell.  To determine which triangle contains a point, first
//	determine which box contains the point.  Next, check if the point is
//	contained in the tringle half to the left of the diagonal dividing line
//	or the right.
//
//	The computed row and/or column may be -1, or may greater than the
//	triangular grid dimensions if the point lies in the "empty" area
//	outside of the grid (even though it is still within the rectangular
//	bounds of the view).  Callers of this function must be prepared to deal
//	with these "invalid" values.
//
// *1*	Quick rejection test for out-of-bounds points.
// *2*	Calculations:
//	r => Row containing point (counting from zero).
//	c => Column of box containing point (counting from zero).
// *3*	Translate point to box {0,0} to simplify triangle halves check.
//	y => Relative Y location of point in box {0,0}.
//	x => Relative X location of point in box {0,0}.
// *4*	Compute the slope of the line which divides the box in half.  The
//	magnitude of the slope is the same for all boxes, though its sign
//	toggles from box to box.  Later on, when the slope is used in the
//	standard line formula (y = mx + b), the sign will be adjusted as
//	needed.  Note that the coordinate system is flipped from the usual
//	mathematical representation where (0,0) is at the bottom-left.  In this
//	case, (0,0) is at the top-left.
// *5*	Determine the direction of the line which divides the box.
// *6*	Use the standard line formula (y = mx + b) to compute the value of Y on
//	the line at point `x'.
// *7*	If the actual `y' value indicates that `x' lies to the left of the
//	dividing line, then the point actually lies in the triangular cell to
//	the left of the current column.
//-----------------------------------------------------------------------------
soko_bool soko_hit_test_triangle(
    int ix,
    int iy,
    int rows,
    int cols,
    int cell_width,
    int cell_height,
    int cell_half_width,
    int origin_x,
    int origin_y,
    int* pr,
    int* pc )
    {
    int const max_x = origin_x + (cols + 1) * cell_half_width;
    int const max_y = origin_y + rows * cell_height;
    if (ix >= origin_x && ix < max_x && iy >= origin_y && iy < max_y)	// *1*
	{
	int const r = iy / cell_height;					// *2*
	int c = ix / cell_half_width;
	float const y = iy - r * cell_height;				// *3*
	float const x = ix - c * cell_half_width;
	float const m = (float)cell_height / (float)cell_half_width;	// *4*
	if (((r & 1) ^ (c & 1)) == 0)					// *5*
	    {			 // South-facing triangle.
	    float const ylim = m * x;					// *6*
	    if (y > ylim)						// *7*
		c -= 1;
	    }
	else
	    {			 // North-facing triangle.
	    float const ylim = -m * x + cell_height;			// *6*
	    if (y < ylim)						// *7*
		c -= 1;
	    }
	*pr = r;
	*pc = c;
	}
    else
	{
	*pr = -1;
	*pc = -1;
	}
    return (*pr >= 0 && *pr < rows && *pc >= 0 && *pc < cols);
    }


//-----------------------------------------------------------------------------
// soko_coverage_triangle
//	Determine which rows and columns are covered by a rectangle for a
//	triangular grid.
// *IMPLEMENTATION*
//	This is a poor-man's implementation.  Rather than precisely computing
//	the coverage, it merely expands the rectangle and assumes that the
//	cells touched by the corners of the expanded rectangle are the coverage
//	extents.  In the future, this should be re-implemented to perform a
//	precise coverage computation.
//-----------------------------------------------------------------------------
void soko_coverage_triangle(
    int x,
    int y,
    int width,
    int height,
    int rows,
    int cols,
    int cell_width,
    int cell_height,
    int cell_half_width,
    int origin_x,
    int origin_y,
    int* r,
    int* c,
    int* nr,
    int* nc )
    {
    int rn, cn;
    int const max_x = origin_x + (cols + 1) * cell_half_width - 1;
    int const max_y = origin_y + rows * cell_height - 1;

    x = constrain( x - cell_width,  origin_x, max_x );
    y = constrain( y - cell_height, origin_y, max_y );
    soko_hit_test_triangle( x, y, rows, cols, cell_width, cell_height,
	cell_half_width, origin_x, origin_y, r, c );
    *r = constrain( *r, 0, rows - 1 );
    *c = constrain( *c, 0, cols - 1 );

    x = constrain( x + width  + cell_width,  origin_x, max_x );
    y = constrain( y + height + cell_height, origin_y, max_y );
    soko_hit_test_triangle( x, y, rows, cols, cell_width, cell_height,
	cell_half_width, origin_x, origin_y, &rn, &cn );
    *nr = constrain( rn, 0, rows - 1 ) - *r + 1;
    *nc = constrain( cn, 0, cols - 1 ) - *c + 1;
    }
