//-----------------------------------------------------------------------------
// SokoFile.c
//
//	Utility routines for manipulating filenames.
//
// Copyright (c), 1997,2001,2002, Eric Sunshine <sunshine@sunshineco.com>
// Copyright (c), 1997, Paul McCarthy <zarnuk@high-speed-software.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoFile.c,v 1.4 2002/01/29 18:22:18 sunshine Exp $
// $Log: SokoFile.c,v $
// Revision 1.4  2002/01/29 18:22:18  sunshine
// v17:
// -*- Removed soko_get_puzzle_extension() and soko_get_save_extension() from
//     SokoFile.
//
// -*- Relocated functionality of soko_save_filename_for_puzzle(),
//     soko_puzzle_name_for_level(), and soko_level_for_puzzle_name() from
//     SokoFile to SokoPuzzle, where they are class methods of SokoPuzzle.
//
// Revision 1.3  2001/12/23 00:37:01  sunshine
// v15
// -*- Extracted core game logic out of GUI code and generalized it so that
//     the same core code can be used by any platform.  Logic from
//     SokoBoard.m now resides in SokoPuzzle.c, etc.
//
// -*- Replaced the terminology "maze" with "puzzle" throughout the project,
//     including source code, documentation, and all user-visible UI
//     elements.  The only remaining place where "maze" is still used is in
//     the file extension ".sokomaze".  I haven't decided what, if anything,
//     to do about that, yet.
//
// -*- Added the new pseudo-variable $(SokoUser) which points at the user's
//     "personal" file space.  This is where user-specific SokoSave files are
//     stored by default.  This variable complements the existing $(SokoSave)
//     variable.
//
// -*- Added SokoPool which is a memory pool object.  This is used heavily by
//     the string manipulation functions in SokoUtil to ensure that composed
//     pathnames and strings do not go out of scope before the client is
//     finished with them.  This replaces the old statically allocated,
//     circular array of pathname buffers which was used in the past.
//
// -*- Renamed soko_collapse() to soko_collapse_path().  Renamed
//     soko_expand() to soko_expand_path().
//
// -*- The path setting functions in SokoFile now invoke soko_collapse_path()
//     on all set operations.  This ensures that all paths displayed on the
//     preferences panel are properly collapsed, and removes the onus of this
//     task from the UI code.  Previously, soko_collapse_path() was only
//     called by the UI code for the puzzle path.
//
// -*- Added soko_normalize_path() and soko_denormalize_path() to SokoUtil.
//     All path manipulation functions now utilize these functions in order
//     to ensure correctness of path parsing and composition on all
//     platforms.
//
// -*- Moved soko_filename_part(), soko_directory_part(), soko_basename(),
//     and soko_mkdir() from SokoFile to SokoUtil.  Augmented these functions
//     so that they work correctly with pathnames from Unix, Macintosh, and
//     Windows (including Microsoft's UNC-style pathnames).
//
// -*- Added soko_add_path_component(), soko_add_path_extension(),
//     soko_replace_path_extension(), soko_remove_path_extension(), and
//     soko_path_exists() to SokoUtil.  These functions deal correctly with
//     pathnames from Unix, Macintosh, and Windows (including Microsoft's
//     UNC-style pathnames).
//
// -*- Renamed soko_get_maze_extension() and soko_get_maze_directory() in
//     SokoFile to soko_get_puzzle_extension() and
//     soko_get_puzzle_directory(), respectively.
//
// -*- Added soko_set_puzzle_directory(), soko_set_save_directory(), and
//     soko_set_score_file() to SokoFile in order to complement the existing
//     "get" functions and to centralize control over these settings.
//
// -*- Added soko_get_default_puzzle_directory(),
//     get_default_save_directory(), and soko_get_default_score_file() to
//     SokoFile.  These functions return values appropriate for the
//     corresponding fields on the Preferences panel of each port when the
//     "Defaults" button is pressed.
//
// -*- Renamed soko_save_filename_for_maze_name() and
//     soko_maze_name_for_level() in SokoFile to to
//     soko_save_filename_for_puzzle_name() and soko_puzzle_name_for_level(),
//     respectively.
//
// -*- Added soko_level_for_puzzle_name() to SokoFile.  This function is used
//     by the new "Next Puzzle" menu item and shortcut.
//
// -*- Augmented soko_collapse_path() so that it also knows how to emit
//     $(SokoUser), $(HOME), $(TEMP), and $(TMP), in addition to the existing
//     $(SokoSave).
//
// -*- Added SokoSetting implementation which provides a platform-independent
//     API for accessing user settings and well-known paths, such as
//     $(SokoSave) and $(SokoUser).
//-----------------------------------------------------------------------------
#include "SokoPool.h"
#include "SokoFile.h"
#include "SokoSetting.h"
#include "SokoUtil.h"
#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

static char const* SOKO_ENV_WELL_KNOWN[] = { "HOME", "TEMP", "TMP" };
#define SOKO_ENV_WELL_KNOWN_COUNT \
    (sizeof(SOKO_ENV_WELL_KNOWN) / sizeof(SOKO_ENV_WELL_KNOWN[0]))

//-----------------------------------------------------------------------------
// get_settings
//-----------------------------------------------------------------------------
static SokoSetting SOKO_SETTINGS = 0;
static SokoSetting get_settings( void )
    {
    if (SOKO_SETTINGS == 0)
	SOKO_SETTINGS = SokoSetting_new(0);
    return SOKO_SETTINGS;
    }


//-----------------------------------------------------------------------------
// Path accessors
//-----------------------------------------------------------------------------
static void set_path( char const* name, char const* path, cSokoPool pool )
    {
    SokoSetting_set_string(get_settings(), name,
	soko_collapse_path(pool, soko_expand_path(pool, path)));
    }

static char const* get_fallback_path(
    char const* fallback, soko_bool use_factory, cSokoPool pool )
    {
    SokoSetting settings = get_settings();
    char const* p = use_factory ?
	SokoSetting_factory_path( settings, pool ) :
	SokoSetting_user_path( settings, pool );
    if (fallback != 0 && strlen(fallback) != 0)
	p = soko_add_path_component( pool, p, fallback );
    return soko_collapse_path( pool, p );
    }

static char const* get_path( char const* name, char const* fallback,
    soko_bool use_factory, cSokoPool pool )
    {
    char const* s = SokoSetting_get_string( get_settings(), name, 0, pool );
    if (s != 0)
	return soko_collapse_path( pool, soko_expand_path(pool, s) );
    else
	return get_fallback_path( fallback, use_factory, pool );
    }

void soko_set_puzzle_directory( cSokoPool pool, char const* path )
{ set_path( SOKO_SETTING_PUZZLE_DIR, path, pool ); }
char const* soko_get_puzzle_directory( cSokoPool pool )
{ return get_path( SOKO_SETTING_PUZZLE_DIR, SOKO_DEF_PUZZLE_DIR,
    soko_true, pool ); }
char const* soko_get_default_puzzle_directory( cSokoPool pool )
{ return get_fallback_path( SOKO_DEF_PUZZLE_DIR, soko_true, pool ); }

void soko_set_save_directory( cSokoPool pool, char const* path )
{ set_path( SOKO_SETTING_SAVE_DIR, path, pool ); }
char const* soko_get_save_directory( cSokoPool pool )
{ return get_path( SOKO_SETTING_SAVE_DIR, SOKO_DEF_SAVE_DIR,
    soko_false, pool ); }
char const* soko_get_default_save_directory( cSokoPool pool )
{ return get_fallback_path( SOKO_DEF_SAVE_DIR, soko_false, pool ); }

void soko_set_score_file( cSokoPool pool, char const* path )
{ set_path( SOKO_SETTING_SCORE_FILE, path, pool ); }
char const* soko_get_score_file( cSokoPool pool )
{ return get_path( SOKO_SETTING_SCORE_FILE, SOKO_DEF_SCORES,
    soko_false, pool ); }
char const* soko_get_default_score_file( cSokoPool pool )
{ return get_fallback_path( SOKO_DEF_SCORES, soko_false, pool ); }
char const* soko_get_factory_score_file( cSokoPool pool )
    {
    return soko_add_path_component(pool,
	SokoSetting_factory_path(get_settings(),pool),SOKO_DEF_FACTORY_SCORES);
    }


//-----------------------------------------------------------------------------
// soko_try_collapse
//	Assumes incoming paths have already been normalized.
//-----------------------------------------------------------------------------
static char const* soko_try_collapse( char const* path, char const* prefix,
    char const* replacement, soko_bool* collapsed, cSokoPool pool )
    {
    *collapsed = soko_false;
    if (soko_has_prefix( path, prefix, soko_true ))
	{
	int const len = strlen( prefix );
	char* const buff = SokoPool_allocate( pool, FILENAME_MAX );
	sprintf( buff, "$(%s)%s", replacement, path + len );
	path = buff;
	*collapsed = soko_true;
	}
    return path;
    }


//-----------------------------------------------------------------------------
// soko_collapse_path
//	Collapse a full-path to "$(SokoSave)/filename" if it refers to the
//	factory directory; or to "$(SokoUser)/filename" if it refers to the
//	user directory.  Collapse a full-path to "$(ENV)/filename" if it refers
//	to the directory mentioned by the environment variable ENV (which may
//	be one of several well-known environment variables).
//-----------------------------------------------------------------------------
char const* soko_collapse_path( cSokoPool pool, char const* filename )
    {
    soko_bool collapsed;
    char const* path = soko_try_collapse(
	soko_normalize_path(pool, filename),
	soko_normalize_path(pool,
	    SokoSetting_factory_path(get_settings(), pool)), SOKO_ENV_FACTORY,
	    &collapsed, pool );
    if (!collapsed)
	path = soko_try_collapse( path,
	    soko_normalize_path(pool,
		SokoSetting_user_path(get_settings(),pool)), SOKO_ENV_USER,
		&collapsed, pool );
    if (!collapsed)
	{
	int i;
	for (i = 0; i < SOKO_ENV_WELL_KNOWN_COUNT; i++)
	    {
	    char const* var = getenv( SOKO_ENV_WELL_KNOWN[i] );
	    if (var != 0)
		{
		path = soko_try_collapse( path, soko_normalize_path(pool, var),
		    SOKO_ENV_WELL_KNOWN[i], &collapsed, pool );
		if (collapsed)
		    break;
		}
	    }
	}
    return soko_denormalize_path( pool, path );
    }


//-----------------------------------------------------------------------------
// expand_macro
//	Expand a macro.  Recognizes environment variables, and "~/" as an alias
//	for $HOME.  Also recognizes reference to the pseudo-variables
//	$(SokoSave) and $(SokoUser).  Assumes that the path has already been
//	normalized such that "/" is used as a path separator.
//-----------------------------------------------------------------------------
static void expand_macro( char* buff, cSokoPool pool )
    {
    char tmp[ FILENAME_MAX ];
    char var[ 256 ];
    char *s, *t, *p;
    char const* env;
    char const* factory_dir = SokoSetting_factory_path( get_settings(), pool );
    char const* user_dir = SokoSetting_user_path( get_settings(), pool );
    
    strcpy( tmp, buff );
    for (t = buff, s = tmp;  *s;  )
	{
	if ((*s != '~') && (*s != '$'))
	    {
	    *t++ = *s++;
	    }
	else 
	    {
	    env = 0;
	    if (*s == '~')
		{
		s++;
		env = getenv( "HOME" );
		}
	    else // (*s == '$')
		{
		s++;
		p = var;
		if ((*s == '(') || (*s == '{'))
		    {
		    s++;
		    while (*s && *s != ')' && *s != '}')
			*p++ = *s++;
		    if (*s == ')' || *s == '}')
			s++;
		    }
		else
		    {
		    while (*s && !isspace(*s) && *s != '/')
			*p++ = *s++;
		    }
		*p = '\0';
		if (*var != '\0')
		    {
		    if (soko_stricmp( var, SOKO_ENV_FACTORY ) == 0)
			env = factory_dir;
		    else if (soko_stricmp( var, SOKO_ENV_USER ) == 0)
			env = user_dir;
		    else
			env = getenv(var);
		    }
		}
	    if (env != 0)
		while (*env != '\0')
		    *t++ = *env++;
	    }
	}
    *t = '\0';
    }


//-----------------------------------------------------------------------------
// soko_expand_path
//-----------------------------------------------------------------------------
char const* soko_expand_path( cSokoPool pool, char const* filename )
    {
    char const* result = soko_normalize_path( pool, filename );
    if (strchr(result,'~') != 0 || strchr(result,'$') != 0)
	{
	char* const buff = SokoPool_allocate( pool, FILENAME_MAX );
	strcpy( buff, result );
	do { expand_macro( buff, pool ); }
	    while (strchr(buff,'~') != 0 || strchr(buff,'$') != 0);
	result = buff;
	}
    return soko_denormalize_path( pool, result );
    }
