//-----------------------------------------------------------------------------
// SokoEncode.c
//
//	Routines for using run length encoding for SokoSave.
//
// Copyright (c), 1997, Paul McCarthy <zarnuk@high-speed-software.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoEncode.c,v 1.4 2001/12/21 21:56:55 sunshine Exp $
// $Log: SokoEncode.c,v $
// Revision 1.4  2001/12/21 21:56:55  sunshine
// v15
// -*- Extracted core game logic out of GUI code and generalized it so that
//     the same core code can be used by any platform.  Logic from
//     SokoBoard.m now resides in SokoPuzzle.c, etc.
//
// -*- Augmented all input/output logic so that it now deals gracefully with
//     line terminators from all common platforms; Unix (LF), Macintosh (CR),
//     and Windows/DOS (CRLF).
//
//-----------------------------------------------------------------------------
#include "SokoEncode.h"
#include <ctype.h>

//-----------------------------------------------------------------------------
// soko_run_length_encode_char
//-----------------------------------------------------------------------------
void soko_run_length_encode_char( FILE *fp, char ch, int run_length )
    {
    if (run_length > 1)
	fprintf( fp, "%d", run_length );
    fputc( ch, fp );
    }


//-----------------------------------------------------------------------------
// soko_run_length_encode_string
//-----------------------------------------------------------------------------
void soko_run_length_encode_string( FILE *fp, char const *str, int str_length )
    {
    int run_length = 0;
    char last_ch,ch;

    if (str != 0)
	{
	last_ch = *str;
	while ((ch = *str++) != '\0' && str_length-- > 0)
	    {
	    if (ch == last_ch)
		run_length++;
	    else
		{
		soko_run_length_encode_char( fp, last_ch, run_length );
		run_length = 1;
		last_ch = ch;
		}
	    }
	if (run_length > 0)
	    soko_run_length_encode_char( fp, last_ch, run_length );
	}
    fputc( '\n', fp );
    }


//-----------------------------------------------------------------------------
// soko_run_length_decode_char
//-----------------------------------------------------------------------------
soko_bool soko_run_length_decode_char( FILE *fp, char *ch, int *run_length )
    {
    int rc = soko_false;	// Assume failure until success.
    int len = 0;
    int c;
    
    if (!feof(fp) && !ferror(fp))
	{
	c = fgetc( fp );
	if (c != EOF && isdigit(c))
	    {
	    do	{
		len = (len * 10) + (c - '0');
		c = fgetc( fp );
		}
	    while (c != EOF && isdigit(c));
	    }
	else
	    {
	    len = 1;
	    }
	if (c != EOF)
	    {
	    *ch = (char)c;
	    *run_length = len;
	    rc = soko_true;
	    }
	}
    
    return rc;
    }


//-----------------------------------------------------------------------------
// soko_run_length_decode_string
//-----------------------------------------------------------------------------
int soko_run_length_decode_string( FILE *fp, char *buff, int buff_len )
    {
    int pos = 0;	// Position to store next character in buff.
    int run_length;
    char ch;
    soko_bool saw_eol;
    
    while (soko_run_length_decode_char( fp, &ch, &run_length ) &&
	ch != '\n' && ch != '\r')
	{
	if (pos + run_length <= buff_len)
	    {
	    while (run_length-- > 0)
		buff[ pos++ ] = ch;
	    }
	else
	    {
	    pos = -1;
	    break;
	    }
	}

    saw_eol = (ch == '\n' || ch == '\r');
    if (ch == '\r') // Eat LF, CR, or CRLF terminator.
	{
	int const c = fgetc( fp );
	if (c != '\n' && c != EOF)
	    ungetc( c, fp );
	}

    return (saw_eol && pos == buff_len) ? soko_true : soko_false;
    }
