//-----------------------------------------------------------------------------
// SokoScores.cpp
//
//	High score panel for SokoSave.
//
// Copyright (c), 2001,2002, Eric Sunshine <sunshine@sunshineco.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoScores.cpp,v 1.2 2002/01/29 21:27:05 sunshine Exp $
// $Log: SokoScores.cpp,v $
// Revision 1.2  2002/01/29 21:27:05  sunshine
// v17
// -*- SokoPuzzle now calculates two new scores in addition to "moves" and
//     "pushes".  The "runs" score is the number of straight lines in which
//     crates have been pushed.  Looked at another way, it is the number of
//     turns crates have made while being pushed.  The "focus" score is the
//     number of times the player's focus has changed from one crate to
//     another.
//
// -*- Added "runs" and "focus" scores columns to scores panel and to "New
//     Score" panel.
//
// -*- Fixed a number of aesthetic problems which showed up when using larger
//     control sizes and fonts.  Controls would clobber one another and text
//     would be clipped.  The problems were particularly acute when using
//     "Large Fonts" from the Windows Display/Advanced settings.
//
// -*- SokoScores now adjusts row height of the TStringGrid, if necessary, to
//     account for large fonts.
//
// Revision 1.1  2001-12-21 16:45:58-05  sunshine
// v15
// -*- Extracted core game logic out of GUI code and generalized it so that
//     the same core code can be used by any platform.  Logic from
//     SokoBoard.m now resides in SokoPuzzle.c, etc.
//
// -*- Created a native Windows port of SokoSave using C++Builder.  The
//     back-end logic for this port is provided by the new "sokocore" common
//     code which used to be merged with the UI code for the
//     OpenStep/NextStep ports.  From the user-experience, this port is
//     nearly identical to the existing OpenStep and NextStep ports.  Builds
//     successfully with C++Builder versions 4 and 5.
//
// -*- Added a shortcut toolbar for quickly starting new games, opening saved
//     games, saving games, launching the scores panel, and launching the
//     help file.  Added a toggle-switch to the preferences panel to control
//     presence of toolbar.  From the user-standpoint, this is the only major
//     difference from the OpenStep/NextStep ports, since they do not feature
//     a shortcut toolbar.
//
// -*- Created an InnoSetup installer script for SokoSave.  This scripts
//     facilitates the creation of a stand-alone "setup" program
//     (soksetup.exe) for SokoSave.  With this program, Windows users can
//     download and install SokoSave in a fashion in which they are already
//     familiar.
//-----------------------------------------------------------------------------
#include <vcl.h>
#pragma hdrstop

#include "SokoScores.h"
#include "SokoAlerter.h"
#include "SokoPool.h"
#include "SokoPuzzle.h"
#include "SokoScore.h"
#include "SokoSetting.h"
#include <stdio.h>

#pragma package(smart_init)
#pragma resource "*.dfm"
TSokoScoresForm* SokoScoresForm = 0;

enum
    {
    COL_PUZZLE,
    COL_MOVES,
    COL_PUSHES,
    COL_RUNS,
    COL_FOCUS,
    COL_PLAYER,
    COL_DATE,
    COL_NOTES
    };
#define COL_MAX (COL_NOTES + 1)

static int COL_WIDTH[ COL_MAX ] = { 48, 50, 50, 50, 50, 80, 120, 100 };
static char const* const COL_TITLE[ COL_MAX ] =
    {
    "Puzzle",
    "Moves",
    "Pushes",
    "Runs",
    "Focus",
    "Player",
    "Date",
    "Notes"
    };

struct SokoScoresColumn { int col; SokoScoreSort sort; };
static SokoScoresColumn COL_SORT[ COL_MAX ] =
    {
    { COL_PUZZLE, SOKO_SCORE_PUZZLE },
    { COL_MOVES,  SOKO_SCORE_MOVES  },
    { COL_PUSHES, SOKO_SCORE_PUSHES },
    { COL_RUNS,   SOKO_SCORE_RUNS   },
    { COL_FOCUS,  SOKO_SCORE_FOCUS  },
    { COL_PLAYER, SOKO_SCORE_PLAYER },
    { COL_DATE,   SOKO_SCORE_DATE   },
    { COL_NOTES,  SOKO_SCORE_NOTES  }
    };
#define COLUMN_INFO "ScoreColumns"

//-----------------------------------------------------------------------------
// SokoScoresDelegate
//	Concrete implementation of abstract SokoScoreDelegate which provides
//	SokoScore module with required implementation-specific functionality.
//-----------------------------------------------------------------------------
class SokoScoresDelegate : public SokoScoreDelegate, public SokoAlerter
    {
public:
    virtual void alert(SokoScore*,SokoAlert,char const* title,char const* msg);
    };

void SokoScoresDelegate::alert(
    SokoScore* s, SokoAlert sev, char const* title, char const* msg )
    { send_alert( sev, title, msg ); }


//-----------------------------------------------------------------------------
// shared_instance
//-----------------------------------------------------------------------------
static TSokoScoresForm* shared_instance()
    {
    if (SokoScoresForm == 0)
	SokoScoresForm = new TSokoScoresForm( Application );
    return SokoScoresForm;
    }


//-----------------------------------------------------------------------------
// launch
//-----------------------------------------------------------------------------
void TSokoScoresForm::launch()
    {
    TSokoScoresForm* form = shared_instance();
    if (form->WindowState == wsMinimized)
	ShowWindow( form->Handle, SW_RESTORE );
    form->Show();
    }


//-----------------------------------------------------------------------------
// record_score
//-----------------------------------------------------------------------------
void TSokoScoresForm::record_score( SokoPuzzle& puzzle, int moves, int pushes,
    int runs, int focus, char const* player, char const* notes )
    {
    TSokoScoresForm* form = shared_instance();
    form->add_score( puzzle, moves, pushes, runs, focus, player, notes );
    launch();
    }


//-----------------------------------------------------------------------------
// Constructor
//-----------------------------------------------------------------------------
__fastcall TSokoScoresForm::TSokoScoresForm( TComponent* owner ) :
    TSokoForm(owner)
    {
    delegate = new SokoScoresDelegate;
    scores = new SokoScore( delegate, 0 );
    }


//-----------------------------------------------------------------------------
// Destructor
//-----------------------------------------------------------------------------
__fastcall TSokoScoresForm::~TSokoScoresForm()
    {
    save_column_info();
    delete scores;
    delete delegate;
    }


//-----------------------------------------------------------------------------
// restore_column_info
//-----------------------------------------------------------------------------
void TSokoScoresForm::restore_column_info()
    {
    SokoPool pool;
    SokoSetting setting;
    char const* s = setting.get_string( COLUMN_INFO, 0, pool );
    if (s != 0)
	{
	int wstored[ COL_MAX ];
	SokoScoresColumn ostored[ COL_MAX ];
	int rc = sscanf( s, "%d:%d %d:%d %d:%d %d:%d %d:%d %d:%d %d:%d %d:%d",
	    &ostored[COL_PUZZLE].col, &wstored[COL_PUZZLE],
	    &ostored[COL_MOVES ].col, &wstored[COL_MOVES ],
	    &ostored[COL_PUSHES].col, &wstored[COL_PUSHES],
	    &ostored[COL_RUNS  ].col, &wstored[COL_RUNS  ],
	    &ostored[COL_FOCUS ].col, &wstored[COL_FOCUS ],
	    &ostored[COL_PLAYER].col, &wstored[COL_PLAYER],
	    &ostored[COL_DATE  ].col, &wstored[COL_DATE  ],
	    &ostored[COL_NOTES ].col, &wstored[COL_NOTES ] );
	if (rc == COL_MAX * 2)
	    {
	    for (int i = 0; i < COL_MAX; i++)
		{
		int const c = ostored[i].col;
		ostored[i].sort = COL_SORT[c].sort;
		COL_WIDTH[c] = wstored[i];
		}
	    memcpy( COL_SORT, ostored, sizeof(COL_SORT) );
	    }
	}
    for (int i = 0; i < COL_MAX; i++)
	table->ColWidths[i] = COL_WIDTH[COL_SORT[i].col];
    }


//-----------------------------------------------------------------------------
// save_column_info
//-----------------------------------------------------------------------------
void TSokoScoresForm::save_column_info()
    {
    AnsiString s;
    for (int i = 0; i < COL_MAX; i++)
	{
	if (i != 0)
	    s += ' ';
	s += COL_SORT[i].col;
	s += ':';
	s += table->ColWidths[i];
	}
    SokoSetting setting;
    setting.set_string( COLUMN_INFO, s.c_str() );
    }


//-----------------------------------------------------------------------------
// form_create
//-----------------------------------------------------------------------------
void __fastcall TSokoScoresForm::form_create( TObject* sender )
    {
    int const h = table->Font->Size * table->Font->PixelsPerInch / 72 + 3;
    if (table->DefaultRowHeight < h)
	table->DefaultRowHeight = h;
    table->ColCount = COL_MAX;
    restore_column_info();
    set_titles();
    sort_scores();
    load_scroll();
    }


//-----------------------------------------------------------------------------
// form_close
//-----------------------------------------------------------------------------
void __fastcall TSokoScoresForm::form_close(
    TObject* sender, TCloseAction& action )
    {
    action = caHide;        
    }


//-----------------------------------------------------------------------------
// set_titles
//-----------------------------------------------------------------------------
void TSokoScoresForm::set_titles()
    {
    for (int i = 0; i < COL_MAX; i++)
	table->Cells[i][0] = COL_TITLE[COL_SORT[i].col];
    }


//-----------------------------------------------------------------------------
// sort_scores
//-----------------------------------------------------------------------------
void TSokoScoresForm::sort_scores()
    {
    SokoScoreSort criteria[ COL_MAX ];
    for (int i = 0; i < COL_MAX; i++)
	criteria[i] = COL_SORT[i].sort;
    scores->sort( criteria, sizeof(criteria) / sizeof(criteria[0]) );
    }


//-----------------------------------------------------------------------------
// format_int
//-----------------------------------------------------------------------------
static AnsiString format_int( int n )
    {
    AnsiString s;
    if (n >= 0)
	s = n;
    return s;
    }


//-----------------------------------------------------------------------------
// load_scroll
//	TStringGrid has the strange limitation that there must be at least one
//	data row if any "fixed rows" have been defined.  Therefore, we must
//	ensure that at least one data row exists even if we have no data with
//	which to fill that row.  This is accomplished when setting RowCount.
//-----------------------------------------------------------------------------
void TSokoScoresForm::load_scroll()
    {
    AnsiString values[ COL_MAX ];
    int const n = scores->score_count();
    table->RowCount = (n != 0 ? n : 1) + table->FixedRows;
    for (int i = 0; i < n; i++)
	{
	SokoHighScore const* s = scores->get_score(i);
	values[COL_PUZZLE] = s->puzzle;
	values[COL_MOVES ] = format_int( s->moves  );
	values[COL_PUSHES] = format_int( s->pushes );
	values[COL_RUNS  ] = format_int( s->runs   );
	values[COL_FOCUS ] = format_int( s->focus  );
	values[COL_PLAYER] = s->player;
	values[COL_DATE  ] = s->date_formatted;
	values[COL_NOTES ] = s->notes;

	int const r = i + 1; // Plus one to skip title row.
	for (int j = 0; j < COL_MAX; j++)
	    table->Cells[j][r] = values[COL_SORT[j].col];
	}
    }


//-----------------------------------------------------------------------------
// table_column_moved
//-----------------------------------------------------------------------------
void __fastcall TSokoScoresForm::table_column_moved(
    TObject* sender, int from, int to )
    {
    SokoScoresColumn const c = COL_SORT[ from ];
    if (from < to)
	memmove( COL_SORT + from, COL_SORT + from + 1, (to-from) * sizeof(c) );
    else // (from > to)
	memmove( COL_SORT + to + 1, COL_SORT + to, (from-to) * sizeof(c) );
    COL_SORT[ to ] = c;

    sort_scores();
    set_titles();
    load_scroll();
    }


//-----------------------------------------------------------------------------
// scroll_row_to_visible
//	Scroll row to visible if not already so.  When scrolling, try to
//	center row vertically.
//-----------------------------------------------------------------------------
void TSokoScoresForm::scroll_row_to_visible( int row )
    {
    table->LeftCol = 0;
    row += table->FixedRows; // Skip over column titles.
    if (row < table->TopRow || row >= table->TopRow + table->VisibleRowCount)
	{
	int top = row - table->VisibleRowCount / 2;
	if (top < 0)
	    top = table->FixedRows;
	table->TopRow = top;
	}
    }


//-----------------------------------------------------------------------------
// highlight_row
//-----------------------------------------------------------------------------
void TSokoScoresForm::highlight_row( int row )
    {
    TGridRect r;
    row += table->FixedRows; // Skip over column titles.
    r.Top = row;
    r.Bottom = row;
    r.Left = 0;
    r.Right = table->ColCount - 1;
    table->Selection = r;
    }


//-----------------------------------------------------------------------------
// add_score
//-----------------------------------------------------------------------------
void TSokoScoresForm::add_score( SokoPuzzle& puzzle, int moves, int pushes,
    int runs, int focus, char const* player, char const* notes )
    {
    int row = scores->add( puzzle, moves, pushes, runs, focus, player, notes );
    load_scroll();
    scroll_row_to_visible( row );
    highlight_row( row );
    }
