//-----------------------------------------------------------------------------
// SokoRendezvous.h
//
//	A facility which provides a mechanism for ensuring that only a single
//	instance of a particular application is running at any particular time.
//
// Copyright (c), 2001, Eric Sunshine <sunshine@sunshineco.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoRendezvous.cpp,v 1.1 2001/12/21 21:46:24 sunshine Exp $
// $Log: SokoRendezvous.cpp,v $
// Revision 1.1  2001/12/21 21:46:24  sunshine
// v15
// -*- Extracted core game logic out of GUI code and generalized it so that
//     the same core code can be used by any platform.  Logic from
//     SokoBoard.m now resides in SokoPuzzle.c, etc.
//
// -*- Created a native Windows port of SokoSave using C++Builder.  The
//     back-end logic for this port is provided by the new "sokocore" common
//     code which used to be merged with the UI code for the
//     OpenStep/NextStep ports.  From the user-experience, this port is
//     nearly identical to the existing OpenStep and NextStep ports.  Builds
//     successfully with C++Builder versions 4 and 5.
//
// -*- Added a shortcut toolbar for quickly starting new games, opening saved
//     games, saving games, launching the scores panel, and launching the
//     help file.  Added a toggle-switch to the preferences panel to control
//     presence of toolbar.  From the user-standpoint, this is the only major
//     difference from the OpenStep/NextStep ports, since they do not feature
//     a shortcut toolbar.
//
// -*- Created an InnoSetup installer script for SokoSave.  This scripts
//     facilitates the creation of a stand-alone "setup" program
//     (soksetup.exe) for SokoSave.  With this program, Windows users can
//     download and install SokoSave in a fashion in which they are already
//     familiar.
//
//-----------------------------------------------------------------------------
#include "SokoRendezvous.h"

#define SOKO_RENDEZVOUS_CLASS "SokoRendezvous"

//-----------------------------------------------------------------------------
// Constructor
//-----------------------------------------------------------------------------
SokoRendezvous::SokoRendezvous() : listener(0), mutex(0), window(0)
    {
    }


//-----------------------------------------------------------------------------
// Destructor
//-----------------------------------------------------------------------------
SokoRendezvous::~SokoRendezvous()
    {
    if (window != 0)
	DestroyWindow( window );
    if (mutex != 0)
	{
	ReleaseMutex( mutex );
	CloseHandle( mutex );
	}
    }


//-----------------------------------------------------------------------------
// window_proc
//	When a secondary instance of this application is invoked, it will send
//	a WM_COPYDATA message containing its command-line arguments to the
//	SOKO_RENDEZVOUS_CLASS window owned by the first running instance. This
//	function passes the command-line arguments along to the listener.
//-----------------------------------------------------------------------------
LRESULT CALLBACK SokoRendezvous::window_proc(
    HWND w, UINT msg, WPARAM wparam, LPARAM lparam )
    {
    LRESULT rc;
    if (msg != WM_COPYDATA)
	rc = DefWindowProc( w, msg, wparam, lparam );
    else
	{
	SokoRendezvous* p = (SokoRendezvous*)GetWindowLong( w, GWL_USERDATA );
	if (p != 0)
	    p->listener( (char const*)(((COPYDATASTRUCT*)lparam)->lpData) );
	rc = TRUE;
	}
    return rc;
    }


//-----------------------------------------------------------------------------
// send_command_line
//	Send the command line arguments as a packed buffer of null-terminated
//	strings.  The entire list of strings is terminated by a null byte.
//-----------------------------------------------------------------------------
void SokoRendezvous::send_command_line( HWND w )
    {
    int i, len = 1, n = ParamCount();
    for (i = 0; i < n; i++)
	len += ParamStr( i + 1 ).Length() + 1;
    char* s = new char[ len ];
    char* t = s;
    for (i = 0; i < n; i++)
	{
	AnsiString const& p = ParamStr( i + 1 );
	int const l = p.Length() + 1;
	memcpy( t, p.c_str(), l );
	t += l;
	}
    *t = '\0';

    COPYDATASTRUCT r;
    r.dwData = n;
    r.lpData = s;
    r.cbData = len;
    SendMessage( w, WM_COPYDATA, 0, (WPARAM)&r );
    delete[] s;
    }


//-----------------------------------------------------------------------------
// rendezvous
//-----------------------------------------------------------------------------
bool SokoRendezvous::rendezvous( char const* signature )
    {
    bool ok = false;
    if (mutex != 0)
	throw EInvalidOperation("SokoRendezvous rendezvous() already called.");

    mutex = CreateMutex( 0, true, signature ); // Grab as early as possible.
    if (mutex != 0)
	{
	DWORD const mutex_error = GetLastError();
    
	WNDCLASS wc;
	memset( &wc, 0, sizeof(wc) );
	wc.lpfnWndProc = window_proc;
	wc.hInstance = HInstance;
	wc.lpszClassName = SOKO_RENDEZVOUS_CLASS;
    
	if (RegisterClass( &wc ) != 0)
	    {
	    if (mutex_error != ERROR_ALREADY_EXISTS)
		{
		window = CreateWindow( SOKO_RENDEZVOUS_CLASS, signature,
		    0, 0, 0, 0, 0, 0, 0, HInstance, 0 );
		if (window != 0)
		    {
		    SetWindowLong( window, GWL_USERDATA, (LONG)this );
		    ok = true;
		    }
		}
	    else if (WaitForSingleObject( mutex, 5000 ) == WAIT_OBJECT_0)
		{
		HWND w = FindWindow( SOKO_RENDEZVOUS_CLASS, signature );
		if (w != 0)
		    send_command_line(w);
		ReleaseMutex( mutex );
		}
	    }
	}
    return ok;
    }


//-----------------------------------------------------------------------------
// listen
//-----------------------------------------------------------------------------
void SokoRendezvous::listen( Listener p )
    {
    if (p != 0 && listener != 0)
	throw EInvalidOperation("SokoRendezvous listener already registered.");
    if (p == 0 && listener == 0)
	throw EInvalidOperation("SokoRendezvous null listener registered.");
    listener = p;
    ReleaseMutex( mutex ); // Other instances may now contact us.
    }
