//-----------------------------------------------------------------------------
// SokoPref.cpp
//
//	User preference management form for SokoSave.
//
// Copyright (c), 2001,2002, Eric Sunshine <sunshine@sunshineco.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoPref.cpp,v 1.2 2002/01/29 21:29:18 sunshine Exp $
// $Log: SokoPref.cpp,v $
// Revision 1.2  2002/01/29 21:29:18  sunshine
// v17
// -*- Fixed a number of aesthetic problems which showed up when using larger
//     control sizes and fonts.  Controls would clobber one another and text
//     would be clipped.  The problems were particularly acute when using
//     "Large Fonts" from the Windows Display/Advanced settings.
//
// -*- Added `user_resizing' property to SokoForm which is `true' during a
//     user- initiated window resize operation.  This provides clients with
//     the capability to make constraint decisions based upon the agent which
//     is resizing a window, rather than always applying constraints,
//     regardless of whether resizing was initiated by the user or by Windows
//     (for instance, if font metrics had changed).
//
// -*- SokoPref controls no longer become all jumbled together if user has
//     selected large fonts.  Also, added a TPanel behind all of the controls
//     to keep them from falling off bottom of window when user changes
//     Window display scheme.
//
// Revision 1.1  2001-12-21 16:46:17-05  sunshine
// v15
// -*- Extracted core game logic out of GUI code and generalized it so that
//     the same core code can be used by any platform.  Logic from
//     SokoBoard.m now resides in SokoPuzzle.c, etc.
//
// -*- Created a native Windows port of SokoSave using C++Builder.  The
//     back-end logic for this port is provided by the new "sokocore" common
//     code which used to be merged with the UI code for the
//     OpenStep/NextStep ports.  From the user-experience, this port is
//     nearly identical to the existing OpenStep and NextStep ports.  Builds
//     successfully with C++Builder versions 4 and 5.
//
// -*- Added a shortcut toolbar for quickly starting new games, opening saved
//     games, saving games, launching the scores panel, and launching the
//     help file.  Added a toggle-switch to the preferences panel to control
//     presence of toolbar.  From the user-standpoint, this is the only major
//     difference from the OpenStep/NextStep ports, since they do not feature
//     a shortcut toolbar.
//
// -*- Created an InnoSetup installer script for SokoSave.  This scripts
//     facilitates the creation of a stand-alone "setup" program
//     (soksetup.exe) for SokoSave.  With this program, Windows users can
//     download and install SokoSave in a fashion in which they are already
//     familiar.
//-----------------------------------------------------------------------------
#include <vcl.h>
#pragma hdrstop

#include "SokoPool.h"
#include "SokoPref.h"
#include "SokoBoard.h"
#include "SokoFile.h"
#include "SokoSetting.h"
#include "SokoUtil.h"
#include <stdlib.h>

#pragma package(smart_init)
#pragma resource "*.dfm"
TSokoPrefForm* SokoPrefForm = 0;

#define SOKO_SHOW_TOOLBAR "ShowToolbar"

//-----------------------------------------------------------------------------
// launch
//-----------------------------------------------------------------------------
void TSokoPrefForm::launch()
    {
    if (SokoPrefForm == 0)
	SokoPrefForm = new TSokoPrefForm( Application );
    if (SokoPrefForm->WindowState == wsMinimized)
	ShowWindow( SokoPrefForm->Handle, SW_RESTORE );
    SokoPrefForm->Show();
    }


//-----------------------------------------------------------------------------
// should_show_toolbar
//-----------------------------------------------------------------------------
bool TSokoPrefForm::should_show_toolbar()
    {
    SokoSetting setting;
    return setting.get_bool( SOKO_SHOW_TOOLBAR, soko_true );
    }


//-----------------------------------------------------------------------------
// get_next_level
//-----------------------------------------------------------------------------
int TSokoPrefForm::get_next_level()
    {
    SokoSetting setting;
    return setting.get_int( SOKO_SETTING_LEVEL, 1 );
    }


//-----------------------------------------------------------------------------
// set_next_level
//	*NOTE* If the preference panel has been instantiated, then also set
//	the value of the "level_field".  This works correctly whether or not
//	the panel is currently visible.  Even if it is visible, just setting
//	the value of the field is sufficient; there is no need to set the
//	"dirty" flag (since the new setting is already saved).
//-----------------------------------------------------------------------------
void TSokoPrefForm::set_next_level( int n )
    {
    if (n >= 0)
	{
	SokoSetting setting;
	setting.set_int( SOKO_SETTING_LEVEL, n );
	if (SokoPrefForm != 0)
	    SokoPrefForm->level_field->Text = n;
	}
    }


//-----------------------------------------------------------------------------
// Constructor
//-----------------------------------------------------------------------------
__fastcall TSokoPrefForm::TSokoPrefForm( TComponent* owner ) :
    TSokoForm(owner),
    dirty(false),
    toolbar_switch_toggled(false),
    min_width(0),
    current_width(0)
    {
    setting = new SokoSetting;
    }


//-----------------------------------------------------------------------------
// Destructor
//-----------------------------------------------------------------------------
__fastcall TSokoPrefForm::~TSokoPrefForm()
    {
    delete setting;
    }


//-----------------------------------------------------------------------------
// set_dirty
//-----------------------------------------------------------------------------
void TSokoPrefForm::set_dirty( bool flag )
    {
    if (flag != dirty)
	{
	dirty = flag;
	AnsiString s( Caption );
	if (dirty)
	    s.Insert( "* ", 1 ); // AnsiString is 1-based.
	else
	    s.Delete( 1, 2 );    // AnsiString is 1-based.
	Caption = s;
	}
    }


//-----------------------------------------------------------------------------
// load_defaults
//-----------------------------------------------------------------------------
void TSokoPrefForm::load_defaults()
    {
    SokoPool pool;
    save_path_field->Text = soko_get_default_save_directory( pool );
    auto_save_switch->Checked = true;
    score_path_field->Text = soko_get_default_score_file( pool );
    puzzle_path_field->Text = soko_get_default_puzzle_directory( pool );
    level_field->Text = 1;
    bool const toolbar_default = true;
    if (!toolbar_switch_toggled)
	toolbar_switch_toggled = (toolbar_switch->Checked != toolbar_default);
    toolbar_switch->Checked = toolbar_default;
    }


//-----------------------------------------------------------------------------
// load_settings
//-----------------------------------------------------------------------------
void TSokoPrefForm::load_settings()
    {
    SokoPool pool;
    save_path_field->Text = soko_get_save_directory( pool );
    auto_save_switch->Checked =
	setting->get_bool( SOKO_SETTING_AUTO_SAVE, soko_true );
    score_path_field->Text = soko_get_score_file( pool );
    puzzle_path_field->Text = soko_get_puzzle_directory( pool );
    level_field->Text = setting->get_int( SOKO_SETTING_LEVEL, 1 );
    toolbar_switch->Checked =
	setting->get_bool( SOKO_SHOW_TOOLBAR, soko_true );
    toolbar_switch_toggled = false;
    set_dirty( false );
    }


//-----------------------------------------------------------------------------
// save_settings
//-----------------------------------------------------------------------------
void TSokoPrefForm::save_settings()
    {
    SokoPool pool;
    if (!save_path_field->Text.IsEmpty())
	soko_set_save_directory( pool, save_path_field->Text.c_str() );
    setting->set_bool( SOKO_SETTING_AUTO_SAVE, auto_save_switch->Checked );
    if (!score_path_field->Text.IsEmpty())
	soko_set_score_file( pool, score_path_field->Text.c_str() );
    if (!puzzle_path_field->Text.IsEmpty())
	soko_set_puzzle_directory( pool, puzzle_path_field->Text.c_str() );
    int const level = atoi( level_field->Text.c_str() );
    if (level >= 1)
	setting->set_int( SOKO_SETTING_LEVEL, level );
    setting->set_bool( SOKO_SHOW_TOOLBAR, toolbar_switch->Checked );

    if (toolbar_switch_toggled)
	TSokoBoardForm::enable_toolbars( toolbar_switch->Checked );

    // Load settings in case some settings were rejected (above).  Also clears
    // "dirty" and "toolbar_switch_toggled" flags as a side-effect.
    load_settings();
    }


//-----------------------------------------------------------------------------
// form_create
//-----------------------------------------------------------------------------
void __fastcall TSokoPrefForm::form_create( TObject* sender )
    {
    min_width = Width;
    current_width = Width;
    load_settings();
    }


//-----------------------------------------------------------------------------
// form_close
//-----------------------------------------------------------------------------
void __fastcall TSokoPrefForm::form_close(TObject* sender,TCloseAction& action)
    {
    action = caHide;
    if (dirty)
	{
	int rc = Application->MessageBox( "Save changes to your preferences?",
	    "SokoSave", MB_YESNOCANCEL | MB_ICONQUESTION );
	if (rc == IDYES)
	    save_settings();	// Clears "dirty" as side-effect.
	else if (rc == IDNO)
	    load_settings();	// Clears "dirty" as side-effect.
	else // (rc == IDCANCEL)
	    action = caNone;
	}
    }


//-----------------------------------------------------------------------------
// form_can_resize
//	Constrain the form so that the user can only resize horizontally.
//-----------------------------------------------------------------------------
void __fastcall TSokoPrefForm::form_can_resize(
    TObject* sender, int& w, int& h, bool& allow )
    {
    if (user_resizing)
	{
	h = Height;
	if (w < min_width)
	    w = min_width;
	allow = true;
	}
    }


//-----------------------------------------------------------------------------
// form_resize
//	If Windows itself resized the form (perhaps in response to a font
//	change), rather than the user resizing it, then it may be necessary to
//	update the minimum window constraint.
//-----------------------------------------------------------------------------
void __fastcall TSokoPrefForm::form_resize( TObject* sender )
    {
    if (!user_resizing && Width != current_width)
	min_width += Width - current_width;
    current_width = Width;
    }


//-----------------------------------------------------------------------------
// set_button_click
//-----------------------------------------------------------------------------
void __fastcall TSokoPrefForm::set_button_click( TObject* sender )
    {
    Visible = false;
    save_settings();
    }


//-----------------------------------------------------------------------------
// cancel_button_click
//-----------------------------------------------------------------------------
void __fastcall TSokoPrefForm::cancel_button_click( TObject* sender )
    {
    Visible = false;
    load_settings();
    }


//-----------------------------------------------------------------------------
// defaults_button_click
//-----------------------------------------------------------------------------
void __fastcall TSokoPrefForm::defaults_button_click( TObject* sender )
    {
    load_defaults();
    set_dirty( true );
    }


//-----------------------------------------------------------------------------
// toolbar_switch_click
//-----------------------------------------------------------------------------
void __fastcall TSokoPrefForm::toolbar_switch_click( TObject* sender )
    {
    toolbar_switch_toggled = true;
    set_dirty( true );
    }


//-----------------------------------------------------------------------------
// setting_changed
//-----------------------------------------------------------------------------
void __fastcall TSokoPrefForm::setting_changed( TObject* sender )
    {
    bool changed = true;
    TEdit* e = dynamic_cast<TEdit*>(sender);
    if (e != 0)
	changed = e->Modified;
    set_dirty( changed );
    }
