//-----------------------------------------------------------------------------
// SokoNewScore.cpp
//
//	Form allowing user input upon new high-score for SokoSave.
//
// Copyright (c), 2001,2002, Eric Sunshine <sunshine@sunshineco.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoNewScore.cpp,v 1.2 2002/01/29 23:09:16 sunshine Exp $
// $Log: SokoNewScore.cpp,v $
// Revision 1.2  2002/01/29 23:09:16  sunshine
// v17
// -*- Fixed a number of aesthetic problems which showed up when using larger
//     control sizes and fonts.  Controls would clobber one another and text
//     would be clipped.  The problems were particularly acute when using
//     "Large Fonts" from the Windows Display/Advanced settings.
//
// -*- Added `user_resizing' property to SokoForm which is `true' during a
//     user- initiated window resize operation.  This provides clients with
//     the capability to make constraint decisions based upon the agent which
//     is resizing a window, rather than always applying constraints,
//     regardless of whether resizing was initiated by the user or by Windows
//     (for instance, if font metrics had changed).
//
// -*- SokoNewScore controls no longer become all jumbled together if user
//     has selected large fonts.  Also, added a TPanel behind all of the
//     controls to keep them from falling off bottom of window when user
//     changes Window display scheme.
//
// Revision 1.1  2001-12-21 16:46:20-05  sunshine
// v15
// -*- Extracted core game logic out of GUI code and generalized it so that
//     the same core code can be used by any platform.  Logic from
//     SokoBoard.m now resides in SokoPuzzle.c, etc.
//
// -*- Created a native Windows port of SokoSave using C++Builder.  The
//     back-end logic for this port is provided by the new "sokocore" common
//     code which used to be merged with the UI code for the
//     OpenStep/NextStep ports.  From the user-experience, this port is
//     nearly identical to the existing OpenStep and NextStep ports.  Builds
//     successfully with C++Builder versions 4 and 5.
//
// -*- Added a shortcut toolbar for quickly starting new games, opening saved
//     games, saving games, launching the scores panel, and launching the
//     help file.  Added a toggle-switch to the preferences panel to control
//     presence of toolbar.  From the user-standpoint, this is the only major
//     difference from the OpenStep/NextStep ports, since they do not feature
//     a shortcut toolbar.
//
// -*- Created an InnoSetup installer script for SokoSave.  This scripts
//     facilitates the creation of a stand-alone "setup" program
//     (soksetup.exe) for SokoSave.  With this program, Windows users can
//     download and install SokoSave in a fashion in which they are already
//     familiar.
//-----------------------------------------------------------------------------
#include <vcl.h>
#pragma hdrstop

#include "SokoPool.h"
#include "SokoNewScore.h"
#include "SokoPuzzle.h"
#include "SokoScores.h"
#include "SokoSetting.h"
#include "SokoUtil.h"
#include <stdlib.h>

#pragma package(smart_init)
#pragma resource "*.dfm"
TSokoNewScoreForm* SokoNewScoreForm;

#define SOKO_PLAYER_NAME "PlayerName"

//-----------------------------------------------------------------------------
// default_player_name
//	Try to determine the player's name in order to seed the "Name" field
//	on the panel.  Attempts to glean the user's name from these sources:
//
//	- GetUserName() [Win32 API]
//	- $(USERNAME)
//	- $(LOGNAME)
//
//	If all sources fail, the empty string is returned, in which case the
//	user must enter his or her name manually.
//-----------------------------------------------------------------------------
AnsiString TSokoNewScoreForm::default_player_name()
    {
    AnsiString player;
    char const* s;
    char buff[ 1024 ];
    unsigned long n = sizeof(buff);
    if (GetUserName( buff, &n ))
	player = buff;
    else if ((s = getenv( "USERNAME" )) != 0)
	player = s;
    else if ((s = getenv( "LOGNAME" )) != 0)
	player = s;
    return player;
    }


//-----------------------------------------------------------------------------
// Constructor
//-----------------------------------------------------------------------------
__fastcall TSokoNewScoreForm::TSokoNewScoreForm( TComponent* owner ) :
    TSokoForm(owner), min_width(0), current_width(0)
    {
    }


//-----------------------------------------------------------------------------
// form_create
//-----------------------------------------------------------------------------
void __fastcall TSokoNewScoreForm::form_create( TObject* sender )
    {
    SokoPool pool;
    SokoSetting setting;
    stored_name = setting.get_string( SOKO_PLAYER_NAME, "", pool );
    if (stored_name.Length() != 0)
	name_field->Text = stored_name;
    else
	name_field->Text = default_player_name();
    }


//-----------------------------------------------------------------------------
// form_close
//-----------------------------------------------------------------------------
void __fastcall TSokoNewScoreForm::form_close(
    TObject* sender, TCloseAction& action )
    {
    action = caHide;
    }

//-----------------------------------------------------------------------------
// form_resize
//      If Windows itself resized the form (perhaps in response to a font
//      change), rather than the user resizing it, then it may be necessary to
//      update the minimum window constraint.
//-----------------------------------------------------------------------------
void __fastcall TSokoNewScoreForm::form_resize( TObject* sender )
    {
    if (!user_resizing && Width != current_width)
        min_width += Width - current_width;
    current_width = Width;
    }


//-----------------------------------------------------------------------------
// form_can_resize
//      Constrain the form so that the user can only resize horizontally.
//-----------------------------------------------------------------------------
void __fastcall TSokoNewScoreForm::form_can_resize(
    TObject* sender, int& w, int& h, bool& allow )
    {
    if (user_resizing)
        {
        h = Height;
        if (w < min_width)
            w = min_width;
        allow = true;
        }
    }


//-----------------------------------------------------------------------------
// refresh_controls
//-----------------------------------------------------------------------------
void TSokoNewScoreForm::refresh_controls()
    {
    AnsiString s = name_field->Text;
    bool bad_name = (s.Length() == 0);
    if (!bad_name)
	bad_name = (s.Trim().Length() == 0);
    if (bad_name == okay_button->Enabled)
	okay_button->Enabled = !bad_name;
    }


//-----------------------------------------------------------------------------
// key_typed
//-----------------------------------------------------------------------------
void __fastcall TSokoNewScoreForm::key_typed(
    TObject* sender, WORD& key, TShiftState shift )
    {
    TEdit* field = dynamic_cast<TEdit*>(sender);
    if (field == name_field)
	refresh_controls();
    }


//-----------------------------------------------------------------------------
// okay_button_click
//	By deft keyboard-mouse coordination, it is possible for the user to
//	press the OK button even when the Name field is empty (though we try
//	to protect against that), so double-check the validity of that field.
//-----------------------------------------------------------------------------
void __fastcall TSokoNewScoreForm::okay_button_click( TObject* sender )
    {
    refresh_controls(); // Ensure Name field not empty.
    if (okay_button->Enabled)
	{
	ModalResult = mrOk;
	SokoPool pool;
	char const* name = soko_collapse( pool, name_field->Text.c_str() );
	if (name_field->Text != name)
	    name_field->Text = name;
	if (stored_name != name)
	    {
	    stored_name = name;
	    SokoSetting setting;
	    setting.set_string( SOKO_PLAYER_NAME, name );
	    }
	}
    }


//-----------------------------------------------------------------------------
// populate
//-----------------------------------------------------------------------------
void TSokoNewScoreForm::populate(
    SokoPuzzle& p, int moves, int pushes, int runs, int focus )
    {
    SokoPool pool;
    puzzle_field->Caption = soko_remove_path_extension(
	pool, soko_filename_part( pool, p.get_puzzle_file_name(pool) ));
    moves_field->Caption = moves;
    pushes_field->Caption = pushes;
    runs_field->Caption = runs;
    focus_field->Caption = focus;
    note_field->Text = "";
    refresh_controls();
    ActiveControl = name_field;
    }


//-----------------------------------------------------------------------------
// run_modal
//-----------------------------------------------------------------------------
void TSokoNewScoreForm::run_modal(
    SokoPuzzle& p, int moves, int pushes, int runs, int focus )
    {
    populate( p, moves, pushes, runs, focus );
    if (ShowModal() == mrOk)
	TSokoScoresForm::record_score( p, moves, pushes, runs, focus,
	    name_field->Text.c_str(), note_field->Text.c_str() );
    }


//-----------------------------------------------------------------------------
// record_score
//-----------------------------------------------------------------------------
void TSokoNewScoreForm::record_score(
    SokoPuzzle& p, int moves, int pushes, int runs, int focus )
    {
    if (SokoNewScoreForm == 0)
	SokoNewScoreForm = new TSokoNewScoreForm( Application );
    SokoNewScoreForm->run_modal( p, moves, pushes, runs, focus );
    }
