//-----------------------------------------------------------------------------
// SokoGridTriangle.cpp
//
//	A triangular-celled grid for SokoSave.
//
// Copyright (c), 2002, Eric Sunshine <sunshine@sunshineco.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoGridTriangle.cpp,v 1.2 2002/05/06 05:49:00 sunshine Exp $
// $Log: SokoGridTriangle.cpp,v $
// Revision 1.2  2002/05/06 05:49:00  sunshine
// v19
// -*- Fixed v17 Win9x problem where animation of Hexoban and Trioban puzzles
//     involved unsightly "flashing" which resulted from Windows first
//     painting the on-screen hexagonal and triangular tiles black before
//     blitting in the actual tile.  To work around the problem,
//     SokoGridHexagon and SokoGridTriangle now maintain a backing-store in
//     which the puzzle image is rendered.  When the on-screen image needs to
//     be repainted, only non-transparent rectangles are blitted to the
//     screen.  This patch cuts animation speed effectively in half for
//     Hexoban and Trioban puzzles, but the visual quality of the animation
//     is much improved.  This problem did not affect WinNT or Win2000.
//
// Revision 1.1  2002/02/19 07:57:24  sunshine
// v18
// -*- Added support for new triangular-style Trioban puzzles.
//
// -*- Consolidated all of the geometry-related utility code into the new
//     SokoGeometry facility.  This mechanism provides hit-testing and
//     coverage-testing facilities for square-, hexagonal-, and
//     triangular-tiled grids.  All high-level GUI code now utilizes these
//     facilities rather than relying upon their own copy/paste
//     implementations.
//-----------------------------------------------------------------------------
#include "SokoPool.h"
#include "SokoGridTriangle.h"
#include "SokoGeometry.h"
#include <controls.hpp>

#define CELL_WIDTH      (24)
#define CELL_HEIGHT     (22)
#define CELL_HALF_WIDTH (CELL_WIDTH / 2)

static Graphics::TBitmap* IMAGES[ SOKO_CELL_MAX * 2 ];
static char const* const IMAGE_NAMES[ SOKO_CELL_MAX * 2 ] =
    {
    "SokoEmptyTriS",		 // SOKO_CELL_EMPTY
    "SokoEmptyTriN",		 // SOKO_CELL_EMPTY
    "SokoEmptySafeTriS",	 // SOKO_CELL_EMPTY_SAFE
    "SokoEmptySafeTriN",	 // SOKO_CELL_EMPTY_SAFE
    "SokoWallTriS",		 // SOKO_CELL_WALL
    "SokoWallTriN",		 // SOKO_CELL_WALL
    "SokoPlayerTriS",		 // SOKO_CELL_PLAYER
    "SokoPlayerTriN",		 // SOKO_CELL_PLAYER
    "SokoPlayerSafeTriS",	 // SOKO_CELL_PLAYER_SAFE
    "SokoPlayerSafeTriN",	 // SOKO_CELL_PLAYER_SAFE
    "SokoCrateTriS",		 // SOKO_CELL_CRATE
    "SokoCrateTriN",		 // SOKO_CELL_CRATE
    "SokoCrateSafeTriS",	 // SOKO_CELL_CRATE_SAFE
    "SokoCrateSafeTriN",	 // SOKO_CELL_CRATE_SAFE
    "SokoCrateSelectedTriS",	 // SOKO_CELL_CRATE_SELECTED
    "SokoCrateSelectedTriN",	 // SOKO_CELL_CRATE_SELECTED
    "SokoCrateSafeSelectedTriS", // SOKO_CELL_CRATE_SELECTED_SAFE
    "SokoCrateSafeSelectedTriN", // SOKO_CELL_CRATE_SELECTED_SAFE
    0,				 // SOKO_CELL_NULL
    0                            // SOKO_CELL_NULL
    };


//-----------------------------------------------------------------------------
// load_images
//-----------------------------------------------------------------------------
static void load_images()
    {
    static bool loaded = false;
    if (!loaded)
	{
	loaded = true;
	for (int i = 0; i < SOKO_CELL_MAX * 2; i++)
	    {
	    char const* const s = IMAGE_NAMES[i];
	    if (s == 0)
		IMAGES[i] = 0;
	    else
		{
		Graphics::TBitmap* b = new Graphics::TBitmap;
		b->LoadFromResourceName( (int)HInstance, s );
		b->Transparent = true;
		IMAGES[i] = b;
		}
	    }
	}
    }


//-----------------------------------------------------------------------------
// Constructor
//-----------------------------------------------------------------------------
__fastcall TSokoGridTriangle::TSokoGridTriangle(TComponent* o, SokoPuzzle* p) :
    superclass( o, p ), cache(0)
    {
    SetBounds( 0, 0, (cols + 1) * CELL_HALF_WIDTH, rows * CELL_HEIGHT );
    load_images();
    }


//-----------------------------------------------------------------------------
// Destructor
//-----------------------------------------------------------------------------
__fastcall TSokoGridTriangle::~TSokoGridTriangle()
    {
    if (cache != 0)
	delete cache;
    }


//-----------------------------------------------------------------------------
// create_cache
//-----------------------------------------------------------------------------
void TSokoGridTriangle::create_cache()
    {
    if (cache == 0)
	{
	TRect b;
	b.Top = 0;
	b.Left = 0;
	b.Bottom = Height;
	b.Right = Width;
	cache = new Graphics::TBitmap;
	cache->Transparent = false;
	cache->Width = Width;
	cache->Height = Height;
	cache->Canvas->Brush->Color = Color;
	cache->Canvas->FillRect(b);
	for (int r = 0; r < rows; r++)
	    for (int c = 0; c < cols; c++)
		refresh_cache_cell( r, c, puzzle->cell_type(r, c) );
	}
    }


//-----------------------------------------------------------------------------
// refresh_cache_cell
//-----------------------------------------------------------------------------
void TSokoGridTriangle::refresh_cache_cell( int r, int c, SokoCellType type )
    {
    if (type != SOKO_CELL_NULL)
	{
	int const north_south = (r & 1) ^ (c & 1);
	cache->Canvas->Draw( c * CELL_HALF_WIDTH, r * CELL_HEIGHT,
	    IMAGES[ type * 2 + north_south ] );
	}
    }


//-----------------------------------------------------------------------------
// note_color_change
//-----------------------------------------------------------------------------
void TSokoGridTriangle::note_color_change()
    {
    superclass::note_color_change();
    if (cache != 0)
	{
	delete cache;
	cache = 0;
	}
    }


//-----------------------------------------------------------------------------
// Paint
//-----------------------------------------------------------------------------
void __fastcall TSokoGridTriangle::Paint()
    {
    create_cache();
    Canvas->Draw( 0, 0, cache );
    }


//-----------------------------------------------------------------------------
// draw_cell
//-----------------------------------------------------------------------------
void TSokoGridTriangle::draw_cell( int r, int c, SokoCellType type )
    {
    TRect z;
    z.Left   = c * CELL_HALF_WIDTH;
    z.Top    = r * CELL_HEIGHT;
    z.Right  = z.Left + CELL_WIDTH;
    z.Bottom = z.Top + CELL_HEIGHT;
    create_cache();
    refresh_cache_cell( r, c, type );
    Canvas->CopyRect( z, cache->Canvas, z );
    }


//-----------------------------------------------------------------------------
// mouse_to_cell
//-----------------------------------------------------------------------------
bool TSokoGridTriangle::mouse_to_cell( int x, int y, int& r, int& c )
    {
    return soko_hit_test_triangle( x, y, rows, cols, CELL_WIDTH, CELL_HEIGHT,
	CELL_HALF_WIDTH, 0, 0, &r, &c );
    }
