//-----------------------------------------------------------------------------
// SokoGridSquare.cpp
//
//	A square-celled grid for SokoSave.
//
// Copyright (c), 2002, Eric Sunshine <sunshine@sunshineco.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoGridSquare.cpp,v 1.2 2002/02/19 07:57:30 sunshine Exp $
// $Log: SokoGridSquare.cpp,v $
// Revision 1.2  2002/02/19 07:57:30  sunshine
// v18
// -*- Added support for new triangular-style Trioban puzzles.
//
// -*- Consolidated all of the geometry-related utility code into the new
//     SokoGeometry facility.  This mechanism provides hit-testing and
//     coverage-testing facilities for square-, hexagonal-, and
//     triangular-tiled grids.  All high-level GUI code now utilizes these
//     facilities rather than relying upon their own copy/paste
//     implementations.
//
// Revision 1.1  2002-01-29 17:13:40-05  sunshine
// v17
// -*- Added support for the new hexagonal-style puzzles.
//
// -*- No longer uses TDrawGrid for display of puzzle.
//
// -*- Added TSokoGrid, a subclass of TGraphicControl which is the base class
//     for grids which display puzzle boards.
//
// -*- Added TSokoGridSquare, a subclass of TSokoGrid, which knows how to
//     draw square-tiled puzzles, and perform appropriate hit-testing.
//
// -*- Added TSokoGridHexagon, a subclass of TSokoGrid, which knows how to
//     draw hexagon-tiled puzzles, and perform appropriate hit-testing.
//-----------------------------------------------------------------------------
#include "SokoPool.h"
#include "SokoGridSquare.h"
#include "SokoGeometry.h"
#include <controls.hpp>

#define CELL_WIDTH  (16)
#define CELL_HEIGHT (16)

static Graphics::TBitmap* IMAGES[ SOKO_CELL_MAX ];
static char const* const IMAGE_NAMES[ SOKO_CELL_MAX ] =
    {
    "SokoEmpty",		// SOKO_CELL_EMPTY
    "SokoEmptySafe",		// SOKO_CELL_EMPTY_SAFE
    "SokoWall",			// SOKO_CELL_WALL
    "SokoPlayer",		// SOKO_CELL_PLAYER
    "SokoPlayerSafe",		// SOKO_CELL_PLAYER_SAFE
    "SokoCrate",		// SOKO_CELL_CRATE
    "SokoCrateSafe",		// SOKO_CELL_CREATE_SAFE
    "SokoCrateSelected",	// SOKO_CELL_CRATE_SELECTED
    "SokoCrateSafeSelected",	// SOKO_CELL_CRATE_SELECTED_SAFE
    0				// SOKO_CELL_NULL
    };


//-----------------------------------------------------------------------------
// load_images
//-----------------------------------------------------------------------------
static void load_images()
    {
    static bool loaded = false;
    if (!loaded)
	{
	loaded = true;
	for (int i = 0; i < SOKO_CELL_MAX; i++)
	    {
	    char const* const s = IMAGE_NAMES[i];
	    if (s == 0)
		IMAGES[i] = 0;
	    else
		{
		Graphics::TBitmap* b = new Graphics::TBitmap;
		b->LoadFromResourceName( (int)HInstance, s );
		IMAGES[i] = b;
		}
	    }
	}
    }


//-----------------------------------------------------------------------------
// Constructor
//-----------------------------------------------------------------------------
__fastcall TSokoGridSquare::TSokoGridSquare( TComponent* o, SokoPuzzle* p ) :
    superclass( o, p )
    {
    SetBounds( 0, 0, cols * CELL_WIDTH, rows * CELL_HEIGHT );
    load_images();
    }


//-----------------------------------------------------------------------------
// Paint
//-----------------------------------------------------------------------------
void __fastcall TSokoGridSquare::Paint()
    {
    for (int r = 0; r < rows; r++)
	for (int c = 0; c < cols; c++)
	    draw_cell( r, c, puzzle->cell_type(r, c) );
    }


//-----------------------------------------------------------------------------
// draw_cell
//-----------------------------------------------------------------------------
void TSokoGridSquare::draw_cell( int r, int c, SokoCellType type )
    {
    if (type != SOKO_CELL_NULL)
	Canvas->Draw( c * CELL_WIDTH, r * CELL_HEIGHT, IMAGES[type] );
    }


//-----------------------------------------------------------------------------
// mouse_to_cell
//-----------------------------------------------------------------------------
bool TSokoGridSquare::mouse_to_cell( int x, int y, int& r, int& c )
    {
    return soko_hit_test_square(
	x, y, rows, cols, CELL_WIDTH, CELL_HEIGHT, 0, 0, &r, &c );
    }
