//-----------------------------------------------------------------------------
// SokoGridHexagon.cpp
//
//	A hexagonal-celled grid for SokoSave.
//
// Copyright (c), 2002, Eric Sunshine <sunshine@sunshineco.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoGridHexagon.cpp,v 1.3 2002/05/06 05:49:11 sunshine Exp $
// $Log: SokoGridHexagon.cpp,v $
// Revision 1.3  2002/05/06 05:49:11  sunshine
// v19
// -*- Fixed v17 Win9x problem where animation of Hexoban and Trioban puzzles
//     involved unsightly "flashing" which resulted from Windows first
//     painting the on-screen hexagonal and triangular tiles black before
//     blitting in the actual tile.  To work around the problem,
//     SokoGridHexagon and SokoGridTriangle now maintain a backing-store in
//     which the puzzle image is rendered.  When the on-screen image needs to
//     be repainted, only non-transparent rectangles are blitted to the
//     screen.  This patch cuts animation speed effectively in half for
//     Hexoban and Trioban puzzles, but the visual quality of the animation
//     is much improved.  This problem did not affect WinNT or Win2000.
//
// Revision 1.2  2002/02/19 07:57:35  sunshine
// v18
// -*- Added support for new triangular-style Trioban puzzles.
//
// -*- Consolidated all of the geometry-related utility code into the new
//     SokoGeometry facility.  This mechanism provides hit-testing and
//     coverage-testing facilities for square-, hexagonal-, and
//     triangular-tiled grids.  All high-level GUI code now utilizes these
//     facilities rather than relying upon their own copy/paste
//     implementations.
//
// Revision 1.1  2002-01-29 17:13:44-05  sunshine
// v17
// -*- Added support for the new hexagonal-style puzzles.
//
// -*- No longer uses TDrawGrid for display of puzzle.
//
// -*- Added TSokoGrid, a subclass of TGraphicControl which is the base class
//     for grids which display puzzle boards.
//
// -*- Added TSokoGridSquare, a subclass of TSokoGrid, which knows how to
//     draw square-tiled puzzles, and perform appropriate hit-testing.
//
// -*- Added TSokoGridHexagon, a subclass of TSokoGrid, which knows how to
//     draw hexagon-tiled puzzles, and perform appropriate hit-testing.
//-----------------------------------------------------------------------------
#include "SokoPool.h"
#include "SokoGridHexagon.h"
#include "SokoGeometry.h"
#include <controls.hpp>

#define CELL_WIDTH   (18)
#define CELL_HEIGHT  (22)
#define CELL_X_INSET (9)
#define CELL_Y_INSET (6)

static Graphics::TBitmap* IMAGES[ SOKO_CELL_MAX ];
static char const* const IMAGE_NAMES[ SOKO_CELL_MAX ] =
    {
    "SokoEmptyHex",		// SOKO_CELL_EMPTY
    "SokoEmptySafeHex",		// SOKO_CELL_EMPTY_SAFE
    "SokoWallHex",		// SOKO_CELL_WALL
    "SokoPlayerHex",		// SOKO_CELL_PLAYER
    "SokoPlayerSafeHex",	// SOKO_CELL_PLAYER_SAFE
    "SokoCrateHex",		// SOKO_CELL_CRATE
    "SokoCrateSafeHex",		// SOKO_CELL_CRATE_SAFE
    "SokoCrateSelectedHex",	// SOKO_CELL_CRATE_SELECTED
    "SokoCrateSafeSelectedHex",	// SOKO_CELL_CRATE_SELECTED_SAFE
    0				// SOKO_CELL_NULL
    };


//-----------------------------------------------------------------------------
// load_images
//-----------------------------------------------------------------------------
static void load_images()
    {
    static bool loaded = false;
    if (!loaded)
	{
	loaded = true;
	for (int i = 0; i < SOKO_CELL_MAX; i++)
	    {
	    char const* const s = IMAGE_NAMES[i];
	    if (s == 0)
		IMAGES[i] = 0;
	    else
		{
		Graphics::TBitmap* b = new Graphics::TBitmap;
		b->LoadFromResourceName( (int)HInstance, s );
		b->Transparent = true;
		IMAGES[i] = b;
		}
	    }
	}
    }


//-----------------------------------------------------------------------------
// Constructor
//-----------------------------------------------------------------------------
__fastcall TSokoGridHexagon::TSokoGridHexagon( TComponent* o, SokoPuzzle* p ) :
    superclass( o, p ), cache(0)
    {
    SetBounds( 0, 0,
	(cols * CELL_WIDTH) + (rows <= 1 ? 0 : CELL_X_INSET),
	 rows * (CELL_HEIGHT - CELL_Y_INSET) + CELL_Y_INSET );
    load_images();
    }


//-----------------------------------------------------------------------------
// Destructor
//-----------------------------------------------------------------------------
__fastcall TSokoGridHexagon::~TSokoGridHexagon()
    {
    if (cache != 0)
	delete cache;
    }


//-----------------------------------------------------------------------------
// create_cache
//-----------------------------------------------------------------------------
void TSokoGridHexagon::create_cache()
    {
    if (cache == 0)
	{
	TRect b;
	b.Top = 0;
	b.Left = 0;
	b.Bottom = Height;
	b.Right = Width;
	cache = new Graphics::TBitmap;
	cache->Transparent = false;
	cache->Width = Width;
	cache->Height = Height;
	cache->Canvas->Brush->Color = Color;
	cache->Canvas->FillRect(b);
	for (int r = 0; r < rows; r++)
	    for (int c = 0; c < cols; c++)
		refresh_cache_cell( r, c, puzzle->cell_type(r, c) );
	}
    }


//-----------------------------------------------------------------------------
// refresh_cache_cell
//-----------------------------------------------------------------------------
void TSokoGridHexagon::refresh_cache_cell( int r, int c, SokoCellType type )
    {
    if (type != SOKO_CELL_NULL)
	{
	int const x = c * CELL_WIDTH + ((r & 1) == 0 ? 0 : CELL_X_INSET);
	int const y = r * (CELL_HEIGHT - CELL_Y_INSET);
	cache->Canvas->Draw( x, y, IMAGES[type] );
	}
    }


//-----------------------------------------------------------------------------
// note_color_change
//-----------------------------------------------------------------------------
void TSokoGridHexagon::note_color_change()
    {
    superclass::note_color_change();
    if (cache != 0)
	{
	delete cache;
	cache = 0;
	}
    }


//-----------------------------------------------------------------------------
// Paint
//-----------------------------------------------------------------------------
void __fastcall TSokoGridHexagon::Paint()
    {
    create_cache();
    Canvas->Draw( 0, 0, cache );
    }


//-----------------------------------------------------------------------------
// draw_cell
//-----------------------------------------------------------------------------
void TSokoGridHexagon::draw_cell( int r, int c, SokoCellType type )
    {
    TRect z;
    z.Left   = c * CELL_WIDTH + ((r & 1) == 0 ? 0 : CELL_X_INSET);
    z.Top    = r * (CELL_HEIGHT - CELL_Y_INSET);
    z.Right  = z.Left + CELL_WIDTH;
    z.Bottom = z.Top + CELL_HEIGHT;
    create_cache();
    refresh_cache_cell( r, c, type );
    Canvas->CopyRect( z, cache->Canvas, z );
    }


//-----------------------------------------------------------------------------
// mouse_to_cell
//-----------------------------------------------------------------------------
bool TSokoGridHexagon::mouse_to_cell( int x, int y, int& r, int& c )
    {
    return soko_hit_test_hexagon( x, y, rows, cols, CELL_WIDTH, CELL_HEIGHT,
	CELL_X_INSET, CELL_Y_INSET, 0, 0, &r, &c );
    }
