//-----------------------------------------------------------------------------
// SokoForm.cpp
//
//	Base class of all forms.  Defines behavior common to all forms in
//	SokoSave.
//
// Copyright (c), 2001,2002, Eric Sunshine <sunshine@sunshineco.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoForm.cpp,v 1.2 2002/01/29 21:28:03 sunshine Exp $
// $Log: SokoForm.cpp,v $
// Revision 1.2  2002/01/29 21:28:03  sunshine
// v17
// -*- Fixed a number of aesthetic problems which showed up when using larger
//     control sizes and fonts.  Controls would clobber one another and text
//     would be clipped.  The problems were particularly acute when using
//     "Large Fonts" from the Windows Display/Advanced settings.
//
// -*- Added `user_resizing' property to SokoForm which is `true' during a
//     user- initiated window resize operation.  This provides clients with
//     the capability to make constraint decisions based upon the agent which
//     is resizing a window, rather than always applying constraints,
//     regardless of whether resizing was initiated by the user or by Windows
//     (for instance, if font metrics had changed).
//
// Revision 1.1  2001-12-21 16:46:45-05  sunshine
// v15
// -*- Extracted core game logic out of GUI code and generalized it so that
//     the same core code can be used by any platform.  Logic from
//     SokoBoard.m now resides in SokoPuzzle.c, etc.
//
// -*- Created a native Windows port of SokoSave using C++Builder.  The
//     back-end logic for this port is provided by the new "sokocore" common
//     code which used to be merged with the UI code for the
//     OpenStep/NextStep ports.  From the user-experience, this port is
//     nearly identical to the existing OpenStep and NextStep ports.  Builds
//     successfully with C++Builder versions 4 and 5.
//
// -*- Added a shortcut toolbar for quickly starting new games, opening saved
//     games, saving games, launching the scores panel, and launching the
//     help file.  Added a toggle-switch to the preferences panel to control
//     presence of toolbar.  From the user-standpoint, this is the only major
//     difference from the OpenStep/NextStep ports, since they do not feature
//     a shortcut toolbar.
//
// -*- Created an InnoSetup installer script for SokoSave.  This scripts
//     facilitates the creation of a stand-alone "setup" program
//     (soksetup.exe) for SokoSave.  With this program, Windows users can
//     download and install SokoSave in a fashion in which they are already
//     familiar.
//-----------------------------------------------------------------------------
#include "SokoForm.h"
#include <stdlib.h>

static bool CASCADE_READY     = false;
static int  CASCADE_X_COUNT   = 0;
static int  CASCADE_Y_COUNT   = 0;
static int  CASCADE_BASE_TOP  = 0;
static int  CASCADE_BASE_LEFT = 0;

//-----------------------------------------------------------------------------
// SokoFormMenuResponder
//	Helper class for responding to items programmatically added to the
//	"Window" menu.
//-----------------------------------------------------------------------------
class SokoFormMenuResponder : public TObject
    {
public:
    void __fastcall on_click( TObject* sender );
    };

void __fastcall SokoFormMenuResponder::on_click( TObject* sender )
    {
    TMenuItem* item = dynamic_cast<TMenuItem*>(sender);
    if (item != 0 && item->Tag != 0)
	{
	TForm* w = (TForm*)item->Tag;
	if (w->WindowState == wsMinimized)
	    ShowWindow( w->Handle, SW_RESTORE );
	w->BringToFront();
	}
    }


//-----------------------------------------------------------------------------
// reset_cascade
//-----------------------------------------------------------------------------
static void reset_cascade()
    {
    CASCADE_X_COUNT = 0;
    CASCADE_Y_COUNT = 0;
    }


//-----------------------------------------------------------------------------
// init_cascade
//-----------------------------------------------------------------------------
static void init_cascade()
    {
    if (!CASCADE_READY)
	{
	CASCADE_READY = true;
	CASCADE_BASE_TOP  = Screen->Height / 20;
	CASCADE_BASE_LEFT = Screen->Width  / 10;
	reset_cascade();
	}
    }


//-----------------------------------------------------------------------------
// cascade_window
//-----------------------------------------------------------------------------
void TSokoForm::cascade_window( TForm* w )
    {
    init_cascade();
    w->Top  = CASCADE_BASE_TOP  + CASCADE_Y_COUNT * 20 + CASCADE_X_COUNT * 10;
    w->Left = CASCADE_BASE_LEFT + CASCADE_Y_COUNT * 20 + CASCADE_X_COUNT * 40;
    if (++CASCADE_Y_COUNT == 10)
	{
	CASCADE_Y_COUNT = 0;
	if (++CASCADE_X_COUNT == 10)
	    CASCADE_X_COUNT = 0;
	}
    TSokoForm* sw = dynamic_cast<TSokoForm*>(w);
    if (sw != 0)
	sw->cascaded = true;
    }


//-----------------------------------------------------------------------------
// set_window_animation
//	Returns old animation setting.  Provides a mechanism for cascade_all()
//	to disable the animation which occurs when windows are de-miniaturized
//	since animation is both slow and annoying (especially when applied to
//	numerous windows in succession).
//-----------------------------------------------------------------------------
static bool set_window_animation( bool animate )
    {
    bool old_animate = true;
    ANIMATIONINFO info;
    UINT const sz = sizeof(info);
    info.cbSize = sz;
    if (SystemParametersInfo( SPI_GETANIMATION, sz, &info, 0 ) != 0)
	old_animate = (info.iMinAnimate != 0);
    if (animate != old_animate)
	{
	info.iMinAnimate = animate;
	SystemParametersInfo( SPI_SETANIMATION, sz, &info, 0);
	}
    return old_animate;
    }


//-----------------------------------------------------------------------------
// title_cmp
//-----------------------------------------------------------------------------
static int title_cmp( void const* p1, void const* p2 )
    {
    TForm* w1 = *(TForm**)p1;
    TForm* w2 = *(TForm**)p2;
    return w1->Caption.AnsiCompareIC( w2->Caption );
    }


//-----------------------------------------------------------------------------
// cascade_all
//-----------------------------------------------------------------------------
void TSokoForm::cascade_all()
    {
    int nforms = 0;
    int const screen_forms = Screen->FormCount;
    TForm** forms = new TForm*[ screen_forms ];
    for (int i = 0; i < screen_forms; i++)
	{
	TSokoForm* form = dynamic_cast<TSokoForm*>(Screen->Forms[i]);
	if (form != 0 && form->Visible && form->should_cascade)
	    forms[ nforms++ ] = form;
	}
    if (nforms != 0)
	{
	bool const old_animate = set_window_animation( false );
	reset_cascade();
	if (nforms > 1)
	    qsort( forms, nforms, sizeof(forms[0]), title_cmp );
	for (int i = nforms - 1; i >= 0; i--)
	    {
	    TForm* form = forms[i];
	    cascade_window( form );
	    if (form->WindowState == wsMinimized)
 		ShowWindow( form->Handle, SW_SHOWNORMAL );
	    form->BringToFront();
	    }
	set_window_animation( old_animate );
	}
    delete[] forms;
    }


//-----------------------------------------------------------------------------
// register_window
//-----------------------------------------------------------------------------
void TSokoForm::register_window( TMenuItem* menu, TForm* w )
    {
    static SokoFormMenuResponder* responder = 0;
    if (responder == 0)
	responder = new SokoFormMenuResponder;

    AnsiString name( w->Caption );
    TMenuItem* item = new TMenuItem(0);
    item->Caption = name;
    item->Tag = (int)w;
    item->OnClick = responder->on_click;

    int i;
    for (i = menu->Count - 1; i >= 0; i--)
	{
	AnsiString s = menu->Items[i]->Caption;
	if (s == "-" || s.AnsiCompareIC( name ) < 0)
	    break;
	}
    menu->Insert( i + 1, item );
    }


//-----------------------------------------------------------------------------
// register_window
//-----------------------------------------------------------------------------
void TSokoForm::register_window( TForm* w )
    {
    for (int i = Screen->FormCount - 1; i >= 0; i--)
	{
	TSokoForm* f = dynamic_cast<TSokoForm*>(Screen->Forms[i]);
	if (f != 0 && f->f_window_menu != 0)
	    register_window( f->f_window_menu, w );
	}
    }


//-----------------------------------------------------------------------------
// deregister_window
//-----------------------------------------------------------------------------
void TSokoForm::deregister_window( TForm* w )
    {
    for (int i = Screen->FormCount - 1; i >= 0; i--)
	{
	TSokoForm* f = dynamic_cast<TSokoForm*>(Screen->Forms[i]);
	if (f != 0 && f->f_window_menu != 0)
	    {
	    for (int j = f->f_window_menu->Count - 1; j >= 0; j--)
		{
		TMenuItem* item = f->f_window_menu->Items[j];
		if (item->Tag == (int)w)
		    {
		    f->f_window_menu->Remove( item );
		    delete item;
		    break;
		    }
		}
	    }
	}
    }


//-----------------------------------------------------------------------------
// get_caption
//-----------------------------------------------------------------------------
AnsiString TSokoForm::get_caption()
    {
    return superclass::Caption;
    }


//-----------------------------------------------------------------------------
// set_caption
//-----------------------------------------------------------------------------
void TSokoForm::set_caption( AnsiString s )
    {
    AnsiString old( superclass::Caption );
    superclass::Caption = s;
    if (f_can_publish && Visible && s != old)
	{
	deregister_window( this );
	register_window( this );
	}
    }


//-----------------------------------------------------------------------------
// DoShow
//
// *SW_HIDE*
//	TSokoForm windows have their parent handle set to the desktop window
//	rather than the global application handle.  See CreateParams() for an
//	explanation of why this is necessary.  Unfortunately, this has the bad
//	side-effect that TApplication::UpdateVisible() annoyingly makes the
//	global application taskbar button visible at a number of undesirable
//	times, including certain times when TSokoForm windows are shown and
//	hidden.  Since we only want the global taskbar button visible when the
//	application is actually "hidden" (see "Minimize All" in SokoApp), but
//	not at any other time, we need to manually re-hide the global taskbar
//	button after TApplication has made it visible.
//-----------------------------------------------------------------------------
void __fastcall TSokoForm::DoShow()
    {
    if (!cascaded && f_should_cascade)
	cascade_window( this );
    if (f_can_publish)
	register_window( this );
    superclass::DoShow();
    ShowWindow( Application->Handle, SW_HIDE );		// *SW_HIDE*
    }


//-----------------------------------------------------------------------------
// DoHide
//-----------------------------------------------------------------------------
void __fastcall TSokoForm::DoHide()
    {
    if (f_can_publish)
	deregister_window( this );
    superclass::DoHide();
    ShowWindow( Application->Handle, SW_HIDE );		// *SW_HIDE*
    }


//-----------------------------------------------------------------------------
// set_can_publish
//-----------------------------------------------------------------------------
void TSokoForm::set_can_publish( bool b )
    {
    if (b != f_can_publish)
	{
	f_can_publish = b;
	if (!Visible && f_can_publish)
	    register_window( this );
	else if (Visible && !f_can_publish)
	    deregister_window( this );
	}
    }


//-----------------------------------------------------------------------------
// set_window_menu
//-----------------------------------------------------------------------------
void TSokoForm::set_window_menu( TMenuItem* m )
    {
    f_window_menu = m;
    if (f_window_menu != 0)
	{
	for (int i = Screen->FormCount - 1; i >= 0; i--)
	    {
	    TSokoForm* f = dynamic_cast<TSokoForm*>(Screen->Forms[i]);
	    if (f != 0 && f->f_can_publish && f->Visible)
		register_window( f_window_menu, f );
	    }
	}
    }


//-----------------------------------------------------------------------------
// handle_shortcut
//	By default, TSokoForm will handle the following shortcuts, provided
//	that the form's OnShortCut event handler has not been changed, and
//	provided that the form does not sport a menu:
//
//	Shift+Ctrl+C = Cascade
//	Ctrl+M = Minimze
//	Ctrl+H = Minimize All
//	Ctrl+Q = Quit
//	Ctrl+W = Close
//-----------------------------------------------------------------------------
void __fastcall TSokoForm::handle_shortcut(Messages::TWMKey& k, bool& handled)
    {
    if (Menu == 0 && GetKeyState(VK_CONTROL) < 0)
	{
	Word const c = k.CharCode;
	if (GetKeyState(VK_SHIFT) < 0)
	    {
	    if (c == 'C')
		{
		cascade_all();
		handled = true;
		}
	    }
	else // Not shifted.
	    {
	    if (c == 'M')
		{
		Screen->ActiveForm->WindowState = wsMinimized;
		handled = true;
		}
	    else if (c == 'H')
		{
		Application->Minimize();
		handled = true;
		}
	    else if (c == 'Q')
		{
		Application->Terminate();
		handled = true;
		}
	    else if (c == 'W')
		{
		Screen->ActiveForm->Close();
		handled = true;
		}
	    }
	}
    }


//-----------------------------------------------------------------------------
// wm_enter_sizemove
//-----------------------------------------------------------------------------
void TSokoForm::wm_enter_sizemove( Messages::TMessage& m )
    {
    f_user_resizing = true;
    superclass::Dispatch(&m);
    }


//-----------------------------------------------------------------------------
// wm_exit_sizemove
//-----------------------------------------------------------------------------
void TSokoForm::wm_exit_sizemove( Messages::TMessage& m )
    {
    f_user_resizing = false;
    superclass::Dispatch(&m);
    }


//-----------------------------------------------------------------------------
// CreateParams
//	The default behavior of VCL forms when minimized is to minimize to the
//	desktop rather than minimizing to the taskbar.  This is ugly,
//	obnoxious, and non-intuitive.  Furthermore, by default, even when
//	visible, VCL forms are not represented by buttons on the taskbar.
//	Adjusting the window's ExStyle forces both a taskbar button and
//	minimization to the taskbar.
//
//	Additionally, VCL assigns the application's handle as the parent of all
//	windows.  This has the unfortunate side-effect of causing all windows
//	to be ordered-front any time the application is activated even though
//	the user clicked on only a single window.  Setting the window's
//	WndParent to the desktop window eliminates this obnoxious behavior.
//-----------------------------------------------------------------------------
void __fastcall TSokoForm::CreateParams( TCreateParams& p )
    {
    superclass::CreateParams(p);
    p.ExStyle |= WS_EX_APPWINDOW;
    p.WndParent = GetDesktopWindow();
    }


//-----------------------------------------------------------------------------
// Constructor
//-----------------------------------------------------------------------------
__fastcall TSokoForm::TSokoForm( TComponent* owner ) :
    superclass(owner),
    cascaded(false),
    f_should_cascade(false),
    f_can_publish(true),
    f_user_resizing(false),
    f_window_menu(0)
    {
    OnShortCut = handle_shortcut;
    }
