//-----------------------------------------------------------------------------
// SokoDefaults.cpp
//
//	Concrete implementation of abstract SokoSettingDelegate which
//	provides SokoSetting module with required implementation-specific
//	functionality.
//
// Copyright (c), 2001, Eric Sunshine <sunshine@sunshineco.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoDefaults.cpp,v 1.1 2001/12/21 21:46:47 sunshine Exp $
// $Log: SokoDefaults.cpp,v $
// Revision 1.1  2001/12/21 21:46:47  sunshine
// v15
// -*- Extracted core game logic out of GUI code and generalized it so that
//     the same core code can be used by any platform.  Logic from
//     SokoBoard.m now resides in SokoPuzzle.c, etc.
//
// -*- Created a native Windows port of SokoSave using C++Builder.  The
//     back-end logic for this port is provided by the new "sokocore" common
//     code which used to be merged with the UI code for the
//     OpenStep/NextStep ports.  From the user-experience, this port is
//     nearly identical to the existing OpenStep and NextStep ports.  Builds
//     successfully with C++Builder versions 4 and 5.
//
// -*- Added a shortcut toolbar for quickly starting new games, opening saved
//     games, saving games, launching the scores panel, and launching the
//     help file.  Added a toggle-switch to the preferences panel to control
//     presence of toolbar.  From the user-standpoint, this is the only major
//     difference from the OpenStep/NextStep ports, since they do not feature
//     a shortcut toolbar.
//
// -*- Created an InnoSetup installer script for SokoSave.  This scripts
//     facilitates the creation of a stand-alone "setup" program
//     (soksetup.exe) for SokoSave.  With this program, Windows users can
//     download and install SokoSave in a fashion in which they are already
//     familiar.
//
//-----------------------------------------------------------------------------
#include "SokoPool.h"
#include "SokoDefaults.h"
#include <vcl/vcl.h>
#include <vcl/registry.hpp>
#include <shlobj.h>
#include <stdlib.h>

static char const SOKO_ROOT[] = "Software\\HighSpeedSoftware\\SokoSave";

static SokoDefaults* DELEGATE = 0;

//-----------------------------------------------------------------------------
// SokoRegistry
//	TRegistry helper class which ensures that key is closed and TRegistry
//	is destroyed.
//-----------------------------------------------------------------------------
class SokoRegistry
    {
public:
    TRegistry* registry;
    SokoRegistry( bool read_only );
    ~SokoRegistry();
    bool ok() const { return (registry != 0); }
    };

SokoRegistry::SokoRegistry( bool read_only ) : registry(0)
    {
    bool ok;
    TRegistry* r = new TRegistry();
    if (read_only)
	ok = r->OpenKeyReadOnly( SOKO_ROOT );
    else
	ok = r->OpenKey( SOKO_ROOT, true );
    if (ok)
	registry = r;
    else
	delete r;
    }

SokoRegistry::~SokoRegistry()
    {
    if (registry != 0)
	{
	registry->CloseKey();
	delete registry;
	}
    }


//-----------------------------------------------------------------------------
// initialize
//-----------------------------------------------------------------------------
void SokoDefaults::initialize()
    {
    if (DELEGATE == 0)
	{
	DELEGATE = new SokoDefaults;
	SokoSetting::set_delegate( DELEGATE );
	}
    }


//-----------------------------------------------------------------------------
// shutdown
//-----------------------------------------------------------------------------
void SokoDefaults::shutdown()
    {
    if (DELEGATE != 0)
	{
	SokoSetting::set_delegate(0);
	delete DELEGATE;
	DELEGATE = 0;
	}
    }


//-----------------------------------------------------------------------------
// exists
//-----------------------------------------------------------------------------
soko_bool SokoDefaults::exists( SokoSetting*, char const* name )
    {
    SokoRegistry r( true );
    return (r.ok() && r.registry->ValueExists( name ));
    }


//-----------------------------------------------------------------------------
// get
//-----------------------------------------------------------------------------
char const* SokoDefaults::get( SokoSetting*, char const* name, cSokoPool pool )
    {
    char const* s = 0;
    SokoRegistry r( true );
    if (r.ok() && r.registry->ValueExists( name ))
	{
	try
	    {
	    buffer = r.registry->ReadString( name );
	    s = buffer.c_str();
	    }
	catch (...) {}
	}
    return (s == 0 ? 0 : SokoPool_store(pool, s));
    }


//-----------------------------------------------------------------------------
// set
//-----------------------------------------------------------------------------
void SokoDefaults::set( SokoSetting*, char const* name, char const* value )
    {
    SokoRegistry r( false );
    if (r.ok())
	try { r.registry->WriteString( name, value ); } catch (...) {}
    }


//-----------------------------------------------------------------------------
// remove
//-----------------------------------------------------------------------------
void SokoDefaults::remove( SokoSetting*, char const* name )
    {
    SokoRegistry r( false );
    if (r.ok() && r.registry->ValueExists( name ))
	r.registry->DeleteValue( name );
    }


//-----------------------------------------------------------------------------
// factory_path
//-----------------------------------------------------------------------------
char const* SokoDefaults::factory_path( SokoSetting*, cSokoPool )
    {
    static AnsiString path;
    if (path.IsEmpty())
	path = ExtractFileDir( Application->ExeName );
    return path.c_str();
    }


//-----------------------------------------------------------------------------
// user_path
//	Attempt to determine the user's personal directory.  Tries the
//	following locations in the order given:
//	- The "personal" folder (i.e. "My Documents")
//	- $(HOME)
//	- $(TEMP)
//	- $(TMP)
//	- Root of drive where the "Windows" directory resides
//	- Root directory of current drive
//-----------------------------------------------------------------------------
char const* SokoDefaults::user_path( SokoSetting*, cSokoPool )
    {
    static AnsiString path;
    if (path.IsEmpty())
	{
	LPITEMIDLIST pidl;
	LPMALLOC shell_malloc;
	char dir[ MAX_PATH ];
	if (SUCCEEDED(SHGetMalloc( &shell_malloc )))
	    {
	    if (SUCCEEDED(SHGetSpecialFolderLocation(0,CSIDL_PERSONAL,&pidl)))
		{
		if (SHGetPathFromIDList(pidl, dir) && strlen(dir) != 0)
		    path = dir;
		shell_malloc->Free( pidl );
		}
	    shell_malloc->Release();
	    }
	if (path.IsEmpty())
	    {
	    char const* s = getenv( "HOME" );
	    if (s == 0) s = getenv( "TEMP" );
	    if (s == 0) s = getenv( "TMP"  );
	    if (s != 0)
		path = s;
	    else if (GetWindowsDirectory( dir, sizeof(dir) ) != 0)
		path = ExtractFileDrive( dir );
	    else
		path = "";
	    }
	path += "\\SokoSave";
	}
    return path.c_str();
    }
