//-----------------------------------------------------------------------------
// SokoBoard.cpp
//
//	Implementation of a Sokoban puzzle board for SokoSave.
//
// Copyright (c), 2001,2002, Eric Sunshine <sunshine@sunshineco.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoBoard.cpp,v 1.4 2002/05/06 05:48:40 sunshine Exp $
// $Log: SokoBoard.cpp,v $
// Revision 1.4  2002/05/06 05:48:40  sunshine
// v19
// -*- Fixed v17 Win9x problem where animation of Hexoban and Trioban puzzles
//     involved unsightly "flashing" which resulted from Windows first
//     painting the on-screen hexagonal and triangular tiles black before
//     blitting in the actual tile.  To work around the problem,
//     SokoGridHexagon and SokoGridTriangle now maintain a backing-store in
//     which the puzzle image is rendered.  When the on-screen image needs to
//     be repainted, only non-transparent rectangles are blitted to the
//     screen.  This patch cuts animation speed effectively in half for
//     Hexoban and Trioban puzzles, but the visual quality of the animation
//     is much improved.  This problem did not affect WinNT or Win2000.
//
// Revision 1.3  2002/02/19 07:52:04  sunshine
// v18
// -*- Added support for new triangular-style Trioban puzzles.
//
// -*- Consolidated the user-interface movement constraints into SokoPuzzle
//     rather than having the same code repeated in each port.  This code
//     decides whether or not the direction movement buttons in the
//     user-interface should be enabled depending upon cell type,
//     orientation, etc.  Added the SokoPuzzle methods can_move_constrained()
//     and can_move_diagonal_constrained() to complement the existing
//     can_move() and can_move_diagonal().
//
// Revision 1.2  2002-01-29 17:12:41-05  sunshine
// v17
// -*- Added support for the new hexagonal-style puzzles.
//
// -*- No longer uses TDrawGrid for display of puzzle.
//
// -*- Added TSokoGrid, a subclass of TGraphicControl which is the base class
//     for grids which display puzzle boards.
//
// -*- Added TSokoGridSquare, a subclass of TSokoGrid, which knows how to
//     draw square-tiled puzzles, and perform appropriate hit-testing.
//
// -*- Added TSokoGridHexagon, a subclass of TSokoGrid, which knows how to
//     draw hexagon-tiled puzzles, and perform appropriate hit-testing.
//
// -*- SokoPuzzle now calculates two new scores in addition to "moves" and
//     "pushes".  The "runs" score is the number of straight lines in which
//     crates have been pushed.  Looked at another way, it is the number of
//     turns crates have made while being pushed.  The "focus" score is the
//     number of times the player's focus has changed from one crate to
//     another.
//
// -*- Added "runs" and "focus" scores controls to SokoBoard.
//
// -*- Relocated functionality of soko_save_filename_for_puzzle(),
//     soko_puzzle_name_for_level(), and soko_level_for_puzzle_name() from
//     SokoFile to SokoPuzzle, where they are class methods of SokoPuzzle.
//
// -*- Extended keyboard movement support.  Numeric keypad can now be used,
//     as well as standard arrow keys.  Added key equivalents for hexagonal
//     puzzles.
//
// -*- Fixed a number of aesthetic problems which showed up when using larger
//     control sizes and fonts.  Controls would clobber one another and text
//     would be clipped.  The problems were particularly acute when using
//     "Large Fonts" from the Windows Display/Advanced settings.
//
// -*- SokoBoard's score labels now have AutoSize enabled.  This allows them
//     to grow as needed when user has selected a large font size, thus
//     preventing the text on the labels from being clipped.
//-----------------------------------------------------------------------------
#include <vcl.h>
#pragma hdrstop

#include "SokoPool.h"
#include "SokoBoard.h"
#include "SokoAlerter.h"
#include "SokoApp.h"
#include "SokoAssert.h"
#include "SokoFile.h"
#include "SokoGridHexagon.h"
#include "SokoGridSquare.h"
#include "SokoGridTriangle.h"
#include "SokoNewScore.h"
#include "SokoPref.h"
#include "SokoSetting.h"
#include "SokoTrackBar.h"
#include <vcl/sysutils.hpp>

#pragma package(smart_init)
#pragma resource "*.dfm"
TSokoBoardForm* SokoBoardForm;

//-----------------------------------------------------------------------------
// SokoBoardDelegate
//	Concrete implementation of abstract SokoPuzzleDelegate which provides
//	SokoBoard module with required implementation-specific functionality.
//-----------------------------------------------------------------------------
class SokoBoardDelegate : public SokoPuzzleDelegate, public SokoAlerter
    {
public:
    virtual void alert( SokoPuzzle*, SokoAlert, char const*, char const* );
    virtual void record_score( SokoPuzzle*, int m, int p, int r, int f );
    virtual void record_level( SokoPuzzle*, int level );
    virtual void set_solved( SokoPuzzle*, soko_bool solved );
    virtual void set_dirty( SokoPuzzle*, soko_bool dirty );
    virtual void refresh_controls( SokoPuzzle* );
    virtual void refresh_all_cells( SokoPuzzle* );
    virtual void refresh_cells( SokoPuzzle*, SokoCell const*, int );
    virtual soko_bool should_animate_playback( SokoPuzzle* );
    virtual int playback_threshold( SokoPuzzle* );
    virtual void begin_playback( SokoPuzzle* );
    virtual void end_playback( SokoPuzzle* );
    virtual void begin_animation( SokoPuzzle* );
    virtual void end_animation( SokoPuzzle* );
    virtual void begin_drag( SokoPuzzle* );
    virtual void end_drag( SokoPuzzle* );
    virtual void process_events( SokoPuzzle*, soko_bool await_event );
    };

#define SOKO_BOARD(P) ((TSokoBoardForm*)(p->info))

void SokoBoardDelegate::alert(
    SokoPuzzle* p, SokoAlert sev, char const* title, char const* msg )
    { send_alert( sev, title, msg ); }
void SokoBoardDelegate::record_score(
    SokoPuzzle* p, int moves, int pushes, int runs, int focus )
    { SOKO_BOARD(p)->record_score( moves, pushes, runs, focus ); }
void SokoBoardDelegate::record_level( SokoPuzzle* p, int level )
    { SOKO_BOARD(p)->record_level( level ); }
void SokoBoardDelegate::set_solved( SokoPuzzle* p, soko_bool solved )
    { SOKO_BOARD(p)->set_solved( solved ); }
void SokoBoardDelegate::set_dirty( SokoPuzzle* p, soko_bool dirty )
    { SOKO_BOARD(p)->set_dirty( dirty ); }
void SokoBoardDelegate::refresh_controls( SokoPuzzle* p )
    { SOKO_BOARD(p)->refresh_controls(); }
void SokoBoardDelegate::refresh_all_cells( SokoPuzzle* p )
    { SOKO_BOARD(p)->refresh_all_cells(); }
void SokoBoardDelegate::refresh_cells(SokoPuzzle* p, SokoCell const* c, int n)
    { SOKO_BOARD(p)->refresh_cells( c, n ); }
soko_bool SokoBoardDelegate::should_animate_playback( SokoPuzzle* p )
    { return (soko_bool)SOKO_BOARD(p)->should_animate_playback(); }
int SokoBoardDelegate::playback_threshold( SokoPuzzle* p )
    { return SOKO_BOARD(p)->playback_threshold(); }
void SokoBoardDelegate::begin_playback( SokoPuzzle* p )
    { SOKO_BOARD(p)->begin_playback(); }
void SokoBoardDelegate::end_playback( SokoPuzzle* p )
    { SOKO_BOARD(p)->end_playback(); }
void SokoBoardDelegate::begin_animation( SokoPuzzle* p )
    { SOKO_BOARD(p)->begin_animation(); }
void SokoBoardDelegate::end_animation( SokoPuzzle* p )
    { SOKO_BOARD(p)->end_animation(); }
void SokoBoardDelegate::begin_drag( SokoPuzzle* p )
    { SOKO_BOARD(p)->begin_drag(); }
void SokoBoardDelegate::end_drag( SokoPuzzle* p )
    { SOKO_BOARD(p)->end_drag(); }
void SokoBoardDelegate::process_events( SokoPuzzle* p, soko_bool await_event )
    { SOKO_BOARD(p)->process_events( await_event ); }

#undef SOKO_BOARD

static SokoBoardDelegate SHARED_DELEGATE;


//-----------------------------------------------------------------------------
// get_idle_monitor_list
//-----------------------------------------------------------------------------
static TList* get_idle_monitor_list()
    {
    static TList* list = 0;
    if (list == 0)
	list = new TList();
    return list;
    }


//-----------------------------------------------------------------------------
// start_idle_monitor
//-----------------------------------------------------------------------------
void TSokoBoardForm::start_idle_monitor()
    {
    SOKO_ASSERT( idle_count >= 0 );
    if (idle_count++ == 0)
	get_idle_monitor_list()->Add( this );
    }


//-----------------------------------------------------------------------------
// stop_idle_monitor
//-----------------------------------------------------------------------------
void TSokoBoardForm::stop_idle_monitor()
    {
    SOKO_ASSERT( idle_count > 0 );
    if (--idle_count == 0)
	get_idle_monitor_list()->Remove( this );
    }


//-----------------------------------------------------------------------------
// refresh_window_title
//-----------------------------------------------------------------------------
void TSokoBoardForm::refresh_window_title()
    {
    SokoPool pool;
    AnsiString s( puzzle->get_save_file_name(pool) );
    AnsiString name = ExtractFileName(s);
    AnsiString dir = ExtractFileDir(s);
    AnsiString title( puzzle->puzzle_dirty() ? "* " : "");
    if (dir.IsEmpty())
	title += name;
    else
	title += name + " -- " + dir;
    Caption = title;
    }


//-----------------------------------------------------------------------------
// record_score
//	Delay actual recording of score until application is idle, since this
//	method might otherwise be called during a mouse down, a drag session,
//	etc, and it would be unfriendly to launch the modal "New Score" panel
//	before those activities ceased.
//-----------------------------------------------------------------------------
void TSokoBoardForm::record_score( int moves, int pushes, int runs, int focus )
    {
    record_moves = moves;
    record_pushes = pushes;
    record_runs = runs;
    record_focus = focus;
    start_idle_monitor();
    }


//-----------------------------------------------------------------------------
// record_level
//-----------------------------------------------------------------------------
void TSokoBoardForm::record_level( int level )
    {
    if (level >= TSokoPrefForm::get_next_level())
	TSokoPrefForm::set_next_level( level + 1 );
    }


//-----------------------------------------------------------------------------
// set_solved
//-----------------------------------------------------------------------------
void TSokoBoardForm::set_solved( bool solved )
    {
    solved_label->Caption = (solved ? "SOLVED" : "");
    }


//-----------------------------------------------------------------------------
// set_dirty
//-----------------------------------------------------------------------------
void TSokoBoardForm::set_dirty( bool dirty )
    {
    (void)dirty;
    refresh_window_title();
    }


//-----------------------------------------------------------------------------
// wm_size
//	This method is called whenever the user changes Windows settings such
//	as menu font, menu height, window border thickness, etc.  We need to
//	respond to the user changes to ensure that the window's client area is
//	still large enough to display the entire puzzle since the VCL does not
//	always adjust the client size for us.  For instance, if the user
//	increases the menu height, or especially if a new menu font size causes
//	the menu to wrap, then the menu might begin to occlude the top of the
//	puzzle.
//-----------------------------------------------------------------------------
void TSokoBoardForm::wm_size( Messages::TMessage& m )
    {
    superclass::Dispatch(&m);
    if (!sizing_self && puzzle_grid != 0)
	{
	sizing_self = true;
	adjust_form_size();
	sizing_self = false;
	}
    }


//-----------------------------------------------------------------------------
// wm_syscolorchange
//	This method is called whenever the user changes the Windows system
//	colors.  Puzzle grids which create a cached backing store of their
//	content may need to respond to this message in order to refresh the
//	cache.
//-----------------------------------------------------------------------------
void TSokoBoardForm::wm_syscolorchange( Messages::TMessage& m )
    {
    puzzle_grid->note_color_change();
    superclass::Dispatch(&m);
    }


//-----------------------------------------------------------------------------
// check_for_next_puzzle
//-----------------------------------------------------------------------------
void TSokoBoardForm::check_for_next_puzzle()
    {
    bool ok = false;
    SokoPool pool;
    int const level =
	SokoPuzzle::level_for_puzzle_name(puzzle->get_puzzle_file_name(pool));
    if (level >= 0)
	ok = FileExists( SokoPuzzle::puzzle_name_for_level(level + 1, pool) );
    if (!ok)
	{
	next_puzzle_button->Parent = 0;
	delete next_puzzle_button;
	next_puzzle_button = 0;
	game_menu->Remove( next_item );
	delete next_item;
	next_item = 0;
	}
    }


//-----------------------------------------------------------------------------
// adjust_form_size
//	Resize window to contain puzzle grid.  However, consider designed size
//	as minimum limit, so do not shrink window.  If puzzle grid is smaller
//	than minimum limit, then center grid in the window in the appropriate
//	dimension(s).
//-----------------------------------------------------------------------------
void TSokoBoardForm::adjust_form_size()
    {
    int const grid_width  = puzzle_grid->Width;
    int const grid_height = puzzle_grid->Height;
    int x = grid_inset_x;
    int y = grid_inset_y;
    int w = grid_width  + grid_inset_x * 2;
    int h = grid_height + grid_inset_y * 2;
    if (w < grid_min_width)
	{
	x += (grid_min_width - w) / 2;
	w = grid_min_width;
	}
    if (h < grid_min_height)
	{
	y += (grid_min_height - h) / 2;
	h = grid_min_height;
	}
    SetBounds( Left, Top,
	w + Width  - puzzle_panel->ClientWidth,
	h + Height - puzzle_panel->ClientHeight );
    puzzle_grid->SetBounds( x, y, grid_width, grid_height );
    }


//-----------------------------------------------------------------------------
// capture_form_constraints
//-----------------------------------------------------------------------------
void TSokoBoardForm::capture_form_constraints()
    {
    grid_inset_x = 4;
    grid_inset_y = 4;
    grid_min_width  = puzzle_panel->Width;
    grid_min_height = puzzle_panel->Height;
    }


//-----------------------------------------------------------------------------
// create_grid
//-----------------------------------------------------------------------------
void TSokoBoardForm::create_grid()
    {
    SokoPuzzleStyle const style = puzzle->puzzle_style();
    if (style == SOKO_STYLE_SQUARE)
	puzzle_grid = new TSokoGridSquare ( this, puzzle );
    else if (style == SOKO_STYLE_HEXAGON)
	puzzle_grid = new TSokoGridHexagon( this, puzzle );
    else if (style == SOKO_STYLE_TRIANGLE)
	puzzle_grid = new TSokoGridTriangle( this, puzzle );
    SOKO_ASSERT( puzzle_grid != 0 );
    puzzle_grid->Parent = puzzle_panel;
    puzzle_grid->OnMouseDown = puzzle_grid_mouse_down;
    puzzle_grid->OnMouseUp = puzzle_grid_mouse_up;
    }


//-----------------------------------------------------------------------------
// configure_board
//-----------------------------------------------------------------------------
void TSokoBoardForm::configure_board( SokoPuzzle* p )
    {
    puzzle = p;
    create_grid();
    capture_form_constraints();
    adjust_form_size();
    set_dirty( puzzle->puzzle_dirty() );
    set_solved( puzzle->puzzle_solved() );
    stop_button->Enabled = false;
    check_for_next_puzzle();
    refresh_controls();
    refresh_all_cells();
    refresh_window_title();
    }


//-----------------------------------------------------------------------------
// enrich_pathname
//	This function enriches a path name.  In particular, it iterates over
//	the path and converts each component into its "long" form.  For
//	instance, "progr~1" becomes "Program Files".  This ensures that
//	soko_collapse() will be able to correctly collapse the pathname, since
//	that function works with long names.  However, more importantly, this
//	function ensures that the final path component has a "long" extension.
//	For example, "01~1.sok" becomes "01.sokomaze".  This allows the lower
//	level SokoSave functions to properly recognize a SokoSave file based
//	upon its file extension even though Windows may (and often does) pass
//	the file name along to the program in its short form.  This function
//	works correctly for standard paths ("C:\foobar"), as well as UNC paths
//	("\\host\share").
//-----------------------------------------------------------------------------
static AnsiString enrich_pathname( char const* in_path )
    {
    bool ok = true;
    AnsiString out_path;
    TStringList* list = new TStringList;
    AnsiString path = ExpandFileName( in_path );
    AnsiString prefix = ExtractFileDrive( path );

    if (prefix.Length() != 0)
	{
	path.Delete( 1, prefix.Length() );     // AnsiString is 1-based.
	out_path += prefix;
	}
    if (path.Length() >= 1 && path[1] == '\\') // Ditto.
	{
	path.Delete(1,1);                      // Ditto.
	prefix += '\\';
	}

    for ( ; path.Length() != 0; path = ExtractFileDir(path))
	list->Add( path );

    for (int i = list->Count - 1; i >= 0; i--)
	{
	path = prefix + list->Strings[i];
	WIN32_FIND_DATA data;
	HANDLE h = FindFirstFile( path.c_str(), &data );
	if (h == INVALID_HANDLE_VALUE)
	    ok = false;
	else
	    {
	    out_path += AnsiString('\\') + data.cFileName;
	    FindClose(h);
	    }
	}
    delete list;

    if (!ok)
	out_path = in_path;
    return out_path;
    }


//-----------------------------------------------------------------------------
// open_puzzle
//-----------------------------------------------------------------------------
bool TSokoBoardForm::open_puzzle( char const* path )
    {
    TSokoBoardForm* board;
    AnsiString long_path = enrich_pathname( path );
    SokoPuzzle* puzzle = SokoPuzzle::find_open_puzzle( long_path.c_str() );
    if (puzzle != 0)
	board = (TSokoBoardForm*)puzzle->info;
    else
	{
	board = new TSokoBoardForm( Application );
	puzzle = new SokoPuzzle( &SHARED_DELEGATE, long_path.c_str(), board );
	if (puzzle->instance_valid())
	    board->configure_board( puzzle );
	else
	    {
	    delete puzzle;
	    delete board;
	    board = 0;
	    }
	}
    if (board != 0)
	{
	if (board->WindowState == wsMinimized)
	    ShowWindow( board->Handle, SW_RESTORE );
	board->Show();
	board->ActiveControl = 0;
	}
    return (board != 0);
    }


//-----------------------------------------------------------------------------
// open_puzzle_for_level
//	This method is called at program launch time to automatically open the
//	next puzzle which the user has not yet solved.  On the assumption that
//	the user may have begun working on the puzzle and then saved it with
//	the intention of continuing work on it later, we first try to open a
//	saved game with the given level number.  If a saved game does not
//	exist, then we try starting a new game.
//-----------------------------------------------------------------------------
bool TSokoBoardForm::open_puzzle_for_level( int level )
    {
    bool ok = false;
    SokoPool pool;
    char const* new_game = SokoPuzzle::puzzle_name_for_level( level, pool );
    char const* save_game =
	SokoPuzzle::save_name_for_puzzle_name( new_game, pool );
    if (FileExists( save_game ))
	ok = open_puzzle( save_game );
    if (!ok && FileExists( new_game ))
	ok = open_puzzle( new_game );
    return ok;
    }


//-----------------------------------------------------------------------------
// open_next_puzzle
//-----------------------------------------------------------------------------
bool TSokoBoardForm::open_next_puzzle()
    {
    return open_puzzle_for_level( TSokoPrefForm::get_next_level() ) ||
	   open_puzzle_for_level(1);
    }


//-----------------------------------------------------------------------------
// refresh_all_cells
//-----------------------------------------------------------------------------
void TSokoBoardForm::refresh_all_cells()
    {
    puzzle_grid->Repaint();
    }


//-----------------------------------------------------------------------------
// refresh_cells
//-----------------------------------------------------------------------------
void TSokoBoardForm::refresh_cells( SokoCell const* cells, int ncells )
    {
    for (int i = 0; i < ncells; i++)
	{
	SokoCell const& c = cells[i];
	puzzle_grid->draw_cell( c.row, c.col, c.type );
	}
    }


//-----------------------------------------------------------------------------
// refresh_controls
//-----------------------------------------------------------------------------
void TSokoBoardForm::refresh_controls()
    {
#define ENABLE_DIR(B,D) \
    B##_button->Enabled = puzzle->can_move_constrained(SOKO_DIR_##D)
#define ENABLE_DIAG(B,D) \
    B##_button->Enabled = puzzle->can_move_diagonal_constrained(SOKO_DIAG_##D)
    ENABLE_DIR(up,UP);
    ENABLE_DIR(left,LEFT);
    ENABLE_DIR(right,RIGHT);
    ENABLE_DIR(down,DOWN);
    ENABLE_DIAG(up_left,UP_LEFT);
    ENABLE_DIAG(up_right,UP_RIGHT);
    ENABLE_DIAG(down_left,DOWN_LEFT);
    ENABLE_DIAG(down_right,DOWN_RIGHT);
#undef ENABLE_DIAG
#undef ENABLE_DIR

    int  const hpos = puzzle->history_position();
    int  const hlen = puzzle->history_length();
    bool const undo = (hpos > 0);
    bool const redo = (hpos < hlen);
    undo_move_button->Enabled = undo;
    redo_move_button->Enabled = redo;
    undo_push_button->Enabled = undo;
    redo_push_button->Enabled = redo;
    undo_move_item->Enabled = undo;
    redo_move_item->Enabled = redo;
    undo_push_item->Enabled = undo;
    redo_push_item->Enabled = redo;
    playback_slider->Max      = hlen;
    playback_slider->Position = hpos;
    playback_slider->Enabled  = (hlen > 0);
    playback_slider->PageSize = (hlen >= 3 ? hlen / 3 : 1);

    moves_field->Caption  = puzzle->move_count();
    pushes_field->Caption = puzzle->push_count();
    runs_field->Caption   = puzzle->run_count();
    focus_field->Caption  = puzzle->focus_count();

    if (ActiveControl != 0)
	ActiveControl = 0;
    }


//-----------------------------------------------------------------------------
// should_animate_playback
//-----------------------------------------------------------------------------
bool TSokoBoardForm::should_animate_playback() const
    {
    return animate_switch->Checked;
    }


//-----------------------------------------------------------------------------
// begin_playback
//-----------------------------------------------------------------------------
void TSokoBoardForm::begin_playback()
    {
    stop_button->Enabled = true;
    animate_switch->Enabled = false;
    playback_slider->Enabled = false;
    undo_move_button->Enabled = false;
    redo_move_button->Enabled = false;
    undo_push_button->Enabled = false;
    redo_push_button->Enabled = false;
    undo_move_item->Enabled = false;
    redo_move_item->Enabled = false;
    undo_push_item->Enabled = false;
    redo_push_item->Enabled = false;
    up_button->Enabled = false;
    down_button->Enabled = false;
    left_button->Enabled = false;
    right_button->Enabled = false;
    up_left_button->Enabled = false;
    up_right_button->Enabled = false;
    down_left_button->Enabled = false;
    down_right_button->Enabled = false;
    start_idle_monitor();
    }


//-----------------------------------------------------------------------------
// end_playback
//	Most controls will be re-enabled by the normal control update machinery
//	following conclusion of playback, so we do not need to manually
//	re-enable all of the controls disabled by begin_playback().
//-----------------------------------------------------------------------------
void TSokoBoardForm::end_playback()
    {
    stop_button->Enabled = false;
    animate_switch->Enabled = true;
    playback_slider->Enabled = true;
    stop_idle_monitor();
    }


//-----------------------------------------------------------------------------
// handle_app_idle
//	The `done' argument is "true" by default and means that the application
//	should actually go idle until an event arrives.  Setting `done' to
//	"false" prevents the application from going idle, and instead invokes
//	the idle handler (this method) again without delay.  Therefore, in
//	order to animate playback sessions, we set `done' to "false".
//
//	When a new high score needs to be recorded, only do so when application
//	is really idle.  In other words, no playback active, no drag session
//	active, etc.
//-----------------------------------------------------------------------------
void TSokoBoardForm::handle_app_idle( bool& done )
    {
    // Processing a playback session?
    if (puzzle->playback_active())
	{
	puzzle->advance_playback();
	if (puzzle->playback_active()) // Still active after advancing?
	    done = false;              // Yes, so tell app to call us again.
	}

    // Need to record a new high score?
    if (done && record_moves != -1 && !puzzle->drag_active())
	{
	SOKO_ASSERT( record_pushes != -1 );
	int const m = record_moves;
	int const p = record_pushes;
	int const r = record_runs;
	int const f = record_focus;
	record_moves = record_pushes = record_runs = record_focus = -1;
	stop_idle_monitor();
	TSokoNewScoreForm::record_score( *puzzle, m, p, r, f );
	}
    }


//-----------------------------------------------------------------------------
// app_idle
//	The application is idle (not receiving any events).  Give boards which
//	have requested idle-notification a chance to process it.
//-----------------------------------------------------------------------------
void TSokoBoardForm::app_idle( bool& done )
    {
    TList* list = get_idle_monitor_list();
    for (int i = list->Count - 1; i >= 0; i--)
	((TSokoBoardForm*)list->Items[i])->handle_app_idle( done );
    }


//-----------------------------------------------------------------------------
// abort_playback
//-----------------------------------------------------------------------------
void TSokoBoardForm::abort_playback()
    {
    puzzle->abort_playback();
    }


//-----------------------------------------------------------------------------
// app_deactivated
//	The application deactivated.  Abort active playback sessions.
//-----------------------------------------------------------------------------
void TSokoBoardForm::app_deactivated()
    {
    TList* list = get_idle_monitor_list();
    for (int i = list->Count - 1; i >= 0; i--)
	((TSokoBoardForm*)list->Items[i])->handle_app_deactivated();
    }


//-----------------------------------------------------------------------------
// handle_app_deactivated
//-----------------------------------------------------------------------------
void TSokoBoardForm::handle_app_deactivated()
    {
    abort_playback();
    }


//-----------------------------------------------------------------------------
// form_deactivated
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::form_deactivated( TObject* sender )
    {
    abort_playback();
    }


//-----------------------------------------------------------------------------
// begin_drag
//-----------------------------------------------------------------------------
void TSokoBoardForm::begin_drag()
    {
    puzzle_grid->OnMouseMove = puzzle_grid_mouse_move;
    }


//-----------------------------------------------------------------------------
// end_drag
//-----------------------------------------------------------------------------
void TSokoBoardForm::end_drag()
    {
    puzzle_grid->OnMouseMove = 0;
    }


//-----------------------------------------------------------------------------
// process_events
//-----------------------------------------------------------------------------
void TSokoBoardForm::process_events( bool await_event )
    {
    if (await_event)
	Application->HandleMessage();
    else
	Application->ProcessMessages();
    }


//-----------------------------------------------------------------------------
// Constructor
//-----------------------------------------------------------------------------
__fastcall TSokoBoardForm::TSokoBoardForm( TComponent* owner ) :
    TSokoForm(owner),
    puzzle(0),
    puzzle_grid(0),
    timed_button(0),
    sizing_self(false),
    record_moves(-1),
    record_pushes(-1),
    record_runs(-1),
    record_focus(-1),
    grid_inset_x(0),
    grid_inset_y(0),
    grid_min_width(0),
    grid_min_height(0),
    idle_count(0)
    {
    should_cascade = true;
    }


//-----------------------------------------------------------------------------
// Destructor
//-----------------------------------------------------------------------------
__fastcall TSokoBoardForm::~TSokoBoardForm()
    {
    while (idle_count > 0)
	stop_idle_monitor();
    if (toolbar_panel->Parent == 0)
	toolbar_panel->Parent = this; // Else VCL crashes.
    if (puzzle != 0)
	delete puzzle;
    }


//-----------------------------------------------------------------------------
// refresh_toolbar_state
//-----------------------------------------------------------------------------
void TSokoBoardForm::refresh_toolbar_state()
    {
    toggle_toolbar_item->Caption =
	(toolbar_panel->Parent == 0 ? "Show Toolbar" : "Hide Toolbar");
    }


//-----------------------------------------------------------------------------
// enable_toolbar
//-----------------------------------------------------------------------------
void TSokoBoardForm::enable_toolbar( bool enable )
    {
    bool const enabled = (toolbar_panel->Parent != 0);
    if (enable != enabled)
	{
	sizing_self = true;
	if (enable)
	    {
	    Height = Height + toolbar_panel->Height;
	    toolbar_panel->Parent = this;
	    }
	else
	    {
	    toolbar_panel->Parent = 0;
	    Height = Height - toolbar_panel->Height;
	    }
	sizing_self = false;
	refresh_toolbar_state();
	}
    }


//-----------------------------------------------------------------------------
// enable_toolbars
//-----------------------------------------------------------------------------
void TSokoBoardForm::enable_toolbars( bool enable )
    {
    int const n = SokoPuzzle::open_puzzle_count();
    for (int i = 0; i < n; i++)
	{
	TSokoBoardForm* b =
	    (TSokoBoardForm*)SokoPuzzle::get_open_puzzle(i)->info;
	b->enable_toolbar( enable );
	}
    }


//-----------------------------------------------------------------------------
// configure_images
//-----------------------------------------------------------------------------
void TSokoBoardForm::configure_images()
    {
#define BUTTON_IMAGE(X,Y,Z) BUTTON_IMAGE_2(X,SokoArrow##Y,Z)
#define BUTTON_IMAGE_2(X,Y,Z) \
    X##_button->Glyph->LoadFromResourceName((int)HInstance,#Y); \
    X##_button->NumGlyphs = (Z)
    BUTTON_IMAGE(up,Up,2);
    BUTTON_IMAGE(down,Down,2);
    BUTTON_IMAGE(left,Left,2);
    BUTTON_IMAGE(right,Right,2);
    BUTTON_IMAGE(up_left,UpLeft,2);
    BUTTON_IMAGE(up_right,UpRight,2);
    BUTTON_IMAGE(down_left,DownLeft,2);
    BUTTON_IMAGE(down_right,DownRight,2);
    BUTTON_IMAGE(undo_move,Left,2);
    BUTTON_IMAGE(redo_move,Right,2);
    BUTTON_IMAGE(undo_push,LeftLeft,2);
    BUTTON_IMAGE(redo_push,RightRight,2);
    BUTTON_IMAGE_2(next_puzzle,SokoNextPuzzle,1);
    BUTTON_IMAGE_2(new_game,SokoNewGame,1);
    BUTTON_IMAGE_2(open_game,SokoOpenGame,1);
    BUTTON_IMAGE_2(save_game,SokoSaveGame,1);
    BUTTON_IMAGE_2(show_scores,SokoHighScores,1);
    BUTTON_IMAGE_2(show_help,SokoHelp,1);
#undef BUTTON_IMAGE_2
#undef BUTTON_IMAGE
    }


//-----------------------------------------------------------------------------
// configure_controls
//-----------------------------------------------------------------------------
void TSokoBoardForm::configure_controls()
    {
#define BUTTON_TAG(X,Y) X##_button->Tag = SOKO_##Y
    BUTTON_TAG(up,DIR_UP);
    BUTTON_TAG(down,DIR_DOWN);
    BUTTON_TAG(left,DIR_LEFT);
    BUTTON_TAG(right,DIR_RIGHT);
    BUTTON_TAG(up_left,DIAG_UP_LEFT);
    BUTTON_TAG(up_right,DIAG_UP_RIGHT);
    BUTTON_TAG(down_left,DIAG_DOWN_LEFT);
    BUTTON_TAG(down_right,DIAG_DOWN_RIGHT);
#undef BUTTON_TAG
    }


//-----------------------------------------------------------------------------
// form_create
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::form_create( TObject* sender )
    {
    TSokoTrackBar* p = TSokoTrackBar::subsume( playback_slider );
    p->OnEndTracking = playback_slider_change;
    configure_images();
    configure_controls();
    enable_toolbar( TSokoPrefForm::should_show_toolbar() );
    refresh_toolbar_state(); // Needed first time, despite line above.
    superclass::window_menu = window_menu;
    }


//-----------------------------------------------------------------------------
// form_close
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::form_close( TObject* o, TCloseAction& action )
    {
    action = caFree;
    abort_playback();
    if (SokoPuzzle::open_puzzle_count() == 1)
	Application->Terminate();
    }


//-----------------------------------------------------------------------------
// preferences_item_click
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::preferences_item_click( TObject* sender )
    {
    SokoAppForm->launch_preferences();
    }


//-----------------------------------------------------------------------------
// open_puzzles
//-----------------------------------------------------------------------------
bool TSokoBoardForm::open_puzzles( TStrings* paths )
    {
    bool ok = false;
    int const n = paths->Count;
    for (int i = 0; i < n; i++)
	if (open_puzzle(paths->Strings[i].c_str()))
	    ok = true;
    return ok;
    }


//-----------------------------------------------------------------------------
// new_game_click
//	Only seed the dialog with the name of "next" puzzle if the puzzle
//	exists and if the directory last used by the panel is the same the
//	"factory" puzzle directory.  This way, if the user decides to launch
//	new games from some other directory (containing, for instance, puzzles
//	not distributed with SokoSave), we won't keep jumping back to the
//	factory puzzle directory.
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::new_game_click( TObject* sender )
    {
    bool seeded = false;
    SokoPool pool;
    int const level = TSokoPrefForm::get_next_level();
    AnsiString path( SokoPuzzle::puzzle_name_for_level(level, pool) );
    if (FileExists( path ))
	{
	AnsiString dir( ExtractFileDir(path) );
	if (dir.AnsiCompareIC( SokoAppForm->new_dialog->InitialDir ) == 0)
	    {
	    SokoAppForm->new_dialog->FileName = ExtractFileName( path );
	    seeded = true;
	    }
	}
    if (!seeded)
	SokoAppForm->new_dialog->FileName = ""; // Clear last selected name.
    abort_playback();
    if (SokoAppForm->new_dialog->Execute())
	{
	open_puzzles( SokoAppForm->new_dialog->Files );
	SokoAppForm->new_dialog->InitialDir =
	    ExtractFileDir( SokoAppForm->new_dialog->FileName );
	}
    }


//-----------------------------------------------------------------------------
// next_puzzle_click
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::next_puzzle_click( TObject* sender )
    {
    SokoPool pool;
    int const level =
	SokoPuzzle::level_for_puzzle_name(puzzle->get_puzzle_file_name(pool));
    if (level >= 0)
	open_puzzle_for_level( level + 1 );
    }


//-----------------------------------------------------------------------------
// choose_puzzle
//-----------------------------------------------------------------------------
bool TSokoBoardForm::choose_puzzle()
    {
    bool ok = false;
    if (SokoAppForm->open_dialog->Execute())
	{
	ok = open_puzzles( SokoAppForm->open_dialog->Files );
	SokoAppForm->open_dialog->InitialDir =
	    ExtractFileDir( SokoAppForm->open_dialog->FileName );
	}
    return ok;
    }


//-----------------------------------------------------------------------------
// open_game_click
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::open_game_click( TObject* sender )
    {
    abort_playback();
    choose_puzzle();
    }


//-----------------------------------------------------------------------------
// close_item_click
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::close_item_click( TObject* sender )
    {
    Screen->ActiveForm->Close();
    }


//-----------------------------------------------------------------------------
// save_game_click
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::save_game_click( TObject* sender )
    {
    abort_playback();
    puzzle->save();
    }


//-----------------------------------------------------------------------------
// save_as_click
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::save_as_click( TObject* sender )
    {
    SokoPool pool;
    abort_playback();
    SokoAppForm->save_dialog->FileName =
	ExtractFileName(puzzle->get_save_file_name(pool));
    if (SokoAppForm->save_dialog->Execute() &&
	puzzle->save_as( SokoAppForm->save_dialog->FileName.c_str() ))
	{
	refresh_window_title();
	SokoAppForm->save_dialog->InitialDir =
	    ExtractFileDir( SokoAppForm->save_dialog->FileName );
	}
    }


//-----------------------------------------------------------------------------
// save_all_click
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::save_all_click( TObject* sender )
    {
    SokoPuzzle::save_all_puzzles();
    }


//-----------------------------------------------------------------------------
// exit_item_click
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::exit_item_click( TObject* sender )
    {
    Application->Terminate();
    }


//-----------------------------------------------------------------------------
// cascade_item_click
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::cascade_item_click( TObject* sender )
    {
    TSokoForm::cascade_all();
    }


//-----------------------------------------------------------------------------
// minimize_item_click
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::minimize_item_click( TObject* sender )
    {
    Screen->ActiveForm->WindowState = wsMinimized;
    }


//-----------------------------------------------------------------------------
// minimize_all_item_click
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::minimize_all_item_click( TObject* sender )
    {
    Application->Minimize();
    }


//-----------------------------------------------------------------------------
// show_scores_click
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::show_scores_click( TObject* sender )
    {
    SokoAppForm->launch_scores();
    }


//-----------------------------------------------------------------------------
// show_help_click
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::show_help_click( TObject* sender )
    {
    SokoAppForm->launch_instructions();
    }


//-----------------------------------------------------------------------------
// about_item_click
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::about_item_click( TObject* sender )
    {
    SokoAppForm->launch_about();
    }


//-----------------------------------------------------------------------------
// toggle_toolbar_click
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::toggle_toolbar_click( TObject* sender )
    {
    enable_toolbar( toolbar_panel->Parent == 0 );
    }


//-----------------------------------------------------------------------------
// event_flags_for_state
//-----------------------------------------------------------------------------
SokoEventFlags TSokoBoardForm::event_flags_for_state( TShiftState state ) const
    {
    SokoEventFlags flags = 0;
    if (state.Contains( ssShift )) flags |= SOKO_FLAG_SHIFT;
    if (state.Contains( ssAlt   )) flags |= SOKO_FLAG_ALTERNATE;
    if (state.Contains( ssCtrl  )) flags |= SOKO_FLAG_CONTROL;
    return flags;
    }


//-----------------------------------------------------------------------------
// mouse_action
//-----------------------------------------------------------------------------
void TSokoBoardForm::mouse_action(
    TMouseButton button, soko_bool down, TShiftState shift, int x, int y )
    {
    int soko_button;
    switch (button)
	{
	case mbLeft:   soko_button =  0; break;
	case mbRight:  soko_button =  1; break;
	case mbMiddle: soko_button =  2; break;
	default:       soko_button = -1; break;
	}

    int row, col;
    puzzle_grid->mouse_to_cell( x, y, row, col );

    puzzle->mouse_action(
	soko_button, down, event_flags_for_state(shift), row, col );
    }


//-----------------------------------------------------------------------------
// puzzle_grid_mouse_down
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::puzzle_grid_mouse_down(
    TObject* sender, TMouseButton button, TShiftState shift, int x, int y )
    {
    mouse_action( button, soko_true, shift, x, y );
    }


//-----------------------------------------------------------------------------
// puzzle_grid_mouse_up
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::puzzle_grid_mouse_up(
    TObject* sender, TMouseButton button, TShiftState shift, int x, int y )
    {
    mouse_action( button, soko_false, shift, x, y );
    }


//-----------------------------------------------------------------------------
// puzzle_grid_mouse_move
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::puzzle_grid_mouse_move(
    TObject* sender, TShiftState shift, int x, int y )
    {
    int row, col;
    puzzle_grid->mouse_to_cell( x, y, row, col );
    puzzle->mouse_drag( row, col );
    }


//-----------------------------------------------------------------------------
// direction_button_click
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::direction_button_click( TObject* sender )
    {
    TSpeedButton* b = dynamic_cast<TSpeedButton*>(sender);
    SOKO_ASSERT( b != 0 );
    SokoDirection const dir = (SokoDirection)b->Tag;
    SOKO_ASSERT( dir >= 0 && dir < SOKO_DIR_MAX );
    puzzle->move( dir );
    }


//-----------------------------------------------------------------------------
// diagonal_button_click
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::diagonal_button_click( TObject* sender )
    {
    TSpeedButton* b = dynamic_cast<TSpeedButton*>(sender);
    SOKO_ASSERT( b != 0 );
    SokoDiagonal const dir = (SokoDiagonal)b->Tag;
    SOKO_ASSERT( dir >= 0 && dir < SOKO_DIAG_MAX );
    puzzle->move_diagonal( dir );
    }


//-----------------------------------------------------------------------------
// stop_button_click
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::stop_button_click( TObject* sender )
    {
    abort_playback();
    }


//-----------------------------------------------------------------------------
// animate_switch_click
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::animate_switch_click( TObject* sender )
    {
    ActiveControl = 0;
    }


//-----------------------------------------------------------------------------
// undo_move_click
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::undo_move_click( TObject* sender )
    {
    puzzle->undo_move();
    }


//-----------------------------------------------------------------------------
// redo_move_click
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::redo_move_click( TObject* sender )
    {
    puzzle->redo_move();
    }


//-----------------------------------------------------------------------------
// undo_push_click
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::undo_push_click( TObject* sender )
    {
    puzzle->undo_push();
    }


//-----------------------------------------------------------------------------
// redo_push_click
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::redo_push_click( TObject* sender )
    {
    puzzle->redo_push();
    }


//-----------------------------------------------------------------------------
// playback_slider_change
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::playback_slider_change( TObject* sender )
    {
    puzzle->reenact_history( playback_slider->Position );
    }


//-----------------------------------------------------------------------------
// key_action
//-----------------------------------------------------------------------------
void TSokoBoardForm::key_action( WORD& key, soko_bool down, TShiftState shift )
    {
    SokoKeyCode code = (SokoKeyCode)-1;
    switch (key)
	{
	case VK_UP:      code = SOKO_KEY_UP;         break;
	case VK_DOWN:    code = SOKO_KEY_DOWN;       break;
	case VK_LEFT:    code = SOKO_KEY_LEFT;       break;
	case VK_RIGHT:   code = SOKO_KEY_RIGHT;      break;
	case VK_PRIOR:   code = SOKO_KEY_UP_RIGHT;   break;
	case VK_NEXT:    code = SOKO_KEY_DOWN_RIGHT; break;
	case VK_END:     code = SOKO_KEY_DOWN_LEFT;  break;
	case VK_HOME:    code = SOKO_KEY_UP_LEFT;    break;
	case VK_INSERT:  code = SOKO_KEY_UP_LEFT;    break;
	case VK_DELETE:  code = SOKO_KEY_DOWN_LEFT;  break;
	case VK_NUMPAD1: code = SOKO_KEY_DOWN_LEFT;  break;
	case VK_NUMPAD2: code = SOKO_KEY_DOWN;       break;
	case VK_NUMPAD3: code = SOKO_KEY_DOWN_RIGHT; break;
	case VK_NUMPAD4: code = SOKO_KEY_LEFT;       break;
	case VK_NUMPAD5: code = SOKO_KEY_DOWN;       break;
	case VK_NUMPAD6: code = SOKO_KEY_RIGHT;      break;
	case VK_NUMPAD7: code = SOKO_KEY_UP_LEFT;    break;
	case VK_NUMPAD8: code = SOKO_KEY_UP;         break;
	case VK_NUMPAD9: code = SOKO_KEY_UP_RIGHT;   break;
	case VK_ESCAPE:  code = SOKO_KEY_ESCAPE;     break;
	}
    if (code != (SokoKeyCode)-1)
	{
	puzzle->key_action( code, down, event_flags_for_state(shift) );
	key = 0; // Tell VCL that key was handled.
	}
    }


//-----------------------------------------------------------------------------
// key_down
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::key_down(
    TObject* sender, WORD& key, TShiftState shift )
    {
    key_action( key, soko_true, shift );
    }


//-----------------------------------------------------------------------------
// key_up
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::key_up(
    TObject* sender, WORD& key, TShiftState shift )
    {
    key_action( key, soko_false, shift );
    }


//=============================================================================
// Continuous Mode Button Emulation
//
//	Several SokoBoard buttons are more useful if they send the OnClick
//	notification continuously as long as the button is held down.  VCL
//	buttons normally only send OnClick when the mouse is released.  To work
//	around this limitation, upon mouse down, a timer is configured to send
//	continuous OnClick messages for as long as the button is depressed
//	(though only when the mouse is actually within the button's client
//	area).  In order to de-bounce the button, the timer's initial interval
//	is set fairly high, and is lowered only after the timer fires for the
//	first time following the mouse down.
//
//=============================================================================
//-----------------------------------------------------------------------------
// timer_fired
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::timer_fired( TObject* sender )
    {
    SOKO_ASSERT( timed_button != 0 );
    TSpeedButton* const b = timed_button;
    TPoint const p = b->ScreenToClient( Mouse->CursorPos );
    if (p.x>=0 && p.x <= b->ClientWidth && p.y>=0 && p.y <= b->ClientHeight)
	b->OnClick(b);
    if (button_timer->Interval != 20)
	button_timer->Interval = 20;
    }


//-----------------------------------------------------------------------------
// repeat_button_mouse_down
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::repeat_button_mouse_down(
    TObject* sender, TMouseButton button, TShiftState shift, int x, int y )
    {
    TSpeedButton* b = dynamic_cast<TSpeedButton*>(sender);
    SOKO_ASSERT( b != 0 );
    timed_button = b;
    button_timer->Interval = 250;
    button_timer->Enabled = true;
    }


//-----------------------------------------------------------------------------
// repeat_button_mouse_up
//-----------------------------------------------------------------------------
void __fastcall TSokoBoardForm::repeat_button_mouse_up(
    TObject* sender, TMouseButton button, TShiftState shift, int x, int y )
    {
    button_timer->Enabled = false;
    timed_button = 0;
    }
