//-----------------------------------------------------------------------------
// SokoApp.cpp
//
//	Global hidden form representing application instance.
//
// Copyright (c), 2001,2002, Eric Sunshine <sunshine@sunshineco.com>
// All rights reserved.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: SokoApp.cpp,v 1.3 2002/02/19 07:50:08 sunshine Exp $
// $Log: SokoApp.cpp,v $
// Revision 1.3  2002/02/19 07:50:08  sunshine
// v18
// -*- Added support for new triangular-style Trioban puzzles.
//
// -*- Fixed bug: Some of the file mask descriptions on the new/open panels
//     had an errant leading semi-colon.
//
// -*- Now notifies Windows shell after all file associations have been made,
//     rather than after each registration.
//
// Revision 1.2  2002-01-29 16:36:33-05  sunshine
// v17
// -*- Added support for the new hexagonal-style puzzles.
//
// -*- Now recognize these additional file extensions: .xsb, .hsb, .sokohex.
//
// -*- Added new document icon for .hsb and .sokohex extensions.
//
// -*- Removed soko_get_puzzle_extension() and soko_get_save_extension() from
//     SokoFile.
//
// -*- Added get_puzzle_extensions(), get_puzzle_extensions_count(), and
//     get_save_extension() as class methods to SokoPuzzle.  There are now
//     multiple recognized puzzle extensions, rather than just the one
//     (.sokomaze).
//
// Revision 1.1  2001-12-21 16:47:31-05  sunshine
// v15
// -*- Extracted core game logic out of GUI code and generalized it so that
//     the same core code can be used by any platform.  Logic from
//     SokoBoard.m now resides in SokoPuzzle.c, etc.
//
// -*- Created a native Windows port of SokoSave using C++Builder.  The
//     back-end logic for this port is provided by the new "sokocore" common
//     code which used to be merged with the UI code for the
//     OpenStep/NextStep ports.  From the user-experience, this port is
//     nearly identical to the existing OpenStep and NextStep ports.  Builds
//     successfully with C++Builder versions 4 and 5.
//
// -*- Added a shortcut toolbar for quickly starting new games, opening saved
//     games, saving games, launching the scores panel, and launching the
//     help file.  Added a toggle-switch to the preferences panel to control
//     presence of toolbar.  From the user-standpoint, this is the only major
//     difference from the OpenStep/NextStep ports, since they do not feature
//     a shortcut toolbar.
//
// -*- Created an InnoSetup installer script for SokoSave.  This scripts
//     facilitates the creation of a stand-alone "setup" program
//     (soksetup.exe) for SokoSave.  With this program, Windows users can
//     download and install SokoSave in a fashion in which they are already
//     familiar.
//-----------------------------------------------------------------------------
#include <vcl.h>
#pragma hdrstop

#include "SokoPool.h"
#include "SokoApp.h"
#include "SokoBoard.h"
#include "SokoDefaults.h"
#include "SokoFile.h"
#include "SokoInfo.h"
#include "SokoPref.h"
#include "SokoRendezvous.h"
#include "SokoScores.h"
#include "SokoSetting.h"
#include "SokoUtil.h"
#include <shellapi.h>
#include <shlobj.h>
#include <vcl/registry.hpp>

#define SOKO_IDENTIFIERS_NUMERIC
#include "SokoSave.rc"

#pragma package(smart_init)
#pragma resource "*.dfm"
TSokoAppForm* SokoAppForm;

#define NEW_PLAYER_SETTING "NewPlayer"
#define HELP_FILE "help/SokoHelp.html"

#define NEW_GAME_DEFAULT_FILTER    1
#define OPEN_GAME_DEFAULT_FILTER   1
#define SAVE_GAME_DEFAULT_FILTER   1
#define CHOOSE_GAME_DEFAULT_FILTER (SOKO_STYLE_MAX + 3)

//-----------------------------------------------------------------------------
// Constructor
//-----------------------------------------------------------------------------
__fastcall TSokoAppForm::TSokoAppForm( TComponent* owner ) :
    TSokoForm(owner), hidden_windows(0)
    {
    can_publish = false;
    Application->OnMinimize = app_minimized;
    Application->OnRestore = app_restored;
    Application->OnDeactivate = app_deactivated;
    Application->OnIdle = app_idle;
    SokoDefaults::initialize();
    }


//-----------------------------------------------------------------------------
// Destructor
//-----------------------------------------------------------------------------
__fastcall TSokoAppForm::~TSokoAppForm()
    {
    SokoDefaults::shutdown();
    if (hidden_windows != 0)
	delete hidden_windows;
    }


//-----------------------------------------------------------------------------
// set_rendezvous_point
//-----------------------------------------------------------------------------
void TSokoAppForm::set_rendezvous_point( SokoRendezvous* r )
    {
    r->listen( this->secondary_launched );
    }


//-----------------------------------------------------------------------------
// choose_puzzle
//	At application launch time, give user a chance to explicitly open a
//	puzzle or saved game if TSokoBoardForm was unable to automatically open
//	the "next" puzzle.
//-----------------------------------------------------------------------------
bool TSokoAppForm::choose_puzzle()
    {
    open_dialog->FilterIndex = CHOOSE_GAME_DEFAULT_FILTER;
    open_dialog->InitialDir = new_dialog->InitialDir;
    return TSokoBoardForm::choose_puzzle();
    }


//-----------------------------------------------------------------------------
// report_open_failure
//-----------------------------------------------------------------------------
void TSokoAppForm::report_open_failure( char const* path )
    {
    AnsiString s( "Failed to open file: " );
    s += path;
    Application->MessageBox( s.c_str(), "SokoSave", MB_OK );
    }


//-----------------------------------------------------------------------------
// form_create
//-----------------------------------------------------------------------------
void __fastcall TSokoAppForm::form_create( TObject* sender )
    {
    register_file_associations();
    configure_dialogs();

    bool ok = false;
    for (int i = 0, n = ParamCount(); i < n; i++)
	if (TSokoBoardForm::open_puzzle( ParamStr(i + 1).c_str() ))
	    ok = true;
	else
	    report_open_failure( ParamStr(i + 1).c_str() );

    if (!ok)
	ok = TSokoBoardForm::open_next_puzzle() || choose_puzzle();

    if (!ok)
	Application->Terminate();
    else
	{
	SokoSetting settings;
	if (settings.get_bool( NEW_PLAYER_SETTING, soko_true ))
	    {
	    settings.set_bool( NEW_PLAYER_SETTING, soko_false );
	    launch_about();
	    }
	}
    }


//-----------------------------------------------------------------------------
// launch_about
//-----------------------------------------------------------------------------
void TSokoAppForm::launch_about()
    {
    TSokoInfoForm::launch();
    }


//-----------------------------------------------------------------------------
// launch_preferences
//-----------------------------------------------------------------------------
void TSokoAppForm::launch_preferences()
    {
    TSokoPrefForm::launch();
    }


//-----------------------------------------------------------------------------
// launch_scores
//-----------------------------------------------------------------------------
void TSokoAppForm::launch_scores()
    {
    TSokoScoresForm::launch();
    }


//-----------------------------------------------------------------------------
// launch_instructions
//-----------------------------------------------------------------------------
void TSokoAppForm::launch_instructions()
    {
    SokoPool pool;
    SokoSetting settings;
    char const* path = soko_add_path_component(
	pool, settings.factory_path(pool), HELP_FILE );
    if (soko_path_exists( pool, path ))
	ShellExecute( Handle, "open", path, 0, 0, SW_SHOW );
    else
	{
	AnsiString s( "Unable to locate help file \"" );
	s += soko_filename_part( pool, path );
	s += "\".";
	Application->MessageBox( s.c_str(), "Sorry", MB_OK | MB_ICONERROR );
	}
    }


//-----------------------------------------------------------------------------
// app_deactivated
//	Give SokoBoard a chance to abort playback sessions when the application
//	is backgrounded.
//-----------------------------------------------------------------------------
void __fastcall TSokoAppForm::app_deactivated( TObject* sender )
    {
    TSokoBoardForm::app_deactivated();
    }


//-----------------------------------------------------------------------------
// app_idle
//	Give SokoBoard a chance to animate playback sessions.
//-----------------------------------------------------------------------------
void __fastcall TSokoAppForm::app_idle( TObject* sender, bool& done )
    {
    TSokoBoardForm::app_idle( done );
    }


//-----------------------------------------------------------------------------
// app_minimized
//
// NOTE *1*
//	In the typical VCL application, sending a Minimize() message to the
//	global TApplication object causes all windows owned by the application
//	to be hidden as well.  This provides a convenient way to implement an
//	application-wide "Hide" or "Minimize All" function.  This works because
//	all windows created by a VCL application normally have their
//	"WndParent" set to the application's handle.  Unfortunately, TSokoForm
//	disables this behavior by setting WndParent to the desktop window (for
//	valid reasons of its own), thus we must hide windows manually whenver
//	the application is minimized.
//
// NOTE *2*
//	Unfortunately, mucking with a window's state dislodges its position in
//	TScreen's Forms[] array, thus we must first cache the list of windows.
//
// NOTE *3*
//	The application's global button should only display on the taskbar when
//	the application is hidden.  At all other times, the taskbar will
//	contain one button for each open window, thus there is no need to
//	additionally display the appilcation's global button.
//-----------------------------------------------------------------------------
void __fastcall TSokoAppForm::app_minimized( TObject* sender )
    {
    if (hidden_windows == 0)
	hidden_windows = new TList;

    int i, n = Screen->FormCount;
    TForm** forms = new TForm*[n];		// *2*
    for (i = 0; i < n; i++)
	forms[i] = Screen->Forms[i];
    for (i = 0; i < n; i++)			// *1*
	{
	TForm* f = forms[i];
	if (f->Visible)
	    {
	    f->Visible = false;
	    hidden_windows->Add(f);
	    }
	}
    delete[] forms;

    ShowWindow( Application->Handle, SW_SHOW );	// *3*
    }


//-----------------------------------------------------------------------------
// app_restored
//	See app_minimized() for an explanation of this functionality.
//-----------------------------------------------------------------------------
void __fastcall TSokoAppForm::app_restored( TObject* sender )
    {
    if (hidden_windows != 0)
	{
	for (int i = hidden_windows->Count - 1; i >= 0; i--)
	    {
	    TForm* f = static_cast<TForm*>(hidden_windows->Items[i]);
	    f->Visible = true;
	    }
	hidden_windows->Clear();
	}
    ShowWindow( Application->Handle, SW_HIDE );
    }


//-----------------------------------------------------------------------------
// secondary_launched
//	When a secondary instance of the application is launched, it rendezvous
//	with, and sends its command-line arguments to the first running
//	instance and then terminates itself, thus ensuring that only a single
//	instance of the application runs at any given time.  This method is
//	called by the SokoRendezvous class when command-line arguments are
//	received from a secondary instance.  Note, also, that even in cases
//	where a secondary instance was launched with no command-line arguments,
//	the first running instance still needs to be re-activated as feedback
//	for the user.  Activation may involve restoring the application if
//	hidden, or deminimizing the most recently minimized window if all
//	windows have been minimized.
//-----------------------------------------------------------------------------
void __fastcall TSokoAppForm::secondary_launched( char const* args )
    {
    if (IsIconic( Application->Handle )) // App was hidden, so restore.
	Application->Restore();
    else
	Application->BringToFront();

    bool opened = false;
    for ( ; *args != '\0'; args += strlen(args) + 1)
	if (TSokoBoardForm::open_puzzle( args ))
	    opened = true;
	else
	    report_open_failure( args );

    if (opened) // Opened a new window, so bring application to foreground.
	SetForegroundWindow( Screen->ActiveForm->Handle );
    else
	{ // No new windows opened, so activate an existing window.
	TForm* f = Screen->ActiveForm;
	if (f == 0)
	    {
	    int const n = Screen->FormCount;
	    for (int i = 0; i < n; i++)
		if ((f = dynamic_cast<TSokoBoardForm*>(Screen->Forms[i])) != 0)
		    break;
	    if (f == 0 && n > 0)
		f = Screen->Forms[0];
	    }
	if (f != 0)
	    {
	    if (f->WindowState == wsMinimized)
		ShowWindow( f->Handle, SW_RESTORE );
	    f->Show();
	    SetForegroundWindow( f->Handle );
	    }
	}
    }


//-----------------------------------------------------------------------------
// configure_dialogs
//-----------------------------------------------------------------------------
void TSokoAppForm::configure_dialogs()
    {
    int i;
    SokoPool pool;
    AnsiString wildcard("*.");

    SokoPuzzleExtension const* pexts = SokoPuzzle_get_puzzle_extensions();
    AnsiString puzzle_ext[ SOKO_STYLE_MAX + 1 ];
    for ( ; pexts->extension != 0; pexts++)
	{
	if (puzzle_ext[ pexts->style ].Length() != 0)
	    puzzle_ext[ pexts->style ] += ";";
	if (puzzle_ext[ SOKO_STYLE_MAX ].Length() != 0)
	    puzzle_ext[ SOKO_STYLE_MAX ] += ";";
	puzzle_ext[ pexts->style ] += wildcard + pexts->extension;
	puzzle_ext[ SOKO_STYLE_MAX ] += wildcard + pexts->extension;
	}
    AnsiString save_ext = wildcard + SokoPuzzle_get_save_extension();
    AnsiString any_ext = puzzle_ext[ SOKO_STYLE_MAX ] + ";" + save_ext;

    AnsiString puzzle_filter[ SOKO_STYLE_MAX + 1 ];
    for (i = SOKO_STYLE_MAX; i >= 0; i--)
	{
	puzzle_filter[i] = "SokoSave ";
	if (i != SOKO_STYLE_MAX)
	    puzzle_filter[i] += SokoPuzzle_get_style_name((SokoPuzzleStyle)i);
	puzzle_filter[i] += " puzzle (" + puzzle_ext[i] + ")|" + puzzle_ext[i];
	}
    AnsiString save_filter =
	AnsiString("SokoSave saved game (") + save_ext + ")|" + save_ext;
    AnsiString any_filter =
	AnsiString("SokoSave file (") + any_ext + ")|" + any_ext;

    AnsiString new_dialog_filter;
    for (i = SOKO_STYLE_MAX; i >= 0; i--)
	{
	if (i != SOKO_STYLE_MAX)
	    new_dialog_filter += "|";
	new_dialog_filter += puzzle_filter[i];
	}
    AnsiString save_dialog_filter = save_filter;
    AnsiString open_dialog_filter =
	save_filter + "|" + new_dialog_filter + "|" + any_filter;

    AnsiString puzzle_dir =
	soko_expand_path( pool, soko_get_puzzle_directory(pool) );
    AnsiString save_dir =
	soko_expand_path( pool, soko_get_save_directory(pool) );

    new_dialog->Filter = new_dialog_filter;
    new_dialog->FilterIndex = NEW_GAME_DEFAULT_FILTER;
    new_dialog->InitialDir = puzzle_dir;

    open_dialog->Filter = open_dialog_filter;
    open_dialog->FilterIndex = OPEN_GAME_DEFAULT_FILTER;
    open_dialog->InitialDir = save_dir;

    save_dialog->Filter = save_dialog_filter;
    save_dialog->FilterIndex = SAVE_GAME_DEFAULT_FILTER;
    save_dialog->InitialDir = save_dir;
    }


//-----------------------------------------------------------------------------
// read_key
//-----------------------------------------------------------------------------
static AnsiString read_key( TRegistry* r, AnsiString key, AnsiString name )
    {
    AnsiString s;
    try
	{
	if (r->OpenKeyReadOnly( key ))
	    {
	    s = r->ReadString( name );
	    r->CloseKey();
	    }
	}
    catch (...) { r->CloseKey(); }
    return s;
    }


//-----------------------------------------------------------------------------
// write_key
//-----------------------------------------------------------------------------
static void write_key(
    TRegistry* r, AnsiString key, AnsiString name, AnsiString value )
    {
    try
	{
	if (r->OpenKey( key, true ))
	    {
	    r->WriteString( name, value );
	    r->CloseKey();
	    }
	}
    catch (...) { r->CloseKey(); }
    }


//-----------------------------------------------------------------------------
// register_file_association
//	Register a file type association with Windows.  The file type is
//	registered if it does not already exist or if an existing registration
//	specifies the incorrect path.
//-----------------------------------------------------------------------------
bool TSokoAppForm::register_file_association(
    char const* ext, AnsiString icon_path, int icon_index, char const* desc )
    {
    bool changed = false;
    TRegistry* reg = new TRegistry;
    reg->RootKey = HKEY_CLASSES_ROOT;

    AnsiString doc_key("SokoSave."); doc_key += ext;
    AnsiString ext_key(".");         ext_key += ext;
    AnsiString ico_key(doc_key);     ico_key += "\\DefaultIcon";
    AnsiString cmd_key(doc_key);     cmd_key += "\\shell\\Open\\command";
    if (icon_index >= 0)
	icon_path = icon_path + "," + icon_index;

    AnsiString s = read_key( reg, ico_key, "" );
    if (s.AnsiCompareIC( icon_path ) != 0)
	{
	write_key( reg, ext_key, "", doc_key );
	write_key( reg, doc_key, "", desc );
	write_key( reg, ico_key, "", icon_path );
	write_key( reg, cmd_key, "", Application->ExeName + " \"%1\"" );
	changed = true;
	}

    delete reg;
    return changed;
    }


//-----------------------------------------------------------------------------
// register_file_associations
//-----------------------------------------------------------------------------
void TSokoAppForm::register_file_associations()
    {
    SokoPool pool;
    bool changed = false;
    AnsiString icon_path( Application->ExeName );
    SokoPuzzleExtension const* p = SokoPuzzle_get_puzzle_extensions();
    for ( ; p->extension != 0; p++)
	{
	AnsiString desc( "SokoSave " );
	desc += SokoPuzzle_get_style_name( p->style );
	desc += " puzzle";
	int const icon_index =
	    (p->style == SOKO_STYLE_HEXAGON  ? SOKO_ICON_PUZZLE_HEX :
	    (p->style == SOKO_STYLE_TRIANGLE ? SOKO_ICON_PUZZLE_TRI :
	    SOKO_ICON_PUZZLE));
	changed |= register_file_association(
	    p->extension, icon_path, icon_index, desc.c_str() );
	}
    changed |= register_file_association( SokoPuzzle_get_save_extension(),
	icon_path, SOKO_ICON_GAME, "SokoSave saved game" );
    if (changed)
	SHChangeNotify( SHCNE_ASSOCCHANGED, SHCNF_IDLIST, 0, 0 );
    }
